// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  timing_functions.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Daniel Jagszent
// Date:      26.04.2008
// *****************************************************************

#ifndef TIMING_FUNCTIONS_H_
#define TIMING_FUNCTIONS_H_

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>
#include <time.h>

//! Constant for conversion between nanoseconds and seconds.
static const long long NANOSECONDS_PER_SECOND = 1000000000LL;

//! Load the current time into sp. Returns a reference to sp so that you can chain function calls.
inline timespec& getCurrentTime(timespec& sp){
	clock_gettime(CLOCK_REALTIME, &sp);
	return sp;
}

//! Ensures that the tv_nsec member is a number between 0 and NANOSECONDS_PER_SECOND.
inline void normalizeTimespec(timespec& sp) {
	while (sp.tv_nsec >= NANOSECONDS_PER_SECOND) {
		sp.tv_nsec -= NANOSECONDS_PER_SECOND;
		++sp.tv_sec;
	}
	while (sp.tv_nsec < 0) {
		sp.tv_nsec -= NANOSECONDS_PER_SECOND;
		--sp.tv_sec;
	}
}

//! Converts a timespec into nanoseconds.
inline long long getNanoseconds(const timespec& sp){
	long long v = NANOSECONDS_PER_SECOND;
	v *= sp.tv_sec;
	v += sp.tv_nsec;
	return v;
}

//! Operator for comparing two timespecs. Precondition: both timespecs must be normalized.
inline bool operator < (const timespec& a, const timespec& b) {
	if (a.tv_sec == b.tv_sec) {
		return a.tv_nsec < b.tv_nsec;
	} else {
		return a.tv_sec < b.tv_sec;
	}
}


#endif /* TIMING_FUNCTIONS_H_ */
