#include "VFLinuxCapture.h"

CV4LCapture::CV4LCapture(std::string devName, int channel, int width, int height)
{
	// open the device
        deviceName = devName;
	selectedChannel.channel = channel;

	m_iWidth = width;
	m_iHeight = height;
}

bool CV4LCapture::OpenCamera()
{
        deviceHandle = open (deviceName.c_str(), O_RDWR);

        if (deviceHandle == -1)
        {       // could not open device
                printf ("Could not open device %s - %s\n", deviceName.c_str(), sys_errlist[errno]);
                return false;
        }

        // get device capabilities
        if (ioctl (deviceHandle, VIDIOCGCAP, &capability) == -1)
        {       // query failed
                printf ("could not obtain device capabilities\n");
                return false;
        }
        if ((capability.type & VID_TYPE_CAPTURE) == 0)
        {       // this device cannot capture video to memory, exit
                printf ("this device cannot capture video to memory\n");
                return false;
        }

        // enumerate and print out the channels
        int i = 0;
        while (i < capability.channels)
        {
                queryChannel.channel = i;
                if (ioctl (deviceHandle, VIDIOCGCHAN, &queryChannel) != -1)
                {       // ioctl success, queryChannel contains information about this channel
                        printf ("%d. %s\n", queryChannel.channel, queryChannel.name);
                }
                ++ i;
        }

        selectedChannel.norm = VIDEO_MODE_NTSC;
        if (ioctl (deviceHandle, VIDIOCSCHAN, &selectedChannel) == -1)
        {       // could not set the selected channel
                printf ("Could not set channel #%d\nNot a fatal error.", selectedChannel.channel);
        }

	// set the desired width and height
        if ((capability.type & VID_TYPE_SCALES) != 0)
        {       // supports the ability to scale captured images
                captureWindow.x = 0;
                captureWindow.y = 0;
                captureWindow.width = m_iWidth;
                captureWindow.height = m_iHeight;
                captureWindow.chromakey = 0;
                captureWindow.flags = 0;
                captureWindow.clips = 0;
                captureWindow.clipcount = 0;
		if (ioctl (deviceHandle, VIDIOCSWIN, &captureWindow) == -1)
                {       // could not set window values for capture
                        printf ("Could not set desired dimensions\nNot a fatal error.\n");
                }
        }
	
	// retrieve the actual width and height of the capturing images
        if (ioctl (deviceHandle, VIDIOCGWIN, &captureWindow) == -1)
        {       // could not obtain specifics of capture window
                printf ("Could not obtain capture window dimensions.\n");
        }
        m_iWidth = captureWindow.width;
        m_iHeight = captureWindow.height;
        printf ("Capturing dimensions are : %d, %d\n", m_iWidth, m_iHeight);

	// request that we capture to 24bit RGB

        // get image properties
        if (ioctl (deviceHandle, VIDIOCGPICT, &imageProperties) != -1)
        {       // successfully retrieved the default image properties

                // the following values are for requesting 24bit RGB
                imageProperties.depth = 24;
                imageProperties.palette = VIDEO_PALETTE_RGB24;
                if (ioctl (deviceHandle, VIDIOCSPICT, &imageProperties) == -1)
                {       // failed to set the image properties
                        printf ("Could not set the video depth and palette.\nPerhaps not a fatal error.\n");
                }
        }

        // verify that the requested capture pixel depth and palette succeeded
        if (ioctl (deviceHandle, VIDIOCGPICT, &imageProperties) == -1)
        {       // failed to retrieve default image properties
                printf ("Failed to retrieve the video depth and palette.\n");
                return false;
        }

	if ((imageProperties.depth != 24) || (imageProperties.palette != VIDEO_PALETTE_RGB24))
        {       // not a format our program supports
                printf ("Format is not 24bit RGB.\n");
                return false;
        }
	
        printf ("Capture depth is 24bit RGB\n");

        // obtain memory about capture space
        if (ioctl (deviceHandle, VIDIOCGMBUF, &memoryBuffer) == -1)
        {       // failed to retrieve information about capture memory space
                printf ("Failed to retrieve information about MMIO space.\n");
                return false;
        }


        // obtain memory mapped area
        memoryMap = (char*)mmap (0, memoryBuffer.size, PROT_READ | PROT_WRITE, MAP_SHARED, deviceHandle, 0);
        if ((int)memoryMap == -1)
        {       // failed to retrieve pointer to memory mapped area
                printf ("Failed to obtain MMIO space.\n");
                return false;
        }


	printf("Allocating structures...\n");
        // allocate structures
        mmaps = (struct video_mmap*)(malloc (memoryBuffer.frames * sizeof (struct video_mmap)));

        // fill out the fields
        i = 0;
        while (i < memoryBuffer.frames)
        {
                mmaps[i].frame = i;
                mmaps[i].width = captureWindow.width;
                mmaps[i].height = captureWindow.height;
                mmaps[i].format = imageProperties.palette;
                ++ i;
        }

	printf("Requesting capture buffers...\n");
        // request capture to each buffer except the last one
        i = 0;
        while (i < (memoryBuffer.frames-1))
        {
                if (ioctl (deviceHandle, VIDIOCMCAPTURE, &mmaps[i]) == -1)
                {       // capture request failed
                }
                ++ i;
        }

	printf("Set index buffer...\n");
        // set our index to the last buffer
        bufferIndex = memoryBuffer.frames-1;

	return true;
}

bool CV4LCapture::CaptureImage(CByteImage **ppImages)
{
	char* frame = NextFrame();

	ppImages[0]->pixels = (unsigned char*) frame;

	/*
	// Copy image contents
	int n = m_iWidth * m_iHeight;

        for (int index = 0;  index < n;  ++ index)
        {
                ppImages[0]->pixels[index*3+2] = frame[index*3+2];
		ppImages[0]->pixels[index*3+1] = frame[index*3+1];
		ppImages[0]->pixels[index*3+0] = frame[index*3+0];
        }*/

	return true;
}

void CV4LCapture::CloseCamera()
{
	// free the video_mmap structures
        free (mmaps);

        // unmap the capture memory
        munmap (memoryMap, memoryBuffer.size);

        // close the device
        close (deviceHandle);
}

int CV4LCapture::GetWidth()
{
	return m_iWidth;
}

int CV4LCapture::GetHeight()
{
	return m_iHeight;
}

CByteImage::ImageType CV4LCapture::GetType()
{
	return CByteImage::eRGB24;	
}

char* CV4LCapture::NextFrame()
{
	// send a request to begin capturing to the currently indexed buffer
        if (ioctl (deviceHandle, VIDIOCMCAPTURE, &mmaps[bufferIndex]) == -1)
        {       // capture request failed
        }

        // move bufferIndex to the next frame
        ++ bufferIndex;
        if (bufferIndex == memoryBuffer.frames)
        {       // bufferIndex is indexing beyond the last buffer
                // set it to index the first buffer
                bufferIndex = 0;
        }

        // wait for the currently indexed frame to complete capture
        if (ioctl (deviceHandle, VIDIOCSYNC, &mmaps[bufferIndex]) == -1)
        {       // sync request failed
        }

        // return the address of the frame data for the current buffer index
        return (memoryMap + memoryBuffer.offsets[bufferIndex]);
}

CV4LCapture::~CV4LCapture()
{

}
