/*
 * AbstractHttpWorker.java
 */

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.net.Socket;

/**
 * Die Klasse uebernimmt das Parsen der Anforderung, das Konfigurieren der
 * Eingabe- und Ausgabestroeme und den Aufruf der jeweiligen Handlermethoden.
 */
public abstract class AbstractHttpWorker implements Runnable {
    protected static final int HTTP_BAD_REQUEST = 400;
    protected static final int HTTP_NOT_FOUND = 404;
    protected static final int HTTP_BAD_METHOD = 405;
    protected static final int HTTP_OK = 200;
    protected static final String CRLF = "\r\n";

    private final Socket socket;

    /**
     * Initiaslisert die Socketverbindung der Verarbeitung.
     * 
     * @param socket
     */
    protected AbstractHttpWorker(Socket socket) {
        this.socket = socket;
    }

    /**
     * Ausfuehren einer eingehenden Anforderung.
     */
    public final void run() {
        PrintStream ps = null;
        BufferedInputStream is = null;
        try {
            is = new BufferedInputStream(socket.getInputStream());
            ps = new PrintStream(socket.getOutputStream(), true);
            Request request = new Request(is);
            switch (request.getMethod()) {
            case RequestScanner.HEAD:
                doHead(request, ps);
                break;
            case RequestScanner.GET:
                doGet(request, ps);
                break;
            case RequestScanner.ERROR:
                ps.println("HTTP/1.0 " + HTTP_BAD_REQUEST
                        + " syntax-error in request.");
                break;
            default:
                ps.print("HTTP/1.0 " + HTTP_BAD_METHOD
                        + " unsupported method type: ");
                ps.println(request.getURI());
            }
        } catch (IOException report) {
            report.printStackTrace();
        } finally {
            try {
                if (ps != null) ps.close();
                if (is != null) is.close();
                socket.close();
            } catch (IOException ignore) {
            }
        }
    }

    /**
     * Gibt fuer Log-Zwecke Information ueber den Client zurueck. Die
     * Information sieht wie folgt aus:
     * 
     * <pre>
     * &quot;&lt;IP-ADRESSE&gt;(&lt;PORT&gt;)&quot;
     * </pre>
     * 
     * @return String der Client-Information.
     */
    protected final String getClientName() {
        return socket.getInetAddress().getHostAddress() + "("
                + socket.getPort() + ")";
    }

    /**
     * Die Methode soll den HEAD-Request ausfuehren. Die angefragte URI erhaelt
     * man ueber das Request-Objekt. Der Ausgabestrom muss nicht geschlossen
     * werden.
     * 
     * @param r die genaue Anforderung.
     * @param ps der Ausgabestrom.
     * @throws IOException wenn I/O Fehler auftreten.
     */
    protected abstract void doHead(Request r, PrintStream ps)
            throws IOException;

    /**
     * Die Methode soll den GET-Request ausfuehren. Die angefragte URI erhaelt
     * man ueber das Request-Objekt. Der Ausgabestrom muss nicht geschlossen
     * werden.
     * 
     * @param r die genaue Anforderung.
     * @param ps der Ausgabestrom.
     * @throws IOException wenn I/O Fehler auftreten.
     */
    protected abstract void doGet(Request r, PrintStream ps) throws IOException;
}