/*
 * RequestScanner.java
 */

import java.io.BufferedReader;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;

/**
 * Untersucht den Eingabestrom auf einzelne Worteinheiten.
 * Dabei werden HEAD und GET als Schluesselwoerter aufgefasst. 
 */
public final class RequestScanner {

    /**
     * Eingabeende.
     */
    public static final int EOF = -1;
    
    /**
     * Fehler.
     */
    public static final int ERROR = -1;

    /**
     * beliebiges Wortsymbol.
     */
    public static final int OTHER = 0;

    /**
     * Kennung HEAD.
     */
    public static final int HEAD = 1;

    /**
     * Kennung GET.
     */
    public static final int GET = 2;

    /**
     * Kennung Zeilenende.
     */
    public static final int CRLF = 3;

    private BufferedReader in; /* der Eingabestrom */
    private int currChar; /* das naechste Zeichen */
    private String lastWord; /* das gefundene Wort */

    /**
     * Konstruktor.
     * 
     * @param is ein Eingabestrom.
     * @throws IOException wenn I/O-Fehler auftreten.
     */
    public RequestScanner(InputStream is) throws IOException {
        this(new InputStreamReader(is));
    }

    /**
     * Konstruktor.
     * 
     * @param reader ein Reader,
     * @throws IOException wenn I/O-Fehler auftreten.
     */
    public RequestScanner(Reader reader) throws IOException {
        in = new BufferedReader(reader);
        currChar = nextChar();
    }

    /**
     * Gibt den gefundenen Wortinhalt zurueck.
     * 
     * @return das gefundene Wort.
     */
    public String getContents() {
        return lastWord;
    }

    /**
     * Gibt die Kennung des naechsten Wortes zurueck.
     * Der gefundene Inhalt wird mit <code>getContents()</code>
     * abgefragt.
     * <code>nextToken()</code> liest jeweils das naechstfolgende
     * Wort ein. Die Regeln von HTTP 1.1 werden beachtet.
     * 
     * @return Wortkennung.
     * @throws IOException wenn ein I/O-Fehler vorlag.
     */
    public int nextToken() throws IOException {
        boolean afterCRLF = false;
        try {
            while (true) {
                if (afterCRLF && (currChar != ' ' && currChar != '\t')) {
                    afterCRLF = false;
                    return CRLF;
                }
                afterCRLF = false;
                if (currChar == '\r') {
                    currChar = nextChar();
                    if (currChar == '\n') {
                        currChar = nextChar();
                        afterCRLF = true;
                    }
                } else if (Character.isWhitespace((char)currChar)) {
                    currChar = nextChar();
                } else if (currChar != -1) {
                    StringBuffer word = new StringBuffer();
                    while (currChar > 0
                        && !Character.isWhitespace((char)currChar)) {
                        word.append(httpChar());
                        currChar = nextChar();
                    }
                    lastWord = word.toString();
                    if (lastWord.equals("GET"))
                        return GET;
                    else if (lastWord.equals("HEAD"))
                        return HEAD;
                    else {
                        return OTHER;
                    }
                } else {
                    return EOF;
                }
            }
        } catch (EOFException e) {
            return EOF;
        }
    }

    private char httpChar() throws IOException {
        return (char)
            (currChar == '%' ? 16 * hexDigit() + hexDigit() : currChar);
    }

    /**
     * Liest das naechste Zeichen aus der Eingabe.
     * Passt dabei gleichzeitig auf das Eingabeende auf.
     * 
     * @return Das naechste Zeichen.
     * @throws EOFException Eingabeende.
     * @throws IOException sonstiger I/O-Fehler
     */
    private int nextChar() throws IOException {
        int result = in.read();
        if (result < 0 && currChar < 0)
            throw new EOFException("premature EOF");
        return result;
    }

    /**
     * Wandelt ein Hexzeichen der Eingabe in eine Digitalzahl.
     * 
     * @return Digitalzahl.
     * @throws IOException allgemeiner I/O-Fehler.
     */
    private int hexDigit() throws IOException {
        int c = Character.toLowerCase((char)nextChar());
        if (c >= '0' && c <= '9')
            c = c - '0';
        else if (c >= 'a' && c <= 'f')
            c = c - 'a' + 10;
        else
            throw new EOFException("illegal hex digit");
        return c;
    }
}
