/*
 * Parameter.java
 */

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import java.util.logging.FileHandler;
import java.util.logging.Logger;

/**
 * Die Klasse stellt allgemeine Konfigurationsparameter zur
 * Verfuegung.
 * Wenn vorhanden, werden die Parameter aus der Datei
 * <code>server.properties</code> gelesen. Anderfalls
 * werden die voreingestellten Defaultwerte beibehalten.
 */
public final class Parameter {    
    private Parameter() {/* is never called. */}
    
    /* Der Dateiname fuer den Property-File. */
    private static final String PROP_FILE_NAME = "server.properties";

    /* Das oberste Verzeichnis fuer den Server */
    private static File root = new File(System.getProperty("user.dir"));
    
    /* Die Port-Nummer des Servers */
    private static int port = 8080;

    /* Timeout fuer die Uebermittlung einer Anfrage in msec */
    private static int timeout = 5000;

    /* Benutztes Logger-Objekt */
    private static Logger logger;

    /*
     * Die Konfigurationsparameter.
     */
    private static Properties props = new Properties();

    /**
     * Gibt das Wurzelverzeichnis des Servers zurueck.
     * Default: Home-Verzeichnis.
     * 
     * @return Pfadname des Wurzelverzeichnisses.
     */
    public static File getRoot() {
        return root;
    }
    
    /**
     * Definiert die Portnummer fuer den Webserver.
     * ueberschreibt die Angabe der Parameter-Datei,
     * falls die Port-Nummer > 0 ist.
     * 
     * @param port Portnummer.
     */
    public static void setPort(int port) {
        if (port > 0)
            Parameter.port = port;
    }
    
    /**
     * Gibt die Portnummer des Servers zurueck.
     * Default: 8080.
     * 
     * @return Portnummer.
     */
    public static int getPort() {
        return port;
    }

    /**
     * Maximales Zeitfenster fuer eine Anfrage.
     * Default: 5 s.
     * 
     * @return Zeispanne in Millisekunden.
     */
    public static int getTimeout() {
        return timeout;
    }

    /**
     * PrintStream fuer Log-Nachrichten.
     * Default: Standard-Ausgabe.
     * 
     * @return geoeffneter PrintStream.
     */
    public static Logger getLogger() {
        return logger;
    }

    /**
     * Kontrollausgabe der wichtigsten Parameter.
     */
    public static void logParameters() {
        logger.info("root=" + root);
        logger.info("timeout=" + timeout);
        logger.info("port= " + port);
    }
    
    /* beim Progammstart werden die Parameter eingelesen. */
    static {
        try {
            loadProperties();
        } catch (IOException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    /* lade die Eigenschaften aus der Datei server.properties */
    private static void loadProperties() throws IOException {
        logger = Logger.getLogger("server");
        File f = new File(PROP_FILE_NAME);
        if (f.exists()) {
            InputStream is = new BufferedInputStream(new FileInputStream(f));
            props.load(is);
            is.close();
            String r = props.getProperty("root");
            if (r != null) {
                root = new File(r);
                if (!root.exists()) {
                    throw new Error(root + " doesn't exist as server root");
                }
            }
            r = props.getProperty("port");
            if (r != null) {
                port = Integer.parseInt(r);
            }
            r = props.getProperty("timeout");
            if (r != null) {
                timeout = Integer.parseInt(r);
            }
            r = props.getProperty("log");
            if (r != null) {
                System.err.println("opening log file: " + r);
                logger.addHandler(new FileHandler(r));
            }
        }
    }
}