/*
 * Request.java
 */

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

/**
 * In dieser vereinfachten Form, uebernimmt ein Objekt aus dem Eingabestrom
 * eine Http-Anfrage der Form:
 * <pre>
 * "&lt;METHODE> &lt;URI> &lt;VERSION>"
 * </pre>
 * Es werden nur die Methoden HEAD und GET erkannt.
 * <p>
 * Fuer nicht unterstuetzte Methoden wird der Text der Methodenkennung
 * als URI zurueckgegeben.
 * 
 * Die Kodierung der Methodentags ist in der Klasse
 * <code>RequestScanner</code> festgelegt.
 */
public final class Request {
    private int method;
    private String uri;
    private String version;

    /**
     * Konstruktor.
     * Analysiert die Eingabe und speichert die notwendige Information.
     * 
     * @param s Eingabestrom.
     * @throws IOException I/O-Fehler.
     */
    public Request(InputStream s) throws IOException {
        RequestScanner in = new RequestScanner(s);
        method = in.nextToken();
        if (method == RequestScanner.OTHER) {
            uri = in.getContents();
        }
        else {
            in.nextToken();
            uri = in.getContents().replace('/', File.separatorChar);
            in.nextToken();
            version = in.getContents();
            if (in.nextToken()!= RequestScanner.CRLF)
                method = RequestScanner.ERROR;
        }
    }

    /**
     * Gibt die Kennung der Methode zurueck.
     * 
     * @return Kennziffer der Methode.
     * @see server.RequestScanner
     */
    public int getMethod() {
        return method;
    }

    /**
     * Gibt die URI zurueck.
     * 
     * @return URI.
     */
    public String getURI() {
        return uri;
    }

    /**
     * Gibt die von Browser angegebene HTTP-Version zurueck.
     * 
     * @return HTTP-Kennung.
     */
    public String getVersion() {
        return version;
    }
}