/*
 * Worker.java
 */

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.net.Socket;
import java.util.Date;
import java.util.logging.Logger;

/**
 * Implementiert die Logik der GET- und HEAD-Request.
 * Die oeffentlichen Methoden sind von der Klasse
 * <code>AbstractHttpWorker</code> geerbt.
 */
public final class Worker extends AbstractHttpWorker {
    private Logger log = Parameter.getLogger();

    /**
     * Erzeugt ein neues Objekt, das einen per Socket
     * uebergebenen Http-Auftrag ausfuehrt.
     * 
     * @param socket
     */
    public Worker(Socket socket) {
        super(socket);
    }

    protected void doHead(Request rq, PrintStream ps) throws IOException {
        File target = getSource(rq);
        printHeader("Head", ps, target, target.exists());
    }
    
    protected void doGet(Request rq, PrintStream ps) throws IOException {
        File target = getSource(rq);
        if (checkPath(target) && target.exists()) {
            printHeader("GET", ps, target, true);
            sendFile(ps, target);
        } else {
            printHeader("GET", ps, target, false);
            send404(ps);
        }
    }

    /**
     * Prueft ob die in <code>path</code> uebergebene URI ein gueltiger
     * Dateiname innerhalb des Installationsverzeichnisses ist.
     * 
     * @param path URI der Anfrage.
     * @return <code>true</code> falls der Pfad ok ist.
     * @throws IOException bei I/O Fehler
     */
    private static boolean checkPath(File path) throws IOException {
        return path.getCanonicalPath().startsWith(
            Parameter.getRoot().getCanonicalPath());
    }

    /**
     * Ermittle die korrekten Source-Datei aus der  URI.
     * 
     * @param rq HTTP-Request.
     * @return File-Object fuer URI.
     */
    private static File getSource(Request rq) {
        String fileName = rq.getURI();
        while (fileName.startsWith(File.separator)) {
            fileName = fileName.substring(1);
        }
        File target = new File(Parameter.getRoot(), fileName);
        if (target.isDirectory()) {
            File indexHtml = new File(target, "index.html");
            if (indexHtml.exists()) {
                target = indexHtml;
            }
        }
        return target;
    }

    /**
     * Drucke die HTTP-Header Information.
     * 
     * @param ps Output-Stream
     * @param target source file.
     * @param exists <code>true</code> if source exists.
     * @throws IOException
     */
    private void printHeader(String requestName, PrintStream ps, File target,
            boolean exists) throws IOException {
        int rCode;
        if (!checkPath(target) || !exists) {
            rCode = HTTP_NOT_FOUND;
            ps.print("HTTP/1.0 " + rCode + " not found" + CRLF);
        }
        else {
            rCode = HTTP_OK;
            ps.print("HTTP/1.0 " + rCode + " OK" + CRLF);
        }
        log.info(requestName + " from " + getClientName() + ": "
            + target.getAbsolutePath() + "-->" + rCode);
        ps.print("Server: Simple java" + CRLF);
        ps.print("Date: " + (new Date()) + CRLF);
        if (exists)
            printContentType(ps, target);
        ps.print(CRLF);
    }

    /**
     * Fuege MIME-Information dem HTTP-Header hinzu.
     * 
     * @param ps output stream.
     * @param target URI-Objekt.
     */
    private static void printContentType(PrintStream ps, File target) {
        if (!target.isDirectory()) {
            ps.print("Content-length: " + target.length() + CRLF);
            ps.print(
                "Last Modified: " + (new Date(target.lastModified())) + CRLF);
            String name = target.getName();
            ps.print("Content-type: " + MimeTypes.getType(name) + CRLF);
        } else {
            ps.print("Content-type: text/html" + CRLF);
        }
    }

    /**
     * URL nicht gefunden.
     * 
     * @param ps output stream.
     */
    private static void send404(PrintStream ps) {
        ps.print(CRLF);
        ps.print("Not Found");
        ps.print(CRLF);
        ps.print("The requested resource was not found.");
        ps.print(CRLF);
    }

    /**
     * Send den Inhalt der URI.
     * 
     * @param ps output stream-
     * @param target file object.
     * @throws IOException
     */
    private void sendFile(PrintStream ps, File target) throws IOException {
        if (target.isDirectory()) 
            listDirectory(ps, target);
        else 
            copyFile(ps, new FileInputStream(target.getAbsolutePath()));
    }

    /**
     * Kopiere alles vom Input zum Output.
     * 
     * @param ps output.
     * @param is intput.
     * @throws IOException bei I/O-Fehler.
     */
    private void copyFile(PrintStream ps, InputStream is) throws IOException {
        try {
            byte[] buffer = new byte[2048];
            int n;
            while ((n = is.read(buffer)) > 0) {
                ps.write(buffer, 0, n);
            }
        }
        finally {
            is.close();
        }
    }

    /**
     * Generiere eine HTML-Seitee mit einem Listing des ausgewaehlten
     * Directories.
     * 
     * @param ps output stream.
     * @param dir Verzeichnispfad.
     */
    private static void listDirectory(PrintStream ps, File dir) {
        ps.println("<TITLE>Directory listing</TITLE><P>\n");
        ps.println("<A HREF=\"..\">Parent Directory</A><BR>\n");
        String[] list = dir.list();
        for (int i = 0; list != null && i < list.length; i++) {
            File f = new File(dir, list[i]);
            if (f.isDirectory()) {
                ps.println(
                    "<A HREF=\"" + list[i] + "/\">" + list[i] + "/</A><BR>");
            } else {
                ps.println(
                    "<A HREF=\"" + list[i] + "\">" + list[i] + "</A><BR");
            }
        }
        ps.println("<P><HR><BR><I>" + (new Date()) + "</I>");
    }
}