//*********************************************************
// IEEE STD 1364-2001 Verilog tmds_encoder.v 
// Author-EMAIL: Uwe.Meyer-Baese@ieee.org
//*********************************************************
// --------------------------------------------------------
module tmds_encoder (
    input CLK,       // Pixel clock
    input RESET,     // Synchronous reset
    input DE,        // Data enable
    input [7:0] PD,  // Pixel data
    input [1:0] CTL, // Control data
    output reg [9:0] Q_OUT); // Output data 

// -------------------------------------------------------------------
// Compute number of 1s in pixel data 
wire [3:0] N1D = PD[0] + PD[1] + PD[2] + PD[3] + PD[4] + PD[5] + PD[6] + PD[7];

// Compute internal vector Q_M
wire USE_XNOR = (N1D > 4'd4) || (N1D == 4'd4 && PD[0]==1'b0);
wire [8:0]  Q_M = USE_XNOR ? 
            {1'b0, ~(Q_M[6:0] ^ PD[7:1]), PD[0]} : // use XNOR
            {1'b1,   Q_M[6:0] ^ PD[7:1], PD[0]}; // use XOR
            
 // Compute number of 1s in Q_M data 
 wire [3:0] N1Q = Q_M[0] + Q_M[1] + Q_M[2] + Q_M[3] + Q_M[4] + Q_M[5] + Q_M[6] + Q_M[7];
  

wire [6:0] N1N0 = 2*N1Q - 8; // i.e., N1Q - N0Q;
reg INV_Q; 
reg [6:0] CNT, CNT_NEW;
reg [9:0] Q_CTL, Q_DATA;

// determine the control tokens
   always @* 
     case (CTL)
      2'b00   :  Q_CTL <= 10'h354;
      2'b01   :  Q_CTL <= 10'h0AB;
      2'b10   :  Q_CTL <= 10'h154;
      default :  Q_CTL <= 10'h2AB;
    endcase 

// Compute output and new disparity counter value
   always @(*) begin
   INV_Q <= 1'B0; CNT_NEW <= 0;
// Update the disparity counter
    if (CNT==0 || N1N0==0)
      if (Q_M[8]==1'B0) begin
        CNT_NEW <= CNT - N1N0; INV_Q <= 1'B1;
      end else
        CNT_NEW <= CNT + N1N0;
    else
      if (CNT[6] == N1N0[6]) // same sign?
      begin
        INV_Q <= 1'B1;
        if (Q_M[8]==1'b0) 
          CNT_NEW <= CNT - N1N0;
        else
          CNT_NEW <= CNT - N1N0 + 2;
      end else 
        if (Q_M[8]==1'b0)
          CNT_NEW <= CNT + N1N0 - 2;
        else
          CNT_NEW <= CNT + N1N0;
  end
// Compute the q_data vector 
      always @*
      Q_DATA <= INV_Q ? {1'B1, Q_M[8], ~Q_M[7:0]} :
                        {1'B0, Q_M[8],  Q_M[7:0]} ;  
    
// store counter and q_out in registers
    always @(posedge CLK) begin
      if (RESET == 1'b1) begin
        CNT = 1'b0; Q_OUT <= 10'h000;
      end else  // PIXEL DATA (DE=1) OR CONTROL DATA (DE=0) ?
        if (DE == 1'b1) begin 
            CNT <= CNT_NEW;
            Q_OUT <= Q_DATA;
        end else begin 
            CNT <= 0;
            Q_OUT <= Q_CTL;
        end
    end
  
endmodule
// -----------------------------------------------------------------
