//=======================================================
//  PORT declarations
//=======================================================
module HDMI(
input  CLOCK_125,   // CLOCK 125 MHz
output [3:0] LEDR, // 4 LEDs
input  [3:0] KEY,  // Push Buttons 
input  [3:0] SW,   // Slider switches
input  [7:0] X,   // X coordinate
input  [7:0] Y,   // Y coordinate
input  [7:0] D,   // Data value
output HDMI_OUT_EN,
output [2:0] HDMI_D_P, HDMI_D_N,
output HDMI_CLK_P, HDMI_CLK_N
/////// Extra test ports
//output [9:0] TMDS_RED_out
);
//=========================================================
// REG/WIRE declarations
//=========================================================
wire RESET = KEY[0]; // btn[0] active high reset on ZYBO
wire  iCLOCK_25, CLOCK_25; // The 25 MHz clock
wire  iCLOCK_250, CLOCK_250; // The 250 MHz clock
wire CLOCK_25_n; // Complement 25 MHz
reg DLY_RST; // Delay reset for PLL signal
reg [19:0] CONT; // Delay counter
assign HDMI_OUT_EN = 1; // ZyBo is source
//=======================================================
//  Reset Delay Timer
//======================================================= 
always@(posedge CLOCK_125 or posedge RESET)
begin
  if (RESET) begin
    CONT  <= 20'H00000;
    DLY_RST <= 1;
  end 
  else
  //if (CONT < 20'hFFFFF) // Normal operation
  if (CONT < 20'h0003C) // Simulation short cut
    CONT    <= CONT + 1;
  else
    DLY_RST  <=  0;
end 
//===================================================================
//  PLL to generate the 25 and 250 MHz clock
//===================================================================
wire LOCKED, FB_CLK;  // Clock feedback and lock signal
PLLE2_BASE #(
  .BANDWIDTH("OPTIMIZED"), // OPTIMIZED, HIGH, LOW
  .CLKFBOUT_MULT(8), // Multiply value for all CLKOUT, (2-64)
  .CLKFBOUT_PHASE(0.0), // Phase offset in degrees, (-360.0-360.0).
  .CLKIN1_PERIOD(8.0), // Input clock period in ns.
// CLKOUT0_DIVIDE - CLKOUT5_DIVIDE: Divide amount for each CLKOUT (1-128)
  .CLKOUT0_DIVIDE(4),
  .CLKOUT1_DIVIDE(40),
// CLKOUT0..5_DUTY_CYCLE: Duty cycle for each CLKOUT (0.001-0.999).
  .CLKOUT0_DUTY_CYCLE(0.5),
  .CLKOUT1_DUTY_CYCLE(0.5),
// CLKOUT0..5_PHASE: Phase offset for each CLKOUT (-360.000-360.000).
  .CLKOUT0_PHASE(0.0),
  .CLKOUT1_PHASE(0.0),
  .DIVCLK_DIVIDE(1), // Master division value, (1-56)
  .REF_JITTER1(0.01), // Reference input jitter in UI, (0.000-0.999).
  .STARTUP_WAIT("FALSE")) // Delay DONE until PLL Locks, ("TRUE"/"FALSE")
PLLE2_BASE_inst (
  // Clock Outputs: 1-bit outputs: User configurable clock outputs
  .CLKOUT0(iCLOCK_250), // 1-bit output: CLKOUT0
  .CLKOUT1(iCLOCK_25), // 1-bit output: CLKOUT1
// Feedback Clocks: 1-bit (each) output: Clock feedback ports
  .CLKFBOUT(FB_CLK), // 1-bit output: Feedback clock
  .LOCKED(LOCKED), // 1-bit output: LOCK
  .CLKIN1(CLOCK_125), // 1-bit input: Input clock
// Control Ports: 1-bit (each) input: PLL control ports
  .PWRDWN(0), // 1-bit input: Power-down
  .RST(0), // 1-bit input: Reset
// Feedback Clocks: 1-bit (each) input: Clock feedback ports
  .CLKFBIN(FB_CLK)); // 1-bit input: Feedback clock
// End of PLLE2_BASE_inst

BUFG BUFG_250(.I(iCLOCK_250), .O(CLOCK_250));
BUFG BUFG_25 (.I(iCLOCK_25), .O(CLOCK_25));
//=======================================================
//  Sync signal generation
//======================================================
reg [9:0] COUNTER_X, COUNTER_Y;
reg H_SYNC, V_SYNC, DATA_ENABLE;
always @(posedge CLOCK_25) 
  if (DLY_RST) begin 
    COUNTER_X  <= 0; COUNTER_Y  <= 0; 
    DATA_ENABLE <= 0; H_SYNC <= 0; V_SYNC <= 0;
  end else begin 
    // Sync signal are code withing the control BLU
    DATA_ENABLE <= (COUNTER_X>=2) && (COUNTER_X<642) 
                        && (COUNTER_Y<480); // Allow two memory access
    COUNTER_X <= (COUNTER_X==799) ? 0 : COUNTER_X+1;
    if (COUNTER_X==799) COUNTER_Y <= (COUNTER_Y==524) ? 0 :  COUNTER_Y+1;
    H_SYNC <= (COUNTER_X>=658) && (COUNTER_X<754); // delay by two
    V_SYNC <= (COUNTER_Y>=490) && (COUNTER_Y<492); 
  end 
//=======================================================
//  Load image data using X/Y Counters
//=======================================================
reg [7:0] RED, GRN, BLU;
//// Address array is linear:
wire [18:0] ADDR = COUNTER_X[9:3] + COUNTER_Y[9:3]*80; 
wire WE = ((COUNTER_X==(X<<3)) && (COUNTER_Y==(Y<<3)) && KEY[3]);
wire [7:0] DATA = D; // { 4'h3, SW};
wire [7:0] INDEX;
wire PIXEL;
img_ram img_data_inst (
 .ADDRESS(ADDR[12:0]),
 .WE(WE),
 .DATA(DATA),
 .CLOCK (CLOCK_25),
 .Q(INDEX));
////// Load the binary 8x8 character font values
char_rom Char_inst(
 .SW(SW[1:0]),
 .ADDRESS (INDEX[6:0]),
 .CLK (CLOCK_25),
 .COL(COUNTER_X[2:0]),
 .ROW(COUNTER_Y[2:0]),
 .Q(PIXEL));
// Map color data to (default) gray scale 
wire [7:0] GRAY_DATA = (PIXEL) ? 8'hFF : 8'h00;
always @(posedge CLOCK_25) 
  if (DLY_RST) begin 
    RED  <= 0; GRN <= 0; BLU <= 0;
  end else begin 
    if (SW[2]) RED <= 8'h00; else RED <= GRAY_DATA;
    if (SW[3]) GRN <= 8'h00; else GRN <= GRAY_DATA;
    BLU <= GRAY_DATA;
  end
//=======================================================
//  TMDS encoder instantiations
//=======================================================
wire [9:0] TMDS_RED, TMDS_GRN, TMDS_BLU ;
tmds_encoder B_inst(.CLK(CLOCK_25), .RESET(DLY_RST), .PD(BLU), .CTL({V_SYNC,H_SYNC}), .DE(DATA_ENABLE), .Q_OUT(TMDS_BLU));
tmds_encoder G_inst(.CLK(CLOCK_25), .RESET(DLY_RST), .PD(GRN), .CTL(2'b00)          , .DE(DATA_ENABLE), .Q_OUT(TMDS_GRN));
tmds_encoder R_inst(.CLK(CLOCK_25), .RESET(DLY_RST), .PD(RED), .CTL(2'b00)          , .DE(DATA_ENABLE), .Q_OUT(TMDS_RED));
//===========================================================================
//  Parallel to serial may need the OSERDESE2 hard macro for high CLOCK rates
//===========================================================================
reg [3:0] TMDS_MOD10=0;  // modulus 10 counter
reg [9:0] TMDS_SHIFT_RED=0, TMDS_SHIFT_GRN=0, TMDS_SHIFT_BLU=0;
reg TMDS_SHIFT_LOAD=0;
//always @(posedge CLOCK_250) TMDS_SHIFT_LOAD <= (TMDS_MOD10==4'd9);

always @(posedge CLOCK_250)
  if (DLY_RST) begin 
    TMDS_SHIFT_LOAD <= 0;
    TMDS_SHIFT_RED <= 0; TMDS_SHIFT_GRN <= 0;
    TMDS_SHIFT_BLU <= 0; TMDS_MOD10 <= 0;
  end else
  begin
    TMDS_SHIFT_RED <= TMDS_SHIFT_LOAD ? TMDS_RED : TMDS_SHIFT_RED[9:1];
    TMDS_SHIFT_GRN <= TMDS_SHIFT_LOAD ? TMDS_GRN : TMDS_SHIFT_GRN[9:1];
    TMDS_SHIFT_BLU <= TMDS_SHIFT_LOAD ? TMDS_BLU : TMDS_SHIFT_BLU[9:1];	
    TMDS_MOD10 <= (TMDS_MOD10==4'd9) ? 4'd0 : TMDS_MOD10+4'd1;
    TMDS_SHIFT_LOAD <= (TMDS_MOD10==4'd9);    
  end
//===========================================================
//  Generate the differential output signals with hard macro
//===========================================================
OBUFDS OBUFDS_BLU(.I(TMDS_SHIFT_BLU[0]), .O(HDMI_D_P[0]), .OB(HDMI_D_N[0]));
OBUFDS OBUFDS_GRN(.I(TMDS_SHIFT_GRN[0]), .O(HDMI_D_P[1]), .OB(HDMI_D_N[1]));
OBUFDS OBUFDS_RED(.I(TMDS_SHIFT_RED[0]), .O(HDMI_D_P[2]), .OB(HDMI_D_N[2]));
OBUFDS OBUFDS_clock(.I(CLOCK_25), .O(HDMI_CLK_P), .OB(HDMI_CLK_N));

assign LEDR[0] = CLOCK_25;
assign LEDR[1] = CLOCK_250;
assign LEDR[2] = DLY_RST;
assign LEDR[3] = LOCKED;
assign TMDS_RED_out = TMDS_RED;
endmodule

