// ======================================================================
// This program measures the FECG performance on the DE1-SoC board
//
// FECG analysis program has 6 steps:
// 1) Convert int->float
// 2) Remove DC from inputs
// 3) Compute crosscorrelation matrix
// 4) Compute the Eigenvectors (and eigenvalues, not needed)
// 5) Apply eigenvectors, i.e., do the PCA
// 6) Compute MECG and FECG BPMs
// ======================================================================
#include <stdio.h>      /* printf */
#include <math.h>       /* sqrt */
#include <stdlib.h>     /* abs */

#include "rom.h"
#include "jacobi_eigenvalue_float.h"

#define sample_window 375
#define channels 5
#define N 5
#define DEBUG 0
#define ATYPE 1
#define RSTART 2
#define REND 4
#define RINC 2
#define LEDR_BASE    0xFF200000
#define SW_BASE      0xFF200040
#define KEY_BASE     0xFF200050
#define TIMER_BASE   0xFF202000
#define Fcpu         100000000 
#define TB        "ms"

float ymean[channels][sample_window];
float send_array[channels][sample_window];
extern int init;
float VctrMean[channels];
float MtrxOrig[channels][sample_window];
float MtrxMean[channels][sample_window];
float MtrxCxx[channels][channels];
float VctrEivals[channels];
float MtrxEivec[channels][channels];
float MtrxOut[channels][sample_window];

float decg[channels][sample_window];

int main()
{
  float Tscale = 1000.0/Fcpu; // TB= "ms"
  // float Tscale = 1000.0*1000.0/Fcpu; // TB= "us"
   // float Tscale = 1000.0*1000.0*1000.0/Fcpu; // TB= "ns"
   
  volatile int *red_LED_ptr   = (int *) LEDR_BASE;  // Red LED address
  volatile int *SW_switch_ptr = (int *) SW_BASE;    // Slider switch address
  volatile int *interval_timer_ptr = (int *) TIMER_BASE; // Timer address
  int high_half, counter, SW_value;  
  int Value1, Value2, Value21, Total=0;
  int seconds=0;
  int i, j, k, u;
  int rot_num, it_num;
  float samplesdiv = 1/(float)sample_window;   
  int first,second,x1,x2,dx, HR, BPM;
  float Vmax, v1, v2, ssum, nsum, Vmax50;
  float **aa, **vv, **zz;
  float *d2, *d3, *ee;
  float a[N*N], b[N*N];

  d2 = vector(0,N);
  vv = matrix(0,N-1,0,N-1);
  aa = matrix(0,N-1,0,N-1);
  d3 = vector(1,N);
  zz = matrix(1,N,1,N);
  ee = vector(1,N);

  printf("Hello ECG from 100 MHz Nios II no FPH2\n\r");
  
  printf("*** Check conversion FP->INT time...\n\r");
  //start timer
  /* Set the interval timer to 32 bit max */
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
  *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
  *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
  for (i  = 0 ; i < channels ; i++) {
    for (j = 0 ; j < sample_window ; j++) {
    //Write Values in the Original (Untouched) INTEGER Input Matrix
      decg[i][j]= (float) idecg[i][j] / 2048.0; // 12 bit precision data
    }
  }
  //get timer value and compute elapsed time
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0 
  // Make a counter snapshot by wrting a dummy value 
  *(interval_timer_ptr + 0x4) =  0;
    // read the 32-bit counter snapshot from the 16-bit timer registers
  high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
  counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
  Value21 = (0x7FFFFFFF - counter); // Clock cycle divided by CPU frequency
  printf("1) INT->FP 5x375 cycles: %d\n", (int) Value21);
  printf("1) INT->FP 5x375   time: %d %s\r\n", (int) (Value21*Tscale),TB);
  Total = Value21;
  if (DEBUG) {
    printf("--- First 5x5 Input values * 1000 (int) ...\n\r");
    for (i = 0 ; i < channels ; i++) {
      for (j=0; j<5; j++) {
        Value1 = decg[i][j] * 1000;
        printf("%d , ",Value1);
      }
      printf("\n\r");
    }
  }
  printf("*** Compute the DC from ROM Input Matrix...\n\r");    
  printf("*** Subtract mean values from matrix...\n\r");
  //start timer
  /* Set the interval timer to 32 bit max */
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
  *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
  *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
  for (i  = 0 ; i < channels ; i++) {
    VctrMean[i]=0;
    for (j = 0 ; j < sample_window ; j++) {
      //Write Values in the Original (Untouched) Input Matrix
      MtrxOrig[i][j]=decg[i][j];
      //Write the Values in the Matrix that is later used as the Matrix with the substracted Mean Values
      MtrxMean[i][j] = MtrxOrig[i][j];
      //Add the Row Values to a sum, later used for Mean Value Calculation
      VctrMean[i] += MtrxMean[i][j];
    }
    //Scale sum by number of samples to get mean Value
    VctrMean[i] *= samplesdiv;
  }
  // Remove the DC part for each channel
  for (i = 0 ; i < channels ; i++) {
    for (j = 0 ; j < sample_window ; j++) {
      MtrxMean[i][j] -= VctrMean[i];
    };
  };
  //get timer value and compute elapsed time
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0 
  // Make a counter snapshot by wrting a dummy value 
  *(interval_timer_ptr + 0x4) =  0;
  // read the 32-bit counter snapshot from the 16-bit timer registers
  high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
  counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
  Value21 = (0x7FFFFFFF - counter); // Clock cycle divided by CPU frequency
  printf("2) DC calc + Sub mean cycles: %d\n", (int) Value21);
  printf("2) DC calc + Sub mean   time: %d %s\n", (int) (Value21*Tscale),TB);
  Total += Value21;
  if (DEBUG){
    printf("--- The mean values are * 1000(int)...\n\r");
    for (i = 0 ; i < channels ; i++) {
      Value1 = VctrMean[i] *1000;
      Value2 = VctrMean[i];
      printf("Mean[%d]= %d /1000 = %d\r\n",i,Value1,Value2);
      }
  }

  printf("*** Calculate the Covariance-Variance Matrix...\n\r");
  //start timer
  /* Set the interval timer to 32 bit max */
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
  *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
  *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
  for (i = 0; i< channels ; i++)   {
    for (k = 0; k< channels ; k++)      {
      MtrxCxx[i][k] = 0;
      for (j = 0; j< sample_window ; j++)  {
      //Matrix Multiplikation: MtrxMean*MtrxMean'
        MtrxCxx[i][k] += (MtrxMean[i][j] * MtrxMean[k][j]);
      }
      MtrxCxx[i][k] *= samplesdiv;
    }
  }
  //get timer value and compute elapsed time
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0 
  // Make a counter snapshot by wrting a dummy value 
  *(interval_timer_ptr + 0x4) =  0;
  // read the 32-bit counter snapshot from the 16-bit timer registers
  high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
  counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
  Value21 = (0x7FFFFFFF - counter); // Clock cycle divided by CPU frequency
  printf("3) Get Cxx cycles: %d\n", (int) Value21);
  printf("3) Get Cxx   time: %d %s\r\n", (int) (Value21*Tscale),TB);
  Total += Value21;
  
  if (DEBUG) {
    printf("--- The 5x5 Cxx matrix is * 100000 (int) ...\n\r");
    for (i = 0 ; i < channels ; i++) {
      for (j=0; j<channels; j++) {
        Value1 = MtrxCxx[i][j] * 100000;
        printf("%d , ",Value1);
      }
      printf("\n\r");
    }
  }

  printf("*** 3x Calculate the Eigenvectors and Eigenvalues...\n\r");
  // Note ATYPE can be used to select desired algorithm; disabled to calculate time for all 3
  
//if ( ATYPE == 2) { 
     printf("*** Method is Jacobi/Rutishauser by src: Numerical Receips...\n\r");
        for (i=0;i<N;i++) for (j=0;j<N;j++) { aa[i][j]=MtrxCxx[i][j]; }

    //start timer
    /* Set the interval timer to 32 bit max */
    *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
    *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
    *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
    jacobi(aa, channels,d2,vv,&it_num, &rot_num); // from numerical receipts
    for (i=0;i<N;i++) for (j=0;j<N;j++) { MtrxEivec[i][j]=vv[i][j]; }
    for (i=0;i<N;i++) { VctrEivals[i]=d2[i]; }
    //get timer value and compute elapsed time
    *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0 
    // Make a counter snapshot by wrting a dummy value 
    *(interval_timer_ptr + 0x4) =  0;
    // read the 32-bit counter snapshot from the 16-bit timer registers
    high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
    counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
    Value21 = (0x7FFFFFFF - counter); // Clock cycle divided by CPU frequency
//}
  printf("4a) Jacobi NR eig cycles: %d\r\n", Value21);
  printf("4a) Jacobi NR eig   time: %d %s\r\n", (int) (Value21*Tscale),TB);

//if (ATYPE == 1) {
    printf("*** Method is Jacobi/Rutishauser by src: John Burkhardt...\n\r");
    for (i=0;i<N;i++) for (j=0;j<N;j++) { a[i+j*N]=MtrxCxx[i][j]; }
    //start timer
    /* Set the interval timer to 32 bit max */
    *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
    *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
    *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
    jacobi_eigenvalue_float ( channels, a, 100, b, d2, &it_num, &rot_num );
    for (i=0;i<N;i++) for (j=0;j<N;j++) { MtrxEivec[i][j]=b[i+j*N]; }
    for (i=0;i<N;i++) { VctrEivals[i]=d2[i]; }
    //get timer value and compute elapsed time
    *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0 
    // Make a counter snapshot by wrting a dummy value 
    *(interval_timer_ptr + 0x4) =  0;
    // read the 32-bit counter snapshot from the 16-bit timer registers
    high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
    counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
    Value21 = (0x7FFFFFFF - counter); // Clock cycle divided by CPU frequency
    printf("4b) Jacobi JB-FSU EV  cycles: %d\n", (int) Value21);
    printf("4b) Jacobi JB-FSU EV    time: %d %s\r\n", (int) (Value21*Tscale),TB);
//}

//if (ATYPE == 3) {
    printf("*** Method is QL after Wilkison et al. (1968) by src: Numerical Receips...\n\r");
    for (i=0;i<N;i++) for (j=0;j<N;j++) { zz[i+1][j+1]=MtrxCxx[i][j]; }
    //start timer
    /* Set the interval timer to 32 bit max */
    *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
    *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
    *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
    tred2  ( zz, N, d3, ee);// from Numerical Recipes
    if (DEBUG) {
      printf("--- The 5x5 Q matrix is * 100000 (int) ...\n\r");
      for (i = 1 ; i <= channels ; i++) {
        for (j=1; j<=channels; j++) {
          Value1 = zz[i][j] * 100000;
          printf("%d , ",Value1);
        }
        printf("\n\r");
      }
      printf("--- Diagonal elements d[] are...\n\r");
      for (i = 1 ; i <= channels ; i++) {
        Value1 = d3[i] *100000;
        printf("d[%d]= %d /100000 = %d\r\n",i,Value1,Value1/1000);
      }
      printf("--- Off Diagonal elements e[9 are...\n\r");
      for (i = 1 ; i <= channels ; i++) {
        Value1 = ee[i] *100000;
        printf("d[%d]= %d /100000 = %d\r\n",i,Value1,Value1/1000);
      }
    }
    tqli( d3, ee, N, zz);
    for (i=0;i<N;i++) for (j=0;j<N;j++) { MtrxEivec[i][j]=zz[i+1][j+1]; }
    for (i=0;i<N;i++) { VctrEivals[i]=d3[i+1]; }    
    //get timer value and compute elapsed time
    *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0 
    // Make a counter snapshot by wrting a dummy value 
    *(interval_timer_ptr + 0x4) =  0;
    // read the 32-bit counter snapshot from the 16-bit timer registers
    high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
    counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
    Value21 = (0x7FFFFFFF - counter); // Clock cycle divided by CPU frequency
    printf("4c) QL  cycles: %d\n", (int) Value21);
    printf("4c) QL    time: %d %s\n", (int) (Value21*Tscale),TB);
    Total += Value21;
//}
  if (DEBUG) {
    printf("--- Jacobi: Iterations=%d  Rotations=%d\n\r",it_num,rot_num);
    printf("--- The %dx%d Eigenvector matrix is * 1000 (int) ...\n\r",channels,channels);
    for (i = 0 ; i < channels ; i++) {
      for (j=0; j<channels; j++) {
        Value1 = MtrxEivec[i][j] * 1000;
        printf("%d , ",Value1);
      }
      printf("\n\r");
    }
  }

  if (DEBUG){
    printf("--- The Eigenvalues are...\n\r");
    for (i = 0 ; i < channels ; i++) {
      Value1 = VctrEivals[i] *100000;
      printf("EV[%d]= %d /100000 = %d\r\n",i,Value1,Value1/1000);
    }
  }
  // Multiply Eigenvalues with Eigenvectors
  // Eigenvalues Matrix has only diagonal elements
  // No general matrix multiply required
  //MtrxCxx is reused and used as buffer matrix
  printf("*** Done 3x Calculate the Eigenvectors and Eigenvalues\n\r");
  printf("*** Calculate the Reconstruction for %d Channels...\n\r",(REND-RSTART)/RINC+1);
  //start timer
  /* Set the interval timer to 32 bit max */
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
  *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
  *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
  for (i = 0 ; i < channels ; i++)  {
    for (j = 0 ; j < channels ; j++) {
      MtrxCxx[i][j] = MtrxEivec[j][i] / sqrtf(VctrEivals[i]);
    }
  }
  //Compute Output values
  //Compute all output values or the required 2+4
  for (i=RSTART;i<=REND;i+=RINC) { // Do mother and fetus PC only: use K02 k+=2
    for (k = 0; k< sample_window ; k++) {
      send_array[i][k]=0;
      for (j = 0; j< channels ; j++) {
        send_array[i][k] += (MtrxCxx[i][j] * MtrxOrig[j][k]);
      }
    }
  }
  //get timer value and compute elapsed time
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0 
  // Make a counter snapshot by wrting a dummy value 
  *(interval_timer_ptr + 0x4) =  0;
  // read the 32-bit counter snapshot from the 16-bit timer registers
  high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
  counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
  Value21 = (0x7FFFFFFF - counter); // Clock cycle divided by CPU frequency
  printf("5) Reconstruct %d Ch. cycles: %d\n", (REND-RSTART)/RINC+1,Value21);
  printf("5) Reconstruct %d Ch.  time: %d %s\n",(REND-RSTART)/RINC+1, (int) (Value21*Tscale),TB);
  Total += Value21;
  
  printf("*** Calculate %d Ch. BPM for mother and fetus...\n\r",(REND-RSTART)/RINC+1);
  //start timer
  /* Set the interval timer to 32 bit max */
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
  *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
  *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
  if (DEBUG) {
    printf("*** 1) square first to reduce noise\n\r");
    printf("*** 2) Moving average of 5 sample \n\r");
    printf("*** 3) Find maximum in window\n\r");
    printf("*** 4) Inspected data and find first maximum (x1,v1)\n\r");
    printf("*** 5) Inspected data and find second maximum (x2,v2)\n\r");
    printf("*** 6) Compute BPM via 250/(x2-x1)*60\n\r");
    printf("*** 7) Determine S/N (reliability) via S range=>x1+10;x2-10; N= x2+10...x2-10\n\r");
  }
  for (k=RSTART;k<=REND;k+=RINC) { // Do mother and fetus PC only: use K02 k+=2
    for (j = 0; j< sample_window ; j++) { // 1) Square
      send_array[k][j]*=send_array[k][j];
      ymean[k][j]=0.0;
    }
    for (j = 0; j< sample_window ; j++) { // 2) Mean
      if (j>5) {
        for (i=0;i<5;i++) {
          ymean[k][j] += send_array[k][j-i];
        }
      } else {
        ymean[k][j] = send_array[k][j];
      }
    }
    Vmax=0.0;
    for (j = 0; j< sample_window ; j++) { // 3) Get max
      if (ymean[k][j]>Vmax) {
        Vmax=ymean[k][j];i=j;
      }
    }
    //======= Step 4+5: first the first and second maximum
    Vmax50=Vmax*0.5; j=Vmax;
    if (DEBUG) printf(" Search Ch. %d: max is %d at %d\n\r",k,j,i);
    first=1;second=0;v1=0;x1=0;v2=0;x2=0;
    for (i=0; i<= sample_window; i++) {
      if ((first==1) && (ymean[k][i] > v1) && ( ymean[k][i] > Vmax50)) {
        v1 = ymean[k][i]; x1=i; j= v1;
        if (DEBUG) printf("First: new max %d at %d\n\r",j,i);
      }
      if (((i-x1) > 70) && (first==1) && (v1>Vmax50) && (second==0)) {
        second=1;first=first+1;
        if (DEBUG) printf("Look for second: at  %d \n\r",i);
      }
      if ((second==1) && (ymean[k][i] > v2) && (ymean[k][i] >Vmax50)) {
        v2 = ymean[k][i]; x2=i; j=v2;
        if (DEBUG) printf("Second: new max %d at %d\n\r",j,i);
      }
      if (((i-x2) > 30) && (second==1) && (v2>Vmax50)) {
        second=second+1;
        if (DEBUG) printf("Done with second: at %d\n\r",i);
      }
    }

  //======= Step 6: Compute BPM
  dx=x2-x1; HR=250.0/dx*1000; BPM=250.0*60/dx;
  if (DEBUG) printf(" --> Heart rate Ch %d Difference=%d => %d /1000 Hz => %d BPM\n\r",k,dx,HR,BPM);

  //=======  Step 7: Compute S/N
  nsum=0.0;
  for (i=x1+10;i<=x2-10;i++) { /// Add the noise
      nsum = nsum + ymean[k][i] ;
  }
  ssum=0.0;
  for (i=0;i<10;i++) { /// Add the signal
      ssum=ssum+ ymean[k][x1+i];
      ssum=ssum+ ymean[k][x2-i];
  }
  i=ssum; j=nsum; u=10.0*log10(ssum/nsum)*1000;
  if (DEBUG) printf(" --> Ch %d S/N %d/%d = %d /1000 dB\n\r",k,i,j,u);
  }

  //get timer value and compute elapsed time
  *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0 
  // Make a counter snapshot by wrting a dummy value 
  *(interval_timer_ptr + 0x4) =  0;
  // read the 32-bit counter snapshot from the 16-bit timer registers
  high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
  counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
  Value21 = (0x7FFFFFFF - counter); // Clock cycle divided by CPU frequency
  printf("6) BMP + S/N %d Ch. cycles: %d\n", (REND-RSTART)/RINC+1, Value21);
  printf("6) BMP + S/N %d Ch.   time: %d %s\n", (REND-RSTART)/RINC+1,(int) (Value21*Tscale),TB);
  Total += Value21;
  printf("==============================\n");
  printf("== Total cycles: %d   \n", Total);
  printf("== Total   time: %d ms\n", (int) (1000.0*Total/Fcpu));
  printf("== Frame per second: %d\n", (int) (Fcpu/Total));
  printf("==============================\n");   
  while (seconds<5) {
    // Make a counter snapshot by wrting a dummy value 
    *(interval_timer_ptr + 0x4) =  0;
    // read the 32-bit counter snapshot from the 16-bit timer registers
    high_half = *(interval_timer_ptr + 0x5) & 0xFFFF;
    counter = (*(interval_timer_ptr + 0x4) & 0xFFFF) | (high_half << 16);
    Value21 = (0x7FFFFFFF - counter); // Clock cycle
    if (Value21 > 100000000){ // second count is 100 MHz cycles per second
      printf("sec=%d \n",seconds);
      seconds++;
      SW_value = *(SW_switch_ptr);  // Read the SW slider switch values
      *(red_LED_ptr) = second + SW_value;       // Light up the red LEDs
      *(interval_timer_ptr + 1) = 0x8;  // Set STOP=1  START = 0, CONT = 0, ITO = 0  
      *(interval_timer_ptr + 0x2) = 0xFFFF;  *(interval_timer_ptr + 0x3) = 0x7FFF;
      *(interval_timer_ptr + 1) = 0x4;  // Set START = 1, CONT = 0, ITO = 0
    }
  }
  printf("\n");
  printf("*** FECG calculation on Nios II completed\n\r");
  return 0;
}
