LIBRARY ieee; 
USE ieee.std_logic_1164.ALL;
USE ieee.std_logic_arith.ALL;
USE ieee.std_logic_unsigned.ALL;
-- ========================================================
ENTITY urisc IS                      --> Interface
  PORT(CLK      : IN  STD_LOGIC; -- System clock
       reset    : IN  STD_LOGIC; -- Active low asynchronous reset
       in_port  : IN  STD_LOGIC_VECTOR(7 DOWNTO 0); -- Input port
       out_port : OUT STD_LOGIC_VECTOR(7 DOWNTO 0)); -- Output port
END;
-- ========================================================
ARCHITECTURE fpga OF urisc IS

  COMPONENT rom128x16 IS
  PORT(clk      : IN STD_LOGIC;
       address  : IN STD_LOGIC_VECTOR(6 DOWNTO 0);
       data     : OUT STD_LOGIC_VECTOR(15 DOWNTO 0));
  END COMPONENT;

  TYPE STATE_TYPE IS (FE, DC, EX);
  SIGNAL state      : STATE_TYPE;

-- Register array definition
  TYPE RTYPE IS ARRAY(0 TO 15) OF STD_LOGIC_VECTOR(7 DOWNTO 0);
  SIGNAL r : RTYPE;

-- Local signals
  SIGNAL data, ir : STD_LOGIC_VECTOR(15 DOWNTO 0);
  SIGNAL mode, jump : STD_LOGIC;
  SIGNAL rd, rs : INTEGER RANGE 0 TO 15;
  SIGNAL pc, address : STD_LOGIC_VECTOR(6 DOWNTO 0); 
  SIGNAL dif      : STD_LOGIC_VECTOR(7 DOWNTO 0);

BEGIN

  prog_rom: rom128x16           -- Instantiate the LUT
  PORT MAP (clk     => clk,     -- System clock
            address => pc,      -- Program memory address
            data    => data);   -- Program memory data

  FSM: PROCESS (reset, clk)    --> FSM with ROM behavioral style
    VARIABLE  result : STD_LOGIC_VECTOR(8 DOWNTO 0);     -- Temporary register
  BEGIN                                    
    IF reset = '0' THEN               -- asynchronous reset
      FOR k IN 1 TO 15 LOOP
        r(k) <= CONV_STD_LOGIC_VECTOR( -1, 8); -- all set to -1
      END LOOP;
      pc <= (OTHERS => '0');
      state <= FE;
    ELSIF rising_edge(clk) THEN
    CASE state IS
      WHEN FE =>           -- Fetch instruction 
        ir <= data;           -- Get the 16-bit instruction 
        state <= DC;
      WHEN DC =>           -- Decode instruction; split ir 
        rd <= CONV_INTEGER('0' & ir(15 DOWNTO 12));  -- MSB has destination
        rs <= CONV_INTEGER('0' & ir(11 DOWNTO 8));   -- second source operand
        mode <= ir(7);           -- flag for address mode
        address <= ir(6 DOWNTO 0); -- next PC value
        state <= EX;
      WHEN EX =>          -- Process URISC instruction
        result := (r(rd)(7) & r(rd)) - (r(rs)(7) & r(rs));
        IF rd > 0 THEN
          r(rd) <= result(7 DOWNTO 0); -- do not write input port
        END IF;
        IF result(8) = '0' THEN  -- test is false inc PC
          pc <= pc + 1; 
        ELSIF mode = '1' THEN  -- result was negative
          pc <= address;  -- absolute addressing mode
        ELSE
          pc <= pc + address; -- relative addressing mode
        END IF; 
        r(0) <= in_port;
        out_port <= r(15);
        state <= FE; 
    END CASE;
      jump <= result(8);
      dif  <= result(7 DOWNTO 0);
    END IF;
  END PROCESS;

END ARCHITECTURE fpga;
