// ========================================================
// IEEE STD 1364-2001 Verilog file: urisc.v 
// Author-EMAIL: Uwe.Meyer-Baese@ieee.org
// ========================================================
// Title: URISC microprocessor 
// Description: This is the top control path/FSM of the 
// URISC, with a 3 state machine design
// ========================================================
module urisc 
 (input  CLK,              // System clock
  input  RESET,            // Active low asynchronous reset
  input [7:0] IN_PORT,     // Input port
  output reg [7:0] OUT_PORT);// Output port
// ========================================================

// OP Code of instructions:
  parameter FE=0, DC=1, EX=2;
  reg [1:0] STATE;
  
// Register array definition
  reg [7:0] R [15:0];
  
// Local signals
  wire [15:0] DATA; 
  reg [15:0] IR;
  reg MODE; 
  wire JUMP;
  reg [3:0] RD, RS;  
  reg [6:0] ADDRESS, PC; 
  wire [7:0] DIF;
  reg [8:0] RESULT;

  rom128x16 prog_rom    // Instantiate the LUT
  ( .CLK(CLK),          // System clock
    .ADDRESS(PC),       // Program memory address
    .DATA(DATA));       // Program memory data

//FSM with ROM behavioral style
  always @(posedge CLK or negedge RESET) 
  begin : States                 // URISC in behavioral style 
      integer k;            // Temporary counter
      if (~RESET) begin    // all set register to -1
        STATE <= FE;
        for (k=1; k<=15; k=k+1) R[k] = -1;
        PC <= 0;        
      end else begin    // use rising edge
        case (STATE)  
        FE: begin         // Fetch instruction 
              IR <= DATA; // Get the 16-bit instruction
              STATE <= DC;
            end 
        DC: begin       // Decode instruction; split ir
              RD <= IR[15:12]; // MSB has destination
              RS <= IR[11:8];// second source operand
              MODE <= IR[7];  // flag for address mode
              ADDRESS <= IR[6:0]; // next PC value
              STATE <= EX;
            end
        EX: begin     // Process URISC instruction
              RESULT = {R[RD][7], R[RD] } - {R[RS][7], R[RS] };
              if (RD > 0) R[RD] <= RESULT[7:0]; // do not write input port
              if (~RESULT[8]) begin PC <= PC + 1;  // test is false inc PC
                  end else begin  // result was negative
                    if (MODE) PC <= ADDRESS;  // absolute addressing mode
                    else PC <= PC + ADDRESS; // relative addressing mode
                  end
              R[0] <= IN_PORT;
              OUT_PORT <= R[15];
              STATE <= FE;
              end
        default : STATE <= FE; 
        endcase
      end
  end    
  
  // Extra test pins:
  assign JUMP = RESULT[8];   // 
  assign DIF = RESULT[7:0];    // 

endmodule
