function [U,ex,XX,YY,Timexy]=circle_cartesian(M,plt)
%CIRCLE_CARTESIAN generates figure and table for Example 10.7
%   Essential PDES function: DFG; 30 June 2015.

% Solves Laplace's equation on a quarter circle in Cartesian coordinates.
% This function is called by the file
%       circle_driver
% to generate figures and table for Example 10.7 from
%     Essential PDEs, Griffiths, Dold & Silvester

% Plot results if plt>0
% Output:
%  U: numerical solution as a square array (equal to nan outside domain)
%  ex: the exact solution
%  XX,YY coordinates of rectangular grid
%  Timexy: 1) set-uo time, 2) solve time
% Modified: dfg on 28-05-2015
tic
% Define exact solution - used to specify Dirichlet BCs
exact = @(x,y)  x.^5-10*x.^3.*y.^2+5.*x.*y.^4;

h = 1/M;
m = M-1;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Set up grid
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
x = h:h:1-h; y = 0:h:1-h;       % interior grid coordinates
[XX, YY] = meshgrid(0:h:1);		% all grid pts in unit square
% Find coords where y = mh grid lines cut circle
xb = sqrt(1-y.^2);
Ix = floor(xb/h);
dx = xb - Ix*h;                 % horiz dist to point on circle
% Adjust index if grid point exactly on circle
k = dx == 0;  Ix = Ix - k; dx = dx + h*k;
% Find coords where x = l.h grid lines cut circle
% (Could be done by symmetry)
yb = sqrt(1-x.^2);
Iy = floor(yb/h);
dy = yb-Iy*h;
k = dy == 0;  Iy = Iy - k; dy = dy + h*k;

Iy = Iy+1;			% y=0 has to be counted in each col
c=find(XX.^2+YY.^2>=1 | XX == 0);	% identify pts outside the circle
Xg=XX;Xg(c)=nan;
Yg=YY;Yg(c)=nan;
Xb = Xg; Yb = Yg;
Xb(:,1) = zeros(M+1,1); Yb(:,1) = (0:M)'*h;
for j = 1:M
  Xb(j,Ix(j)+2) = xb(j);
  Yb(j,Ix(j)+2) = y(j);
  if j <= m, 
    if plt, plot(j*h*[1,1],[0 yb(j)],':'),end
    Yb(Iy(j)+1,j) = yb(j);
    Xb(Iy(j)+1,j) = x(j);
  end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Build Ax matrices - row-wise on the grid
%% i.e., matrices representing -u_xx as though dealing with
%% the entire square. The equations at internal neighbours of the
%% circular boundary will then be adjusted.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
N = M*m;
e = ones(m,1);
T = spdiags(e*[-1 2 -1],-1:1,m,m)/h^2;
Ax = kron(speye(M),T);      % matrix for entire square
Fx = zeros(N,1);			% rhs due to x-derivatives
%---------------------------------------
%% go thru' each block & adjust "bottom" row
%% for Neumann BC
%---------------------------------------
xyza = zeros(M,3);
for j = 1:M
  idx = (j-1)*m + Ix(j);
  hb = 2/(h+dx(j));
  % next adjust approximation of -u_xx next to boundary
  Ax(idx,idx-1:idx+1) = hb*[-1/h  (1/h+1/dx(j)) 0];
  if Ix(j)<(M-1)
      Ax(idx+1,idx)=0;
  end
  z = exact(xb(j),y(j));   %  use the exact solution of the problem,
						   %  to define the boundary values on the circular boundary.
  xyza(j,:)=[xb(j),y(j),z];
  Fx(idx) = hb/dx(j)*z;        % sets up the appropriate RHS: matrix coeffs times boundary values.
  if Ix(j) < m, Fx(idx+1:j*m) = 0;end
end
Axo=Ax;Fxo=Fx;
%% Now permute rows & cols - we switch from ordering by row to ordering by columns.
z = reshape(1:N,m,M)'; z = z(:);
Ax = Ax(z,z); Fx = Fx(z);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Build Ay matrix - column-wise on the grid
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
m=M-1;
N = M*m;       % Number of grid points inside the rectangle 0<=x<1, 0<y<1
xyzb=zeros(m,3);
e = ones(M,1);
Ty = spdiags(e*[-1 2 -1],-1:1,M,M)*M^2;
Ty(1,2)=-2*M^2;             % due to Neumann BC at y=0
Ay = kron(speye(m),Ty);
Fy = zeros(N,1);			% rhs due to x-derivatives
%---------------------------------------
%% go thru' each block & adjust "bottom" row
%%---------------------------------------
for j = 1:m
  idx = (j-1)*M + Iy(j);
  hb = 2/(h+dy(j));
  % next adjust approximation of -u_yy next to boundary
  Ay(idx,idx-1:idx+1) = hb*[-1/h  (1/h+1/dy(j)) 0];
  z = exact(x(j),yb(j));
  xyzb(j,:) = [x(j),yb(j),z];
  Fy(idx) = hb/dy(j)*z;
  if Iy(j) < M, Fy(idx+1:j*M) = 0;end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Now assemble overall matrix
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
A = Ax + Ay;
Timexy(1) = toc;
v = A\(Fx+Fy);    %Solve!
Timexy(2) = toc;
U=reshape(v,M,m); % change from column vector to square array
U = [U;zeros(1,m)];
U = [zeros(M+1,1),U,[exact(1,0);nan*ones(M,1)]];
U(c)=nan;
% U is set to nan for points outside the domain. Values of v there
% are ignored.
if plt
%%%%%%%%%%% plot grid & domain
  theta = (0:100)*pi/200; % polar angle used in plots
  figure(1)
    plot(Yg,Xg,'k:',Xg,Yg,'k:')
    hold on
    plot((Ix*h+dx),y,'o',cos(theta),sin(theta),':','linewi',1,'markersi',7)
    plot(x,(Iy*h+dy-h),'x','linewi',1,'markersi',8)%,x,yb,'g-')
    hold off
    set(gca,'ytick',0:.25:1), set(gca,'xtick',0:.25:1)
    axis([ 0 1 0 1]), axis('square')
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  plot results                     %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
ex = exact(Xg,Yg);  % exact solution at internal grid points
% cn = isnan(ex);     
% ck = find(cn==0);
% errxy=max(max(abs(ex(ck)-U(ck))));
erp = U-exact(XX,YY);
% fprintf('Max error with M = %g is %g\n',M,errxy );
% fprintf('Timings: Setup = %g,  Solve = %g\n',Timexy);
U(:,1)=0;
if plt
    figure(2), clf
      mesh(XX,YY,U);
      view([60,50])
      colormap([0 0 0])
      xlabel('$x$','fontsi',30)
      ylabel('$y$','fontsi',30)
      set(gca,'fontsi',16)
      hold on
      plot3(cos(theta),sin(theta),exact(cos(theta),sin(theta)),'-')
 %    fill in plot lines from curved boundary to nearest neighbour. 
 %    Change 'k-' to 'r-' to see the effect
  for j = 1:M,
    z = exact(xb(j),y(j));
    plot3(h*Ix(j) +[0 dx(j)], (j-1)*h*[1 1], [U(j,Ix(j)+1) z],'r-')
    if j<M
        z = exact(x(j),yb(j));
        plot3((j)*h*[1 1],h*Iy(j)-h+[0 dy(j)],[U(Iy(j),j+1) z],'r-')
         %  [1 (j)*h*[1 1],h*Iy(j)-h+[0 dy(j)],[erp(Iy(j),j+1) z]]
    end
  end
    plot3((Ix*h+dx),y,exact(Ix*h+dx,y),'.')
    plot3(x,(Iy*h-h+dy),exact(x,Iy*h-h+dy),'k.')
    plot3([0 0],[1-h 1],[0 0],'-')
    plot3([0 ],[ 1],[0],'k.')
    figure(3)
    % Repeat figure 2 with z = global error
    clf
    erp(:,1)=0;
    mesh(XX,YY,erp);
      view([60,50])
      colormap([0 0 0])
      xlabel('$x$','fontsi',30)
      ylabel('$y$','fontsi',30)
      set(gca,'fontsi',16)
      hold on
      plot3(cos(theta),sin(theta),zeros(size(theta)),'-')
   %    fill in plot lines next to curved boundary
    for j = 1:M,
        z = 0;
        plot3(h*Ix(j) +[0 dx(j)], (j-1)*h*[1 1], [erp(j,Ix(j)+1) z],'r-')
        if j<M
            z = 0;
            plot3((j)*h*[1 1],h*Iy(j)-h+[0 dy(j)],[erp(Iy(j),j+1) z],'r-')
            %   [2 (j)*h*[1 1],h*Iy(j)-h+[0 dy(j)],[erp(Iy(j),j+1) z]]
        end
    end
    plot3((Ix*h+dx),y,0*y,'k.')
    plot3(x,(Iy*h-h+dy),0*x,'k.')
    plot3([0 0],[1-h 1],[0 0],'k-');
    plot3([0 ],[ 1],[0],'.')
    axis tight % otherwise small negative values get cropped
    % output figures 2 & 3 if M=16
    printfig('circle_cartesian',2:3,M==16); 
end
