function [U,ex,Timexy]=circle_polar(M,plt)
%CIRCLE_POLAR generates figure and table for Example 10.12
%   Essential PDES function: DJS; 30 June 2015.

% Solve Laplace's equation on a quarter circle in Polar coordinates.
% This function is called by the file
%       circle_driver
% to generate figures and table for Example 10.12 from
%     Essential PDEs, Griffiths, Dold & Silvester

% Plot results if plt>0
% Output:
%  U: numerical solution as a square array (equal to nan outside domain)
%  ex: the exact solution
%  XX,YY coordinates of rectangular grid
%  Timexy: 1) set-uo time, 2) solve time
% Modified: dfg on 28-05-2015

tic;
% Define exact solution - used to specify Dirichlet BCs
exact = @(x,y)  x.^5-10*x.^3.*y.^2+5.*x.*y.^4;
h  = 1/M;	          % grid size in r direction
dt = 0.5*pi*h;	      % grid size in theta direction
t = (0:h:1)*0.5*pi;   % grid lines in theta direction
c  = 1-h/2;	          % multiplier for the boundary term
rat = (h/dt)^2;
R = (h:h:(1-h));	  % interior r-grid points
Rh = (h/2:h:(1-h));   % intergrid r-values
m = length(R);
J = speye(M,M);	J(1,1)=.5;          % identity except for 1/2 to account for 
                                    % bc at theta = 0
tmp = spdiags([-[Rh(2:m) 0]' 2*R' -Rh'],-1:1,m,m)/h^2;
                         % tmp is matrix version of -delta_r(r delta_r U)
                         % on any theta  constant grid line
Ar = kron(tmp,J);	     % approx of r derivatives at all internal points

e0 = [1; 2*ones(M-1,1)];
e = ones(M,1);
tmp = spdiags([-e e0 -e],[-1:1],M,M);  % approx of t-derivatives on any
                                       % r = constant grid line
At = kron(tmp,speye(m))/(dt^2);	  % approx of t-derivatives at all grid points
                                  % not the same order of points as Ar
order = reshape(1:M*m,m,M)';      % number all internal grid points 1:M*m
order = order(:);                 % reshape these into column vector
At = At(order,order);			% reorder rows & cols of the theta matrix
D = kron(diag(1./R),speye(M));
At = D*At;                      % need coeff 1/r of theta derivatives
A = Ar + At;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  set BCs at r = 1                  %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
bv = exact(cos(t),sin(t))';  % boundary values
% now calcuate source vector
f = [zeros(M*(m-1),1); [.5*bv(1);bv(2:M)]*(1-h/2)*M^2];
Timexy(1) = toc; tic
v = A\f;          % Solve!
Timexy(2) = toc;

V=reshape(v,M,m);   % col vector v -> M x m matrix V
                    % next add 0 boundary values at theta = pi/2
V  = [V;zeros(1,m)];
U =  [zeros(M+1,1) V bv]; % add boundary values at r=0 and r=1

ex =  bv*([0 R 1].^5);     % the exact solution
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  plot results if plt>0             %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if plt
    figure(10)
     R = (0:h:1);
     [RR,TT] = meshgrid(R,t);
     mesh(RR,TT,U) %%%%%% 3D plot of solution in (r,theta) plane
     xlabel('$r$','fontsi',20)
     ylabel('$\theta$','fontsi',20)
     set(gca,'fontsi',16)
    figure(11)
      XX = RR.*cos(TT);YY=RR.*sin(TT);
      mesh(XX,YY,U) %%%%%% 3D plot of solution in (x,y) plane
      view([60,50])
      colormap([0 0 0])
      xlabel('$x$','fontsi',30)
      ylabel('$y$','fontsi',30),
      set(gca,'fontsi',16)
    E = (ex-U);   	     % the global error
    figure(12)
      mesh(XX,YY,E); %%%%%% 3D plot of E in (x,y) plane 	
      view([60,50])
      colormap([0 0 0])
      err = max(max(abs(E)));	 % max global error	
      xlabel('$x$','fontsi',30)
      ylabel('$y$','fontsi',30),
      set(gca,'fontsi',16);
    % output figures 11 & 12 
    printfig('circle_polar',10:11);
end