function [G,E,X,Y,times] = poisson(M,prnt)
%POISSON generates figures and table for Example 10.1
%   Essential PDES function: DJS; 30 June 2015.

% This function is called by the file
%       square_driver
% to generate figures and table for Example 10.1 from  
%     Essential PDEs, Griffiths, Dold & Silvester
% 
% Solves Poisson's equation on the unit square 
% Lu = f(x,y) in R
%  u = g(x,y) on boundary
% with a grid of size h = 1/M
% f is defined by the function rhs(x,y), see below
% g is defined by the function exact(x,y), see below
% Output:
%   G: Numerical solution
%   E: Exact solution
%   error = E-G
%   X,Y the grid points
%   times, a 1x2 array
%     times(1) - time for factorization of coefficient matrix
%     times(2) - solve time
if nargin<2
    prnt = 1; % print the matrices when prnt>0
end
h = 1/M;
file = 'poisson'; % stem of file-name for saving figures
% Define the grid points
[X,Y] = meshgrid((0:M)/M);
% internal grid points
x = X(2:end-1,2:end-1);
y = Y(2:end-1,2:end-1);
% build matrix A
e = ones(M-1,1);
D = spdiags(e*[-1, 2, -1],-1:1,M-1,M-1);
I = speye(M-1);
A = kron(D,I)+kron(I,D);
% forcing term
f = rhs(x,y); f = f(:); % convert to a column vector
% BCs
G = exact(X,Y,M);       % exact solution at all grid points
E = G; % save the exact solution for output
g = zeros((M-1)^2,1);   % required vector of BVs
% left 
g(1:M-1) = G(2:M,1);
% right 
g((M-1)^2-M+2:(M-1)^2) = G(2:M,M+1);
% bottom 
g(1:M-1:(M-1)^2) = g(1:M-1:(M-1)^2) + G(1,2:M)';
% top
g(M-1:M-1:(M-1)^2) = g(M-1:M-1:(M-1)^2) +G(M+1,2:M)';
% solve
tic
[L,R] = lu(A);
times(1) = toc;
if M==8 % Draw figure 10.3
    figure(40)
    spy(A), set(gca,'fontsi',20)
    printfig(file,40,prnt)
    %eval(['print -f40 -deps  -40
    figure(41)
    spy(L), set(gca,'fontsi',20)
    printfig(file,41,prnt)
    %print -f41 -deps poisson_square_xylog-41
    figure(42)
    spy(R), set(gca,'fontsi',20)
    printfig(file,42,prnt)
    %print -f42 -deps poisson_square_xylog-42
end
tic
U = R\(L\(h^2*f + g));
times(2)=toc;
U = reshape(U,M-1,M-1);
G(2:M,2:M)=U;
% G now contains the numerical solution + boundary values
%%%%%%%%%%%%%
function f = rhs(x,y)
% f = -u_xx-u_yy, where u is defined by the function exact(x,y)
a =5/4; b =5/4;
f =4*(2*x.*y-b*x-a*y)./((x-a).^2+(y-b).^2);
function u = exact(X,Y,M)
% Exact solution to 
a =5/4; b =5/4;
u = -X.*Y.*log((X-a).^2+(Y-b).^2);

    
    