%CRANK_NICOLSON_FIG generates figure 11.7
%   Essential PDES scriptfile: DJS; 30 June 2015.

% Example 11.19 of Essential PDEs
% by Griffiths, Dold & Silvester
% File: Crank_Nic_step.m
% Crank-Nicolson solution of the heat equation on 0<x<1 with
%              | -1, for 0<x<.5
% IC  u(x,0) = {  0, for x = .5
%              | +1, for .5<x<1
% Exact solution u(x,t) = erf((x-.5)/sqrt(t))
% 
% Dirichlet BCs taken from exact solution. 

% Uses M = 40 & k = h = 1/40 to compute first 3 time steps
% with
%   a) standard Crank-Nicolson
%   b) first half step is BTCS followed by standard Crank-Nicolson
% to produce the results in Fig. 11.7.
%

% Define the exact solution as a function of x & t
exact = @(x,t) erf((x-.5)/sqrt(t));

    M = 40;
    h = 1/M;
    x = h:h:1-h;    % internal grid points
    X = [0 x 1];
    r = M;
    k = r*h^2;
    r1 = 1-r; r2 = r/2; % coefficients
    % construct C-N coefficient matrix
    e = ones(M-1,1);
    Ac = spdiags(e*[-r2 (1+r) -r2],-1:1,M-1,M-1);
    Rc = chol(Ac);    % upper triangular factor in Cholesky factorization        
    Ub = sign(x-.5)'; % initial condition (column vector)
    Uc = Ub;
    figure(1), subplot(4,1,4)
          plot(X,[-1;Uc;1],'.-','markersize',15)
          xlabel('$x$')
          text(.1,0,['$n = ' int2str(0) '$'],'fontsi',25)
    figure(2), subplot(4,1,4)
          plot(X,[-1;Uc;1],'.-','markersize',15)
          xlabel('$x$')
          text(.1,0,['$n = ' int2str(0) '$'],'fontsi',25)
    ex = [-1;Ub;1];
    for n = 1:3
        t = n*k;
        ex_new = exact(X',t);        % exact solution at new time level
        Uc = [ex(1); Uc; ex(end)];   % include boundary values
        rhs= (r2*(Uc(1:M-1)+Uc(3:M+1))+r1*Uc(2:M));
        rhs([1,M-1]) = rhs([1,M-1]) + r2*[ex_new(1);ex_new(end)];
        Uc = Rc\(Rc'\rhs); 
        % Uc no longer contains boundary values
        if n==1
            ex_h = exact(X',k/2);    % exact solution at t = k/2 is needed
                                     % for BCs
            for i = 1:2
                rhs = Ub;
                rhs([1,M-1]) = rhs([1,M-1]) + r2*[ex_new(1);ex_new(end)];
                Ub = Rc\(Rc'\rhs);       %  half-step with BTCS
                ex_h = ex_new;       % update exact solution for 2nd half-step
            end
        else
            Ub = [ex(1); Ub; ex(end)];   % include boundary values
            rhs = (r2*(Ub(1:M-1)+Ub(3:M+1))+r1*Ub(2:M));   
            rhs([1,M-1]) = rhs([1,M-1]) + r2*[ex_new(1);ex_new(end)];
            Ub = Rc\(Rc'\rhs); 
        end
        figure(1), subplot(4,1,4-n)
          plot(X,ex,'-',X,[ex(1);Uc;ex(end)],'.--','markersize',15)
          text(.1,0,['$n = ' int2str(n) '$'],'fontsi',25)
        figure(2), subplot(4,1,4-n)
          plot(X,ex,'-',X,[ex(1);Ub;ex(end)],'.--','markersize',15)
          text(.1,0,['$n = ' int2str(n) '$'],'fontsi',25)
    end
    printfig('Crank_Nic_step',1:2);
    