%FTCS_FIG generates figure 11.3
%   Essential PDES scriptfile: DJS; 30 June 2015.

% Example 11.1 of Essential PDEs
% by Griffiths, Dold & Silvester
% File: ftcs.m
% FTCS solution of the heat equation on 0<x<1 with
% BCs u(0,t) = u(1,t) = 0 
% IC  u(x,0) = sin(pi*x).
% Uses M = 40 and r = 0.5 & 0.52 to produce Fig 11.3.
%
% Instability when r>0.5 in this example is caused by growth of
% rounding error. This is illustrated by computing the solution 
% in two different ways, one by matrix multiplication: u^(n+1) = A*u^n.
% The other solution is denoted by U.
% Note the difference in the amplitude of the oscillations in 
% figures 2 & 4. The level of rounding error is about 1e-16 and the 
% amplification factor with r = 0.52 raised to the power n in the final 
% plot is about 1e+16 so the  product is around 1.
% Figure 4 is the one used in Example 11.1.

% Define the exact solution as a function of x & t
exact = @(x,t) exp(-pi^2*t).*sin(pi*x);
M = 40;
h = 1/M;
x = h:h:1-h;    % internal grid points
X = [0 x 1];
R = [0.5 0.52]; % the two values of the mesh ration r =h/h^2
Tf = .2;        % final time
for j = 1:2
    r = R(j);
    r2 = 1-2*r;
    k = r*h^2;
    N = ceil(Tf/k);  % number of time steps
    e = ones(M-1,1);
    E = [1;e;1];     % extend e by 2
    A = spdiags(e*[r (1-2*r) r],-1:1,M-1,M-1);
    u = exact(x,0)'; % initial condition - column vector
    U = u;
    for i = 0:1      % plot initial condition in 2 figures
        figure(j+2*i)
            subplot(4,1,1:3)
            plot3(X,0*E,[0;u;0],'k.-')
            view(70,35);
            hold on
    end
    for n = 1:N
        t = n*k;
        U = [0;U;0];   % include boundary values
        U = r*(U(1:M-1)+U(3:M+1))+r2*U(2:M); 
        u = A*u;
        if ~rem(n,80)
            ex = exact(X,t);
            figure(j)
                subplot(4,1,1:3)
            plot3(X,t*E,ex,'k-',x,t*e,u,'k.--')
            figure(j+2)
                subplot(4,1,1:3)
            plot3(X,t*E,ex,'k-',x,t*e,U,'k.--')
            if norm(ex(2:end-1)'-u,inf)>0.01
                disp(['Instability detected:'])
                fprintf('  t =  %5.3f, (n = %4i steps), |xi|^%4i = %0.4g\n',t,n,n,abs((1-4*r)^n))
                break % stop the integration
            end
        end
    end
end

    for j = 1:4
        figure(j)
        grid on
        axis([0 1 0 Tf 0 1])
        xlabel('$x$')
        ylabel('$t$')
    end
    printfig('ftcs_b',1:4);
    