function [x,u0,n,X,U0] = hyp1d(method,L,T,M,c,a,plt)
%HYP1D  solves transport equation using different FD schemes
%   Essential PDES function: DJS; 30 June 2015.

%    Title: 	Example using Lax-Wendroff for u_t + a u_x = 0, 0 < x < L
%    Created:	Feb 14 1998
%    Modified:  June 1 2015
%    Author: 	D.F.Griffiths
%
% Periodic boundary conditions
%
% Usage:  u = hyp1d(method,L,T,M,c,a);
% Essential parameter: method - a string, one of
%                  'ftbs'
%                  'btbs'
%                  'lw'
%                  'fr'    (Lax-Friedrichs)
%                  'lf'    (Leap-Frog)
%                  'quick' (3rd order upwind)
%                  'taygal'(3rd order implicit: Taylor-Galerkin)
%                  'box'
% Only the first two characters are significant. 
% The remaining input parameters are optional and take default values if not
% specified.                                           DEFAULT
%   L       Length of domain: 0 < x < L                   4
%   T       Final time of integration: 0 < t <= T         1 (period)
%   M       x(m) = L*m/M (h = L/M)                         400
%   c       Courant Number                               0.45
%   a       speed                                         1
%   plt     plot after each time step if plt>0

% set defaults if insufficient input arguments
if nargin < 7
    plt = 1;
end
if nargin < 6
  a = 1;
end
if nargin < 5
  c = 0.45;
end
if nargin < 4
  M = 90;				%  number of grid points/unit length
end
if nargin < 4
  T = 3;
end
if nargin < 2
  L = 3;
end
if nargin < 1
  method = 'lw';
end

h = L/M;
x=h:h:L;
N = length(x);
x0=[0 x];   
  m = max(size(x0));
k = h*c/a;				 %  time step 

explicit=0;				 %  Label explicit methods (=1)
threelevel = 0;
switch method(1:2)
    case  'ft'
      cl = c;  cc = 1-c; cr = 0; cll = 0;
      name = 'FTBS';
      explicit = 1;
    case 'bt'
      explicit = 0;
      name = 'BTBS';
    case 'lw'
      cl = c*(1+c)/2; 
      cc = 1-c^2;	
      cr = c*(c-1)/2;
      cll = 0;
      name = 'Lax-Wendroff';
      explicit = 1;
    case 'fr'
      cl = (1+c)/2; 
      cc = 0;	
      cr = (1-c)/2;
      cll = 0;
      name = 'Lax-Friedrichs';
      explicit = 1;
    case 'lf'
      threelevel = 1;
      name = 'Leapfrog';
    case 'qu'
      cl = c*(1+c)*(2-c)/2;
      cc = (1-c^2)*(2-c)/2;	
      cr = c*(c-1)*(2-c)/6;
      cll = -c*(1-c^2)/6;
      name = '3rd order upwind';
      explicit = 1;
    case 'bo'
      explicit = 0;
      name = 'Box';
end
u0 = hyp_ic(M)';         	 %  Initial data
n = round(T/k);				 %  number of time levels
X = 0:.01:L;
%figure(1)
U0 = hyp_ic(length(X)-1);
if plt
    figure(plt)
    subplot(3,1,1:2)
    plot(X,U0,'-'), hold on
    P=plot(x0, u0','.');axis([0 L -0.3 1.3]);
    T = title('t=0');
end


%%%%%%%%%%%%%%%% Explicit Methods %%%%%%%%%%%%%%%%%%
if threelevel == 1 % Methods such as leapfrog
  Ul = [u0(1:m-1)];	Ur = [u0(3:m); u0(1)];
  cl = c*(1+c)/2; 
  cc = 1-c^2;	
  cr = c*(c-1)/2;
  u1 = cl*Ul + cc*u0(2:m) + cr*Ur;       % Lax-Wendroff for level 1
  u1 = [u1(m-1);u1];
  for j = 2:n,	
   t = j*k;
   Ul = [u1(1:m-1)];	Ur = [u1(3:m); u1(1)];
   U  = u0(2:m) - c*(Ur-Ul);
   u2 = [U(m-1); U];
   u0 = u1;
   u1 = u2;   
   if plt
    set(P,'ydata',u1');
       z = sprintf('%6.4f',j*k);
       set(T,'string',[name ': $h = 1/',int2str(M/L),', t = ',z,'$']);
       drawnow
   end
 end

elseif explicit==1			% explicit 2-level schemes
 for j = 1:n,	
   t = j*k;
   Ul = [u0(1:m-1)];	Ur = [u0(3:m); u0(1)];
   Ull = [u0(m-1); u0(1:m-2)];
   U = cll*Ull + cl*Ul + cc*u0(2:m) + cr*Ur;
   u0 = [U(m-1); U];
   if plt
    set(P,'ydata',u0');
       z = sprintf('%6.4f',j*k);
       set(T,'string',[name ': $h = 1/',int2str(M/L),', t = ',z,'$']);
       drawnow
   end
 end
else
%%%%%%%%%%%%%%%% Implicit Methods %%%%%%%%%%%%%%%%%%
e =ones(N,1);
if method(1:2) == 'bt'
  A = spdiags([-c*e (1+c)*e],[-1,0],N,N);
  A(1,N) = -c;
  cl =  c; 
  cc = -c;
  cr =  0;
elseif method(1:2) == 'ta'
  cr = (1-c^2)/6;
  A = spdiags([cr*e (1-2*cr)*e cr*e],[-1:1],N,N);
  cl = c*(1+c)/2;			% coeffs for RHS - same as L-W
  cc = -c^2;	
  cr = c*(c-1)/2;
elseif method(1:2) == 'bo'
  A = spdiags([(1-c)*e (1+c)*e],[-1:0],N,N); A(1,N) = (1-c);
  cl = 2*c;				% coeffs for RHS
  cc =-2*c;	
  cr = 0;
else
  error('Method not known')
  return
end
[low,up,Perm] = lu(A);


 for j = 1:n,	
   t = j*k;
   Ul = [u0(1:m-1)];	Ur = [u0(3:m); u0(1)];
   U = cl*Ul + cc*u0(2:m) + cr*Ur;
   U = u0(2:m) + (up\(low\(Perm'*U)));
   u0 = [U(m-1); U];
   if plt
       set(P,'ydata',u0');
       z = sprintf('%6.4f',j*k);
       set(T,'string',[name ': $h = 1/',int2str(M/L),', t = ',z,'$']);
       drawnow
   end
 end
end
if plt
    grid,xlabel('$x$'), ylabel('$u$')
    tx = text(.6*L,1.1,name);
     set(tx,'HorizontalAlignment','right')
end
 x = [0 x];