%  File: bvpex
%  Example 6.2
%  Solve the ode  -u" + R(x).u(x) = rhs(x),  0 < x < 1
%  with Dirichlet boundary conditions: u(0) = 1, u(1) = -sin^2(b)
%  using a 2nd order finite difference method.
%  The exact solution is cos(b.x)^2-tan(bx)/tan(b),
%  where b=1.5 is a  parameter
% Essential PDEs, Griffiths, Dold & Silvester
% Fig. 6.3 & 6.4, Table 6.2
close all
    b=1.5;    
    % definitions of functions specifying the ode
            rhs = @(x,b) 4*b^2*cos(b*x).^2;
            exact =@(x,b) cos(b*x).^2-tan(b*x)/tan(b); % Exact solution
            R = @(x,b) 2*b^2*sec(b*x).^2;
% fine grid & exact solution for plotting
xfine = 0:.01:1;
ufine = exact(xfine,b);
file = 'bvpex_';  % Stem of filename for saved files
%%%%%%%%%%%%%%%%
J = 8;  % use J+1 grid points
error = zeros(J+1,1); % array for saving the max error on each grid
H = error;            % array for corresponding grid sizes
for j = 0:J,
   M = 2^(j+3);      % number of grid points on jth grid
   h = 1/M;
   x = (h:h:(1-h))'; % internal grid points
   n = M-1;          % number of internal grid points
   X = [0;x;1];
   uL = exact(0,b);			   % u(0)
   uR = exact(1,b);		       % u(1)
   f = rhs(x,b);
   f(1) = f(1) + M^2*uL;
   f(n) = f(n) + M^2*uR;
%	construct the coefficient matrix
   D =spdiags(R(x,b),0,n,n);
   e = ones(n,1);
   A = M^2*spdiags([-e 2*e -e],-1:1,n,n); % tridiagonal approximation of second derivative
   A = A+D;
  
   %%%%%%%%%%%%%%%%%%%% Solve the finite difference equations
	u = A\f;
	
	error(j+1) = norm(exact(x,b)-u,inf);
	H(j+1) = h;
   if j<3 % Plot numerical & exact solutions for coarsest grids
     figure(1);
     subplot(2,3,j+1)
     plot(xfine,ufine,'-',X,[uL;u;uR],'.','markersi',9)
     xlabel('$x$','fontsi',14)
     set(gca,'fontsi',14)
     axis([0 1 min(ufine) max(ufine)]);
     % readjust position of x-label
     zh=get(gca,'xlabel');
     zv = get(zh,'Position');
     set(zh,'Position',[.75 .9*zv(2)])
   end
end
     eval(['print -f1 -deps ' file '1']) 
     disp(['File: ' file '1 printed']) 
ratio = [0; error(1:end-1)./error(2:end) ];  % ratio of errors
format short e
% display table max error & error ratio vs H
[H error ratio]
format short
% display results in Latex tabular format - Table 6.2
fprintf('%4i & %8.7f & %10.7f &  %5.2f &  %5.2f \\\\\n',[1./H H error ratio log(ratio)/log(2)]')
figure(2)
% log-log plot of error vs H
loglog(H,error,'.-','markersi',12)
     xlabel('$h$')
     zh=get(gca,'xlabel');
     zv = get(zh,'Position');
     set(zh,'Position',[3e-2  1.5*zv(2)])
     ylabel('Max Error')
     hold on
     loglog([5e-4 5e-2],[2e-5 2e-1],'--')
     text(2e-3,6e-4,'slope = 2','rotation',45,'fontsi',16)
     eval(['print -f2 -deps ' file '2']) 
     disp(['File: ' file '2 printed'])  
     hold off

