%  File: bvpex_numerov
%  Example 6.21
%  Solve the ode  -u" + R(x).u(x) = rhs(x),  0 < x < 1
%  with Dirichlet boundary conditions: u(0) = 1, u(1) = -sin^2(bx)
%  using 
% 1) a 2nd order finite difference method
% 2) Numerov's method (Example 6.21)
% 3) a 5-point method (see Exercise 6.32 & Project 13.2)
%  The exact solution is cos(b.x)^2-tan(bx)/tan(b),
%  where b=1.5 is a  parameters
% Essential PDEs, Griffiths, Dold & Silvester
% Table 6.3


close all
    b=1.5;    
    % definitions of functions specifying the ode
            rhs   = @(x,b) 4*b^2*cos(b*x).^2;
            exact = @(x,b) cos(b*x).^2-tan(b*x)/tan(b);
            R     = @(x,b) 2*b^2*sec(b*x).^2;
J = 8;  % use J+1 grids
error = zeros(J+1,3); % array for saving the max error on each grid
H = zeros(J+1,1);     % array for corresponding grid sizes
% fine grid & exact solution for plotting
xfine = 0:.01:1;
ufine = exact(xfine,b);
stem = 'bvpex_numerov-';  % Stem of filename for saved files
for j = 0:J,
   M = 2^(j+3);      % number of grid points
   h = 1/M;
   x = (h:h:(1-h))';
   X = [0;x;1];
   n = M-1;
   e = ones(n,1);
%	construct the coefficient matrices
   r = R(X,b);
   D = spdiags(r,0,M+1,M+1);
   A = M^2*spdiags(e*[-1 2 -1],-1:1,n,n);
%   1) standard 2nd order method
   D1 = spdiags(r(2:M),0,n,n);
   A1 = A+D1;
%   2) Numerov's method
   B  = spdiags(e*[1 10 1],0:2,n,n+2)/12;
   BD = B*D; BD = BD(:,2:M); % discard 1st & last columns
   A2 = A + BD;
%   3) 5-point method of Exercise 6.32 & Project 13.2
   A3 = D1 + M^2*spdiags(e*[1 -16 30 -16 1],-2:2,n,n)/12;
   % use 3-point formula at first & last internal grid points
   A3(1,:)=A1(1,:);
   A3(n,:)=A1(n,:);
%   construct right hand sides
   F  = rhs(X,b);
   f1 = F(2:M);  % source term on internal grid points
   f2 = B*F;
   f3 = f1;
% now adjust for BCs
   uL = exact(0,b);			   % u(0)
   uR = exact(1,b);		       % u(1)
   f1([1;n]) =f1([1;n])+ M^2*[uL; uR];
   f2([1;n]) =f2([1;n])+ [(M^2-r(1)/12)*uL;(M^2-r(M+1)/12)*uR];
   f3([1;2;n-1;n]) = f3([1;2;n-1;n])+ M^2*[uL; -uL/12; -uR/12; uR];

   %%%%%%%%%%%%%%%%%%%% Solve the finite difference equations
    % store solutions in an (M+1)x3 array U
    U = zeros(M+1,3);
    U(1,:) = uL;
    U(M+1,:) = uR;
    u = A1\f1;
        U(2:M,1) = u;
	u = A2\f2;
        U(2:M,2) = u;
	u = A3\f3;
        U(2:M,3) = u;
%   make the exact solution array the same size as U
	E =   repmat(exact(X,b),1,3); 
	error(j+1,:) = max(abs(E-U));
	H(j+1) = h;
   if j==2  % plot solution & errors on a coarse grid
     for i = 1:3
     figure(1);
         subplot(3,3,[i i+3])
         plot(xfine,ufine,'-',X,U(:,i),'.','markersi',9)
         xlabel('$x$','fontsi',14)
         set(gca,'fontsi',14)
         axis([0 1 min(ufine) max(ufine)]);
         xh=get(gca,'xlabel');
         xv = get(xh,'Position');
         set(xh,'Position',[.75 .9*xv(2)])
     figure(2);
         subplot(3,3,[i i+3])
         plot(X,E(:,i)-U(:,i),'.-')
         xlabel('$x$','fontsi',14)
     end
   end
end
     eval(['print -deps ' stem '1']) 
     disp(['File: print -f1 -deps ' stem '1 printed']) 
format short 
ratio = [zeros(1,3); error(1:end-1,:)./error(2:end,:) ];  % ratio of errors
disp('                  Maximum Errors')
disp('   M   Method 1     Method 2     Method 3        Ratios')
fprintf('%4i  %12.9f %12.9f %12.9f    %5.2f   %5.2f  %5.2f\n',...
    [1./H  error ratio]')
%latex([1./H H error ratio])
figure(3)
loglog(H,error,'.-','markersi',12)
     xlabel('$h$')%,'fontsi',14)
     xh=get(gca,'xlabel');
     xv = get(xh,'Position');
     set(xh,'Position',[3e-2  1.5*xv(2)])
     ylabel('Max Error')%,'fontsi',14)
     hold on
     loglog([5e-4 5e-2],[2e-5 2e-1],'--')
     text(2e-3,6e-4,'slope = 2','rotation',30,'fontsi',16)
     eval(['print -f3 -deps ' stem '3']) 
     disp(['File: print -deps ' stem '3 printed']) 
     hold off
