%  File: bvpex_robin
%  Example 6.16
%  Solve the ode  - u" + R(x).u(x) = rhs(x) on 0 < x < 1
%  with   boundary conditions: u(0) = 1, a.u(1)+u'(1) = s.
%  using 1st & 2nd order approximations of the BC at x=1.
%  The exact solution is exact(x,s), where s (sigma) is a parameter.
% Essential PDEs, Griffiths, Dold & Silvester
% Fig. 6.7 & 6.8.

close all
    fj = 1;
    sigma=4;
    % definitions of functions specifying the ode
            rhs   = @(x,s) zeros(size(x));
            exact = @(x,s) sinh(x*s)/sinh(s);
            R     = @(x,s) s^2*ones(size(x));
            a     = sigma;
            beta  = sigma*exp(sigma)/sinh(sigma);
% fine grid & exact solution for plotting
xfine = 0:.01:1;
ufine = exact(xfine,sigma);    
%%%%%%%%%%%%%%%%
J = 6;  % use J+1 grids

error = zeros(J+1,2); % array for saving the max error on each grid
H = zeros(J+1,1);            % array for corresponding grid sizes
for j = 0:8
   M = 2^(j+3);      % number of grid points on jth grid
   h = 1/M;
   x = (h:h:1)';
   f = rhs(x,sigma);
   uL = exact(0,sigma);			   % u(0)
   uR = exact(1,sigma);		       % u(1)
   fR = rhs(1,sigma);		       % f(1)
%	construct the coefficient matrices
   X = [0;x];
   r = R(x,sigma);
   D =spdiags(r,0,M,M);   
   e = ones(M,1);
   A = M^2*spdiags(e*[-1 2 -1],-1:1,M,M);
   A = A+D;                         % 1st order method
   A(M,M)=(1+h*a)*M^2;
   A2 = A;
   A2(M,M) = A2(M,M)+.5*R(1,sigma); % 2nd order method
   
%	construct the rhs 
   	
   %%%%%%%%%%%%%%%%%%%%
   f(1) = f(1) + uL*M^2;			% 1st order method
   f(M) =  beta/h;
   F = f;
   F(M) = F(M)+.5*rhs(1,sigma);     % 2nd order method
   %%%%%%%%%%%%%%%%%%%%
   
   %%%%%%%%%%%%%%%%%%%%
   U = zeros(M+1,2);                % array for storing both solutions
   U(1,:) = uL;
	U(2:M+1,1) = A\f;
 	U(2:M+1,2) = A2\F;
	E =   repmat(exact(X,sigma),1,2);
	error(j+1,:) = max(abs(E-U));
	H(j+1) = h;
   if j<3 % plot solutions on 3 coarsest grids
     figure(1); %+(N/8-1)*10)
     subplot(2,3,j+1)
     col = ['k.';'b.';'g.';'c.';'r.';'k.';'b.';'g.';'c.';'r.']; col = [col;col];col = [col;col];

     plot(xfine,ufine,'-'), hold on
     plot(X,U(:,1),'b.','markersi',14);
     plot(X,U(:,2),'kx','markersi',10);
     xlabel('$x$','fontsi',14)
     set(gca,'fontsi',14)
     %axis([0 1 min(exe) max(exe)]);%1.2*round(max(exe))])
     zh=get(gca,'xlabel');
     zv = get(zh,'Position');
     set(zh,'Position',[.75 .9*zv(2)])
      if j==0
         ax = axis; % ensure all 3 plots have same axes
     else
         axis(ax);
     end
     text(.2,1,['$M = $' int2str(M)],'fontsi',18)
   end
end
file = 'bvpex_robin-';
     eval(['print -f1 -deps ' file '1']) 
     disp(['File: ' file  '1 printed']) 
format short e
ratio = [0 0; error(1:end-1,:)./error(2:end,:) ];
[H error ratio]
format short
disp('               Maximum Errors')
disp('   M   Method 1     Method 2        Ratios')
fprintf('%4i  %12.9f %12.9f    %5.2f  %5.2f\n',...
    [1./H  error ratio]')
figure(2)
loglog(H,error,'.-','markersi',24),hold on
set(gca,'fontsi',20)
     xlabel('$h$','fontsi',30)
     zh=get(gca,'xlabel');
     zv = get(zh,'Position');
     set(zh,'Position',[3e-2  1.5*zv(2)])
     set(gca,'ytick',10.^(-6:2:0))
     ylabel('Max Global Error','fontsi',28)
     hold on
     S = exp(.5*(log(error(1,1))+log(error(1,2))));
     loglog([1/8 1/256],S*[1 8/256].^2,'--')
     loglog([1/8 1/256],S*[1 8/256],'--')
     %text(7e-3,2e-4*S/.04,'slope = 2','rotation',40,'fontsi',22)
     %text(9e-3,4.3e-3*S/.04,'slope = 1','rotation',45/2,'fontsi',22)
     eval(['print -deps ' file '2']) 
     disp(['File:' file '2'  ' printed']) 
     hold off
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
