function fig4_16
%% Generates fig 4.16 in the book Chaos.
% Shoot for a homoclinic orbit for the Rossler model, when b=0.2, a=0.181
% and c=10.3134491342463.
% This value of c was found by running fig4_16_Find_c.m and searching for close
% approach to a principal homoclinic orbit. Approximate values for a, b and c
% are based on the publication by Barrio, Blesa, Dena and Serrano (2011) where the 
% homoclinic orbit looks interesting and is of Sil'nikov type.
% Mark Mcguinness

options1=odeset('RelTol',2.3e-14,'AbsTol',2.3e-14,'Events', @Revents);  % event, no stopping though
%options2=odeset('RelTol',2.3e-14,'AbsTol',2.3e-14);  % no events

mywidth=1; defFont=24;

b=0.2; a=0.181; c=  1.03134491342463e+01;

% p1=0.5*(-c/a-sqrt(c*c-4*a*b)/a);  % the stable fixed point P1:
% P1=[-a*p1, p1, -p1];  %is further from origin than P2

p2 = 0.5*(-c/a+sqrt(c*c-4*a*b)/a); % the saddle/focus point P2:
P2=[-a*p2, p2, -p2];  % this is [0.0035   -0.0194    0.0194], quite close to origin


% Start on the tangent plane to the unstable manifold near P2
% on which solutions spiral away from P2. This plane is spanned by the first two
% column vectors of T, where the first col vector is real(Vee(:,1)), and
% the second col vector of T is imag(Vee(:,2)), as explained in the book by
% Hirsch, Smale, & Devaney 2004 Elsevier Differential_equations__dynamical_
% systems__and_an_introduction_to_chaos__2ed chapters 5, 6:

linsys=[0, -1, -1; 1, a, 0; -p2, 0, -a*p2-c];  % the DEs linearised about P2 have 
                                             % this matrix
                                             
[Vee, ~] = eig(linsys); 
% display(linsys)
% display(Vee)  % the eigenvectors are the columns of Vee. The last column turns
%               % out to be the stable eigendirection (close to P2, anyway)
% display(Dee)  % the eigenvalues are on the diagonal

spanners = [real(Vee(:,1)), imag(Vee(:,1))];  % these two col vectors span
                                        
% the tangent to the unstable  manifold at P2.
% I am chasing a spiral among spirals. I start a small
% distance away from P2, check the next time the traj crosses that basis direction,
% and then I search on the line segment between these two points. I can take
% advantage of the fact that the transformation matrix T maps from x,y,z
% space to one in which the axes are aligned with the tangents to the
% stable and unstable manifolds of P2. I need a criterion for crossing the
% vector. I use the diagonalised coordinate system
% y = T^{-1} x where x is our solution. This y is in diagonalised space,
% e.g. for real distinct eigenvalues the eigenvectors line up with
% the usual basis vectors for y. In y space, we can use the (new) x axis as our
% starting place, aligned with the first column of T which is
% real(Vee(:,1)). So y is zero when crossing this x axis, and
% I observe that y is 


UnstabVec=spanners(:,1); % use the first column of T as our search direction 
                         % and to define the start point (on new x-axis in
                         % diagonalised space)
MyV1 = UnstabVec; MyV2 = spanners(:,2);                         
                         
T = [spanners, real(Vee(:,3))]; % this is the transformation matrix that diagonalises
                                % our linearised system near P2. I've set
                                % it that x, y in the transformed space is
                                % the tangent plane to where the spirals live, and z
                                % is tangent to the stable 1D manifold. Note that origin
                                % in transformed space is P2 in real space.

% lets start on UnstabVec near row vector P2:
epsilon=1E-02;
ICs = P2 + epsilon*UnstabVec';  % a row vector

transfICs=T\(ICs-P2)';  % solve T.transfICs = ICs the matlab way. This gives [epsilon,0,0] OK

%disp(ICs)
%disp(transfICs)
%angle=atan2(transfICs(2), transfICs(1));
%display(angle);

% just run for a bit before checking for zero angle, as it seems to pick up
% the very first zero angle at the IC

[~,Ysol]=ode45(@RosslerDE,[0, 1],ICs,options1);
xvals1=Ysol(:,1); yvals1=Ysol(:,2); zvals1=Ysol(:,3);
ICs=[xvals1(end); yvals1(end); zvals1(end)];
    
fno=1; 
figure(fno)
clf('reset')

plot3(-a*p2, p2, -p2, '.r','MarkerSize', 30); %the location of P2

hold on;

Vee3 = real(Vee(:,3));

% plot3([-a*p2 -a*p2+ACFv1(1)], [p2 p2+ACFv1(2)], [-p2 -p2+ACFv1(3)], '-k', 'linewidth', 2); % draw a line from P2 to ACF's v1
% plot3([-a*p2 -a*p2+ACFv2(1)], [p2 p2+ACFv2(2)], [-p2 -p2+ACFv2(3)], '-k', 'linewidth', 2); % draw a line from P2 to ACF's v1
plot3([-a*p2 -a*p2+Vee3(1)], [p2 p2+Vee3(2)], [-p2 -p2+Vee3(3)], '-k', 'linewidth', 3); % draw a line from P2 to ACF's v1

plot3([-a*p2 -a*p2+MyV1(1)], [p2 p2+MyV1(2)], [-p2 -p2+MyV1(3)], '-b', 'linewidth', 2); % draw a line from P2 to ACF's v1
plot3([-a*p2 -a*p2+MyV2(1)], [p2 p2+MyV2(2)], [-p2 -p2+MyV2(3)], '-b', 'linewidth', 2); % draw a line from P2 to ACF's v1

tend = 1000;

    [~,Ysol,~,Ye,~]=ode45(@RosslerDE,[0, tend],ICs,options1); % run longer
    
xvals2=Ysol(:,1); yvals2=Ysol(:,2); zvals2=Ysol(:,3);
XvE = Ye(:,1); YvE = Ye(:,2); ZvE=Ye(:,3); % collect the coords at the events
                                           % which are in fact a
                                           % poincare section through
                                           % the attractor
plot3(xvals1,yvals1,zvals1,'-r','linewidth',mywidth)
plot3(xvals2,yvals2,zvals2,'-r','linewidth',mywidth)
plot3(ICs(1),ICs(2),ICs(3),'+k','MarkerSize',10)  %mark the join point
plot3(xvals2(end), yvals2(end), zvals2(end), 'ob','MarkerSize',10) % stop point too

plot3(XvE, YvE, ZvE,'sb','MarkerSize',10);   %mark the event points found

ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=0.5;  ax.Box='off';
xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont,'Rotation',0.0);
zlabel('z','FontSize',1.3*defFont,'Rotation',0.0);
title('Explore Rossler, starting on tangent plane','FontSize',1.3*defFont)
hold off;
    

    
%% now look at a plot in the transformed normal space:
 
fno=fno+1; 
figure(fno)

clf('reset')

% the location of P2 is the new origin
plot3(0, 0, 0, '.r','MarkerSize', 30); %the location of P2


hold on; 
    xvals = [xvals1', xvals2']; % a row of x values, concatenated
    yvals = [yvals1', yvals2']; % a row of y values
    zvals = [zvals1', zvals2']; % a row of v values
    xvarr = [xvals; yvals; zvals]; % a matrix with x,y,z vertically at each time
    
    P2arr = repmat(P2', 1, length(xvals));
    newxv = T\(xvarr-P2arr); % this gives all of the transformed x,y,z values
                             % one column for each time
    P2arrE = repmat(P2', 1, length(XvE));
    
    
    XvEarr=[XvE';YvE';ZvE']; % these events are column vectors


    newXEv = T\(XvEarr-P2arrE);                         
    
    plot3(newxv(1,:), newxv(2,:),newxv(3,:),'-r','linewidth',1)
    plot3(newxv(1,1), newxv(2,1),newxv(3,1),'+k','MarkerSize',10)  %mark the start point  
    plot3(newXEv(1,:), newXEv(2,:), newXEv(3,:),'sb','MarkerSize',10);   %mark the event points found
    
    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.LineWidth=0.5;  ax.Box='off';
    xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont,'Rotation',0.0);
    zlabel('z','FontSize',1.3*defFont,'Rotation',0.0);
    title('Explore Rossler, transformed: tangent plane','FontSize',1.3*defFont)
    hold off;
     
% get a return map through our poincare section for a long time run:   
tend = 800;

    [~,~,~,Ye,~]=ode45(@RosslerDE,[0, tend],ICs,options1); % run longer
   
XvE = Ye(:,1); % collect the coords at the events
               % which are in fact a poincare section through the attractor
fno=fno+1; 
figure(fno)
% now plot x values of events against previous x values
clf('reset')
    
    plot(-XvE(1:end-1), -XvE(2:end),'.r','MarkerSize',20)
    hold on;
    plot([min(-XvE),max(-XvE)],[min(-XvE),max(-XvE)],'-k')
    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.LineWidth=0.5;  ax.Box='off';
    xlabel('x_n','FontSize',1.3*defFont); ylabel('x_{n+1}','FontSize',1.3*defFont,'Rotation',0.0);
    title('Return map, Rossler','FontSize',1.3*defFont)
    hold off;
    
% 
% Search for the lowest "first decrease in x" seen in the
% poincare section. First, look at this, for initial conditions varying
% from start point to next x on the section:
%

epsilon=1E-04;
ICs = P2 + epsilon*UnstabVec';  % a row vector

% start close to P2. P2 is a row vector,
% DesiredDirection is a column vector
                                       
% pick up on the next time this orbit crosses the Poincare section:

% first, run for a bit before checking for zero angle, as otherwise it seems to pick up
% the very first zero angle at the IC:

[~,Ysol]=ode45(@RosslerDE,[0, 1],ICs,options1);
xvals1=Ysol(:,1); yvals1=Ysol(:,2); zvals1=Ysol(:,3); % the first bit
ICs2=[xvals1(end); yvals1(end); zvals1(end)];    
   [~,~,~,Ye,~]=ode45(@RosslerDE,[0, 50],ICs2,options1); % run from previous end of 
                                                % orbit, to next few
                                                % crossings
                                                % of the poincare section
start1=Ye(1,:);  % pick up the first and second events as start points
start2=Ye(2, :);  % this works because the homoclinic orbit must also be on 
                  % the same unstable manifold as the present orbit, and
                  % between the two points, start and once around, are all
                  % of the orbits on the attractor. Including many
                  % homoclinic ones.

display(start1); display(start2);

% run with ICs on a line joining start1 and start2, parameter t:
nICs=100; % number of initial conditions to use on the line

tlist=linspace(0,1,nICs);
tend=200;
FindVal=zeros(3, nICs); % preallocate a place to save the location of the smallest x values 
                        % in the poincare section

fno=fno+1; 
figure(fno)

clf('reset')

plot3(-a*p2, p2, -p2, '.r','MarkerSize', 30); %the location of P2
hold on;
for jj=1:nICs
    t = tlist(jj); 
    ICs = (1-t)*start1 + t*start2;  % this will run along the line from start1 to start2
    %display(ICs)
    
    [timeout,Ysol,Te,Ye,~]=ode45(@RosslerDE,[0, tend],ICs,options1);

    % find the index of the first lesser value of x in Ye(:,1)
    Xev = Ye(:,1); LogicXev = Xev(2:end)>Xev(1:end-1); % LogicXev has 1 where condition is met
    XevInd = find(LogicXev)+1; % finds indices where Xev values get smaller than previous
    %disp(size(XevInd));
    %disp(XevInd(1)); disp(size(Xev));
    FindVal(1,jj) = Xev(XevInd(1)); % I would save only the first one as target,
                            % BUT I want to transform to the space with
                            % axes being the eigenvectors at P2, so I need
                            % all three components to do this now:
    Yev = Ye(:,2); FindVal(2,jj) = Yev(XevInd(1));
    Zev = Ye(:,3); FindVal(3,jj) = Zev(XevInd(1));
    
    LogicTraj = timeout > Te(XevInd(1)+1); % set 1's where times are > the 
                                        % time of the smaller x, ie after
                                        % the orbit has plunged back onto
                                        % the spiral manifold, because I do
                                        % not want to display after this
                                        % has happened. Go one more event
                                        % though, as some events close to homoclinic happen at
                                        % larger z values
    TimeIndex = find(LogicTraj); % find the indices of  nonzero values 
    
    xvals=Ysol(1:TimeIndex,1); yvals=Ysol(1:TimeIndex,2); zvals=Ysol(1:TimeIndex,3);
    plot3(xvals,yvals,zvals,'-r','linewidth',mywidth)
    plot3(xvals(1), yvals(1),zvals(1),'or','MarkerSize', 10)  % mark the start values
    plot3(xvals(end), yvals(end),zvals(end),'.b','MarkerSize', 10)  % mark the end values

end

    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.LineWidth=0.5;  ax.Box='off';
    xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont,'Rotation',0.0);
    zlabel('z','FontSize',1.3*defFont,'Rotation',0.0);
    title('Rossler testing','FontSize',1.3*defFont)
    hold off;

% plot starting t values against found first-minimum x values

fno=fno+1; 
figure(fno)

clf('reset')

P2arr = repmat(P2', 1, length(FindVal(1,:)));
TFoundvals = T\(FindVal-P2arr); % this gives all of the transformed x,y,z values

    
plot(tlist, TFoundvals(1,:),'-r','LineWidth', mywidth)
hold on;

plot([tlist(1), tlist(end)], [0, 0], '-k', 'LineWidth', 1) % the x comp of P2 is zero in transformed space

    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.LineWidth=0.5;  ax.Box='off';
    xlabel('t','FontSize',1.3*defFont); ylabel('max x found','FontSize',1.3*defFont);
    title('Rossler testing','FontSize',1.3*defFont)
hold off;
   
% have matlab find the best t value in a given t range, that gives the
% orbit that comes closest to P2

optionsfmin = optimset('TolFun',1.0E-08, 'TolX',1.0E-08, 'Display', 'iter');
tstart1=0.34; tend1=0.37;
tfound = fminbnd(@GetMinX,tstart1, tend1, optionsfmin);
% fprintf('%17.10f',tfound)

% now use this t value to compute the homoclinic orbit:
    ICs = (1-tfound)*start1 + tfound*start2;  % this is on the line from start1 to start2

    [timeout,Ysol,Te,Ye,~]=ode45(@RosslerDE,[0, tend],ICs,options1);

    % find the index of the first lesser value of x in Ye(:,1)
    Xev = Ye(:,1); LogicXev = Xev(2:end)>Xev(1:end-1); % LogicXev has 1 where condition is met
    XevInd = find(LogicXev)+1; % finds indices where Xev values get smaller than previous
    
    LogicTraj = timeout > Te(XevInd(1)+1); % set 1's where times are > the 
                                % time of the smaller x, ie after
                                % the orbit has plunged back onto
                                % the spiral manifold, because I do
                                % not want to display after this
                                % has happened. Go one more event
                                % though, as some events close to homoclinic happen at
                                % larger z values
    TimeIndex = find(LogicTraj); % find the indices of  nonzero values 
    
    xvals=Ysol(1:TimeIndex,1); yvals=Ysol(1:TimeIndex,2); zvals=Ysol(1:TimeIndex,3);
    
fno=fno+1; 
figure(fno)

clf('reset')

plot3(-a*p2, p2, -p2, '.r','MarkerSize', 30); %the location of P2
hold on;
    plot3(xvals,yvals,zvals,'-r','linewidth',2)
    plot3(xvals(1), yvals(1),zvals(1),'or','MarkerSize', 10)  % mark the start value
    plot3(xvals(end), yvals(end),zvals(end),'.b','MarkerSize', 10)  % mark the end value
    
    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.LineWidth=0.5;  ax.Box='off';
    xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont,'Rotation',0.0);
    zlabel('z','FontSize',1.3*defFont,'Rotation',0.0);
    title('homoclinic orbit','FontSize',1.3*defFont)
hold off;

% Reversing time, and starting on the now unstable 1D
% manifold of P2 near P2 by using the eigenvector of the linearised system
% near P2, reveals that the instability of the attractor means we cannot
% follow the correct orbit for long enough to look like a homoclinic orbit:


function F=RosslerDE(~,yy)
% ROSSLERDE sets up the Rossler equations, to be used by an ode solver:
    x=yy(1);
    y=yy(2);
    z=yy(3);
    
    f1  = -(y+z);
    f2 = x +a*y;
    f3 = b+ z*(x-c); 
    F=[f1; f2; f3];

end

    function [value, isterminal, direction] = Revents(~,u)
        xvt=[u(1);u(2);u(3)];
        % find solution in nice transformed space:
        nxv=T\(xvt-P2'); % T, P2 are passed in from parent routine
        value=nxv(2); % don't need the fancy angle, just y increasing through zero
        isterminal=0;   % no, don't stop here
        direction=1;   % ask for the y value to be increasing through zero
    end


    function xout=GetMinX(t)
        %%GETMINX finds the norm value of closest approach to P2, given t
        % The x value found is in transformed space for which the axes are
        % the eigendirections of the linearised Rossler equations about P2,
        % the spiral saddle. t parameterises the line segment from start1
        % to start2, which segment lies in the unstable manifold of P2. The
        % x value is used to navigate, where am I in terms of has the orbit
        % gone out and up off the unstable manifold to return near the
        % stable manifold or not? This allows me to discard the earlier
        % part of each orbit, and then examine the later part where it
        % plunges down towards P2 after leaving it. Then find the minimum
        % approach to P2 for these points in time, on the orbit. The
        % minimum x did not give such a close approach, because I think the
        % homoclinic orbit cuts the poincare section at high z values. One
        % fix for this is to check the z value for the min x, and if the z
        % value is high, go one more x in the poincare section.
  
    ICsloc = (1-t)*start1 + t*start2;  % this is on the line from start1 to start2

    
    [timeoutloc,Ysoloc,Teloc,Yeloc,~]=ode45(@RosslerDE,[0, 150],ICsloc,options1);

    % find the index of the first lesser value of x in Ye(:,1)
    Xevloc = Yeloc(:,1); LogicXevloc = Xevloc(2:end)>Xevloc(1:end-1); % LogicXev has 1 where condition is met
    XevIndloc = find(LogicXevloc)+1; % finds indices where Xev values get smaller than previous
    %FindValxout(1) = Xevloc(XevIndloc(1)); % I would save only the first one as target,
                            % BUT I want to transform to the space with
                            % axes being the eigenvectors at P2, so I need
                            % all three components to do this now:
    %Yevloc = Yeloc(:,2); FindValxout(2) = Yevloc(XevIndloc(1));
    Zevloc = Yeloc(:,3); zfound = Zevloc(XevIndloc(1));
    if zfound > 2*P2(3) 
        disp('in GetMinX, got a high z at min xmax')
        display(t)
        display(zfound)
        XevIndUSE =  XevIndloc(1) +1;
    else
        XevIndUSE =  XevIndloc(1);
    end
 
    
    LogicTrajloc2 = timeoutloc > Teloc(XevIndUSE); % set 1's where times are > the 
                                        % time of the smaller x, ie after
                                        % the orbit has plunged back onto
                                        % the spiral manifold, because I do
                                        % not want to check distances after this
                                        % has happened. 
    TIndices2 = find(LogicTrajloc2); % find the indices of  nonzero values
    TIndex2 = TIndices2(1); % the first index is where the first one is
    
    LogicTrajloc1 = timeoutloc > Teloc(XevIndloc(1)-1); % set 1's where times are > the 
                                        % time of the smaller x, ie after
                                        % the orbit has plunged back onto
                                        % the spiral manifold, because I do
                                        % not want to check too long after this
                                        % has happened. Go one less event
                                        % though, as I want to look at
                                        % where the orbit is going from
                                        % nearly exiting the unstable
                                        % manifold, to after it has plunged
                                        % back towards P2
    TIndices1 = find(LogicTrajloc1); % find the indices of  nonzero values 
    TIndex1 = TIndices1(1);          % get the first one
    display(TIndex1); display(TIndex2)
    Ypiece=Ysoloc(TIndex1:TIndex2,:)';  % the points on the orbit from near the 
                                    % outermost circuit of unstable
                                    % manifold, to the first plunge close to P2
    P2arrloc = repmat(P2',1,length(Ypiece(1,:)));
    Dists = vecnorm(Ypiece - P2arrloc); % Euclidean distances between points on orbit, and P2
                               % one column for each time
    xout = min(Dists);  % the shortest distance
    
    end

end