function fig4_16_Find_c
%% Shoot on c for existence of a homoclinic orbit, prep for Fig 4.16.
% Shoot for a homoclinic orbit for the Rossler model, when b=0.2,
% and a=0.181. Seeks a c value such that there is a homoclinic
% orbit.
%
% Start on the stable manifold close to P2; run time backwards; the
% decision about bisecting on c is based on whether the orbit veers towards
% positive or negative z values, when it veers away from the unstable
% spiral manifold of P2.
%
% The above process is repeated for initial conditions that are closer and
% closer to the fixed point, until the value obtained for c is accurate to
% 15 significant figures.
% 
% The values chosen for b and a are to the right of but close to 
% the point B, in Barrio, Blesa, Dena and Serrano (2011) where 
% the two branches of homoclinic orbits meet. 
% Shoot for the lower branch, where the homoclinic orbit spirals away from
% P2 (going forwards in time), then rises in z value, then plunges back towards 
% P2 without any further spiralling. 
%
% Note that this method does not provide a way to calculate and plot the
% actual homoclinic orbit, since the rate at which the orbit leaves the
% unstable manifold it spirals inwards on is exp(10t), which quickly takes
% the orbit away from that manifold.
%
% Mark Mcguinness

options=odeset('RelTol',2.3e-14,'AbsTol',2.3e-14,'Events', @Revents2);  % stop when |z| big

% mywidth=3; defFont=24;

b=0.2;  a = 0.181; % c is around 10; 
                   % HowClose is how close the orbit is to the fixed point,
                   % initially, before iterating backwards in time to try
                   % to hit the unstable manifold
 % I did this by hand, repeating for values of HowClose that got smaller
 % and smaller, until the value of c converged to 15 significant figures:
 %
 % The c values I found by repeatedly running this m-file were:
                   % 1.02331040119743e+01 for HowClose=10.0
                   % 1.03126454288201e+01 for HowClose=1.0
                   % 1.03134413344485e+01 for HowClose=1E-01
                   % 1.03134490623088e+01 for HowClose=1E-02
                   % 1.03134491336202e+01              1e-03
                   % 1.03134491342413e+01 for HowClose=1e-04
                   % 1.03134491342463e+01              1e-05
                   % 1.03134491342463e+01              1e-06
                   % search on "c" values between 10.2 and 10.4
% fno=0;
HowClose=10.0E-00;  % this determines the initial conditions before iterating 
                    % backwards in time to shoot for a return to the fixed point
                    % Its value affects the c value found; closer to P2 is closer to
                   % the stable manifold, but slower to move away


% the challenge is to find the lower branch.  We
% iterate backwards in time from the stable 1D manifold of P2, and close to 
% P2. 

% Find two values of c that bracket the desired behaviour; fly up or fly down off
% the spiral plane. Fly up at c=10, fly down at c=11. Look between these two values

% use fzero to search for the final z value being zero
% need to put the evaluation of the final z value into a function form

% The tangent plane to the unstable manifold near P2
% on which solutions spiral away from P2: This plane is spanned by the first two
% column vectors of T, where the first col vector is real(Vee(:,1)), and
% the second col vector of T is imag(Vee(:,2)), as explained in the book by
% Hirsch, Smale, & Devaney 2004 Elsevier Differential_equations__dynamical_
% systems__and_an_introduction_to_chaos__2ed chapters 5, 6:

% spanners = [real(Vee(:,1)), imag(Vee(:,1))];  % these two col vectors span
                                            %  the tangent to the unstable
                                            %  manifold at P2
% the transformation matrix T maps from x,y,z
% space to one in which the axes are aligned with the tangents to the
% stable and unstable manifolds of P2. 
% consider the diagonalised coordinate system
% z = T^{-1} x where x is our solution. This z is in diagonalised space,
% e.g. where for real distinct eigenvalues the eigenvectors line up with
% the usual basis vectors for y. In y space, the (new) x axis is aligned 
% with the first column of T which is
   
    tend=-10;    % a negative end time, and zero start time, will run time backwards
                % longer times go to very large |z| values
  optionsfz = optimset('Display','iter','TolX',1.0E-15);
  
  Crange = [10.2,10.4]; 
  [Cfound] = fzero(@GetZend,Crange,optionsfz);  % seek a c value that is close to the homoclinic value
  
  disp('Found the c value:')
  fprintf('%22.14e\n',Cfound)
  
  % plot the resulting orbit, and two with c values just bracketing the found
  % value:

fno=0; mywidth=3;defFont=24;
PlotReversed(Cfound);
PlotReversed(Cfound*1.00001)
PlotReversed(Cfound*0.99999)

% fno=10;
% Cflin = linspace(Cfound*0.999, Cfound*1.001,10);
% for c = Cflin
%   PlotReversed(c)
% end
% 
% nocee=100; closeness=1.0E-12; % look nearby:
% Cflin = linspace(Cfound*(1.0-closeness), Cfound*(1+closeness),nocee);
% ze=zeros(nocee,1);
% ic=0;
% for c = Cflin
%   ic=ic+1;
%   ze(ic)=GetZend(c); 
% end
% 
% fno=50; 
% figure(fno)
% clf('reset')
% plot(Cflin,ze, '.r','MarkerSize', 10); 
% 
% ax=gca;
% ax.FontSize=defFont; ax.FontName='Times';
% ax.LineWidth=0.5;  ax.Box='off';
% xlabel('c','FontSize',1.3*defFont); 
% ylabel('z','FontSize',1.3*defFont,'Rotation',0.0);
% title(['c found was ',num2str(Cfound)]);


    function PlotReversed(c)
        %% to plot solution to Rossler in reverse time, start on stable man of P2
        
        p2 = 0.5*(-c/a+sqrt(c*c-4*a*b)/a); % the saddle/focus point P2:
        P2=[-a*p2, p2, -p2];

        % The tangent plane to the unstable manifold near P2
        % on which solutions spiral away from P2: This plane is spanned by the first two
        % column vectors of T, where the first col vector is real(Vee(:,1)), and
        % the second col vector of T is imag(Vee(:,2)), as explained in the book by
        % Hirsch, Smale, & Devaney 2004 Elsevier Differential_equations__dynamical_
        % systems__and_an_introduction_to_chaos__2ed chapters 5, 6:

        linsys=[0, -1, -1; 1, a, 0; -p2, 0, -a*p2-c];  % the DEs linearised about P2 have 
                                                     % this matrix
        [Vee,~] = eig(linsys);                        

        % Reverse time, and start on the stable 1D
        % manifold of P2 near P2 by using the eigenvector of the linearised system
        % near P2, iterate on a to find a value that gives the simplest homoclinic
        % orbit:
        DesiredDirection = Vee(:,3); % the last column, stable 1D manifold of P2

        ICs = P2' + HowClose*DesiredDirection; % start close to P2. P2 is a row vector,
                                               % DesiredDirection is a column vector

        % a negative end time, and zero start time, will run time backwards
        % longer times just go to very large |z| values, if options does not
        % prevent this

        RDEloc = @(t,y)RosslerDE(t,y,c);            
        [~,Ysol]=ode45(RDEloc,[0, tend],ICs,options);
        xvalsr=Ysol(:,1); yvalsr=Ysol(:,2); zvalsr=Ysol(:,3);
          
        fno=fno+1; 
        figure(fno)
        clf('reset')
        plot3(-a*p2, p2, -p2, '.r','MarkerSize', 30); %the location of P2
        hold on;    
            plot3(xvalsr,yvalsr,zvalsr,'-r','linewidth',mywidth)
            plot3(ICs(1),ICs(2),ICs(3),'+k','MarkerSize',10)  %mark the start point
            plot3(xvalsr(end), yvalsr(end), zvalsr(end), 'ob','MarkerSize', 15)
            
            ax=gca;
            ax.FontSize=defFont; ax.FontName='Times';
            ax.LineWidth=0.5;  ax.Box='off';
            xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont,'Rotation',0.0);
            zlabel('z','FontSize',1.3*defFont,'Rotation',0.0);
            title(['reversed time, c=' num2str(c)],'FontSize',1.3*defFont)
        hold off;
    end
        


function F=RosslerDE(~,yy,c)
 %% the Rossler equations
 
    x=yy(1);
    y=yy(2);
    z=yy(3);
    
    f1  = -(y+z);
    f2 = x +a*y;
    f3 = b+ z*(x-c); 
    F=[f1; f2; f3];

end


    function [value, isterminal, direction] = Revents2(~,u)
        %% stop solving if solutions get too large in norm
        xvt=[u(1);u(2);u(3)];
        maxnorm=100;
        value = norm(xvt)-maxnorm; % look for this to be zero, i.e. to be on new x axis
        isterminal=1;   %  do stop here
        direction=1;   %   ask for increasing norm, through the value maxnorm
        
    end

    function zgot = GetZend(c) 
        %% to find z after orbit flies away from spiral, time going backwards
        
        p2 = 0.5*(-c/a+sqrt(c*c-4*a*b)/a); % the saddle/focus point P2:
        P2=[-a*p2, p2, -p2];

        % The tangent plane to the unstable manifold near P2
        % on which solutions spiral away from P2: This plane is spanned by the first two
        % column vectors of T, where the first col vector is real(Vee(:,1)), and
        % the second col vector of T is imag(Vee(:,2)), as explained in the book by
        % Hirsch, Smale, & Devaney 2004 Elsevier Differential_equations__dynamical_
        % systems__and_an_introduction_to_chaos__2ed chapters 5, 6:

        linsys=[0, -1, -1; 1, a, 0; -p2, 0, -a*p2-c];  % the DEs linearised about P2 have 
                                                     % this matrix

        [Vee,~] = eig(linsys);                        

        % Reverse time, and start on the stable 1D
        % manifold of P2 near P2 by using the eigenvector of the linearised system
        % near P2, iterate on a to find a value that gives the simplest homoclinic
        % orbit:

        DesiredDirection = Vee(:,3); % the last column, stable 1D manifold of P2
        ICs = P2' + HowClose*DesiredDirection; % start close to P2. P2 is a row vector,
                                               % DesiredDirection is a column vector

        tend=-10;    % a negative end time, and zero start time, will run time backwards
                        % longer times go to very large |z| values
        RDE = @(t,y)RosslerDE(t,y,c);
        [~,Ysol]=ode45(RDE,[0, tend],ICs,options);

        zvalsr=Ysol(:,3);
        zgot = zvalsr(end);
    
        
    end


end