function fig4_5
%% To find and plot the unstable homoclinic orbit for origin in the Lorenz equations
% This generates figures 4.5 and 4.2 in our Chaos book. Shoots for an orbit
% that is close to a homoclinic orbit, when r is 13.926.
% 

% Shoot using Zmax values, as in fig 4.4. Just do one side and then
% use symmetry for the other side (x to -x, y to -y). This also generates
% Fig 4.2 in our "Chaos" book
%
% Mark McGuinness

tend=80;
nt=501;
times=linspace(0,tend,nt);  %use these times to get output

options=odeset('RelTol',1e-13,'AbsTol',1e-13, 'Events',@events);

mywidth=3; defFont=24;

sigma=10; b=8/3;  r=13.926;

xfx=sqrt(b*(r-1));
xf1=[xfx, xfx, r-1];

lambdap=(1+sigma)/2*(-1+ sqrt(1+4*sigma*r/(1+sigma)^2));

xeps=0.01; yeps=xeps*(1+lambdap/sigma); 

% we shoot on z to get unstable  orbit; we run 
% an initial condition near the unstable origin, out into the
% positive quadrant along the unstable manifold. Run
% until the orbit reaches a minimum in z. Split
% this between that min being at a positive x, and at
% a negative x (shooting for x=0 in fact)

    
% We need two initial conditions that bracket the unstable limit cycle
% I checked by hand that z=0 and z=-0.01 do bracket the limit cycle OK
    
 ICpos=[xeps,yeps,0]; ICneg=[xeps,yeps,-0.01];
    
% do interval bisection on ICplus, ICneg to get close to limit cycle
    
success=false; nbisect=70; tol=1.0e-14;
for II=1:nbisect
   ICs = (ICpos + ICneg)/2.0;
 
   [~,Ysol,TE,YE,~]=ode45(@LorenzDE,times,ICs,options);

    maxima= YE;  % the (x,y,z) coords of first minimum in z

    xvals=Ysol(:,1); yvals=Ysol(:,2); zvals=Ysol(:,3);

    if  maxima(1,1)<0             
        %disp(' goes neg ')           
        ICneg=ICs;
    else 
        %disp('goes positive')
        ICpos=ICs;
    end          

    if  norm(ICpos-ICneg) < tol*norm(ICpos) % converged OK 
        %disp('SUCCESS in bisection')
        success=true;
        % but watch out for a final run that flips to the negative
        % quadrant. Simply use the last ICpos run:
        break
    end

end
    
if ~success
    disp('failed to get close to unstable LC, bailing out')
    disp(maxima)
    figure(10)
    plot3(xvals,yvals,zvals,'-k','linewidth',mywidth)
    return
end
    
% now do a nice final run - high time res, and guaranteed same side:

ICs=ICpos;  %use last available ICpos to be on inside:
times=linspace(0,TE(1),501);

[Tsol,Ysol,~,~,~]=ode45(@LorenzDE,times,ICs,options); %get TE and YE
xvals=Ysol(:,1); yvals=Ysol(:,2); zvals=Ysol(:,3);

figure(1)
clf('reset')

plot3(xvals,yvals,zvals,'-k','linewidth',mywidth)
hold on;
plot3(-xvals,-yvals,zvals,'-k','linewidth',mywidth)

ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;
xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont,'Rotation',0.0);
zlabel('z','FontSize',1.3*defFont,'Rotation',0.0);
title('converged shot','FontSize',1.3*defFont)

hold off;

    figure(2)
    clf('reset')

    plot(xvals,zvals,'-r','linewidth',4)
    hold on;
    plot(-xvals,zvals,'-r','linewidth',4)
    plot(xf1(1), xf1(3), '.r','MarkerSize',40);
    plot(-xf1(1), xf1(3), '.r','MarkerSize',40);
    
    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.XTickMode='manual';  
    ax.YTickMode='manual';
    ax.LineWidth=2.0;ax.Box='off';
    xlabel('x','FontSize',1.3*defFont); ylabel('z','FontSize',1.3*defFont,'Rotation',0.0);

    ax.YAxisLocation = 'origin';
    ax.XAxisLocation = 'origin';

    hold off;

figure(3)
clf('reset')
plot(Tsol,xvals,'-k','linewidth',4)

ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;ax.Box='off';
xlabel('t','FontSize',1.3*defFont); ylabel('x','FontSize',1.3*defFont,'Rotation',0.0);


function F=LorenzDE(~,yy)
    
   % Lorenz equations
    x=yy(1);
    y=yy(2);
    z=yy(3);
    
    f1  = sigma*(y-x);
    f2 = (r-z)*x -y;
    f3 = x*y -b*z;
    F=[f1; f2; f3];

end

function [value,isterminal,direction]=events(~,yy)
    isterminal=1; % stop
    direction=1; %inreasing through zero is wanted for local min
    x=yy(1);
    y=yy(2);
    z=yy(3);

    value=x*y -b*z; % detects if this product is zero. It is dz/dt. 
                    % For a minimum, this needs to be an increasing
                    % function, so direction is +1
end

end