function fig4_7
%% Generate Fig 4.7 in Chaos book; homoclinic orbits and a nearby orbit; 2D map
% to generate figure 4.7 in our chaos book 2016. Shoot for the homoclinic orbits 
% when r is 13.926, just like in fig 4.5. But change axes to line up with the
% unstable manifold of origin, before plotting. And add another orbit to
% illustrate the 2D map
% 
% shoot using Zmax values, as for fig 4.4. Just do one side and then
% use symmetry for the other side (x to -x, y to -y)
%
% Mark McGuinness

tend=80;
nt=501;
times=linspace(0,tend,nt);  %use these times to get output at

options=odeset('RelTol',1e-13,'AbsTol',1e-13, 'Events',@events);
options2=odeset('RelTol',1e-13,'AbsTol',1e-13);  % no events

mywidth=2; defFont=24;

sigma=10; b=8/3;  r=13.926;
lambdap=(sigma+1)/2* (-1 + sqrt(1 + 4*sigma*(r-1)/(sigma+1)^2));
lambdan=(sigma+1)/2* (-1 - sqrt(1 + 4*sigma*(r-1)/(sigma+1)^2));

xeps=0.01; yeps=xeps*(1+lambdap/sigma); 
    
% need to bracket the unstable limit cycle:
% I checked by hand that z=0 and z=-0.01 do bracket the limit cycle OK
    
ICpos=[xeps,yeps,0]; ICneg=[xeps,yeps,-0.01];
    
% do interval bisection on ICplus, ICneg to get close to limit
% cycle
  
success=false; nbisect=70; tol=1.0e-14;
for II=1:nbisect
   ICs = (ICpos + ICneg)/2.0;

   [~,Ysol,TE,YE,~]=ode45(@LorenzDE,times,ICs,options);

    maxima= YE;  % the (x,y,z) coords of first minimum in z

    xvals=Ysol(:,1); yvals=Ysol(:,2); zvals=Ysol(:,3);

    if  maxima(1,1)<0             
        %disp(' goes neg ')           
        ICneg=ICs;
    else 
        %disp('goes positive')
        ICpos=ICs;
    end          

    if  norm(ICpos-ICneg) < tol*norm(ICpos) % converged OK
        %disp('SUCCESS in bisection')
        success=true;
        % but watch out for a final run that flips to the negative
        % quadrant. Simply use the last ICpos run:
        break
    end

end

if ~success
    disp('failed to get close to unstable LC, bailing out')
    disp(maxima)
    figure(10)
    plot3(xvals,yvals,zvals,'-k','linewidth',mywidth)
    return
end

% do a nice run - high time res, and guaranteed same side:

ICs=ICpos;  %use last available ICpos to be on inside:
times=linspace(0,TE(1),501);

[~,Ysol,~,~,~]=ode45(@LorenzDE,times,ICs,options); %get TE and YE
xvals=Ysol(:,1); yvals=Ysol(:,2); zvals=Ysol(:,3);

% get an illustrative  orbit that goes from x_1 near 7 and z~3, once
% around and back to a particular z value coming down, then (not dashed) on to
% x_1 near 10 again:

x1box=7; x1start=x1box; x2start=-1.4; zstart=3.0;
zbox=10.0; x2box=x1box*2/3;

xstart = sigma*(x1start+x2start)/(lambdan+lambdap);
ystart= x1start + xstart/sigma *(sigma - lambdan);

ICs=[xstart, ystart, zstart];  %start at right face of box:
    
% do a small time run to get away from face and possible min z value
% 1

tendshort = 0.1;
[~,Ysol]=ode45(@LorenzDE,[0,tendshort],ICs,options2);
xvals2=Ysol(:,1); yvals2=Ysol(:,2); zvals2=Ysol(:,3);

%now run until solution comes down through the upper face of box at z value = zbox:
%2

tendlonger = 5;
options3=odeset('RelTol',1e-13,'AbsTol',1e-13, 'Events',@events3);
ICs = [xvals2(end), yvals2(end), zvals2(end)];
[~,Ysol,~,~,~]=ode45(@LorenzDE,[0,tendlonger],ICs,options3);
xvals3=Ysol(:,1); yvals3=Ysol(:,2); zvals3=Ysol(:,3);

% now run a bit longer, until solution again meets vert face of box:
%3

options4=odeset('RelTol',1e-13,'AbsTol',1e-13, 'Events',@events4);
ICs = [xvals3(end), yvals3(end), zvals3(end)];
[~,Ysol,~,~,~]=ode45(@LorenzDE,[0,tendlonger],ICs,options4);
xvals4=Ysol(:,1); yvals4=Ysol(:,2); zvals4=Ysol(:,3);

% do this all again, now that we are sure to be close to the stable
% manifold:
% 1 again

ICs=[xvals4(end), yvals4(end), zvals4(end)];
[~,Ysol]=ode45(@LorenzDE,[0,tendshort],ICs,options2);
xvals2=Ysol(:,1); yvals2=Ysol(:,2); zvals2=Ysol(:,3);

%now run until solution comes down through the upper face of box at z value = zbox:
%2 again

ICs = [xvals2(end), yvals2(end), zvals2(end)];
[~,Ysol,~,~,~]=ode45(@LorenzDE,[0,tendlonger],ICs,options3);
xvals3=Ysol(:,1); yvals3=Ysol(:,2); zvals3=Ysol(:,3);

% now run a bit longer, until solution again meets right face of box:
% 3 again;

% plot 3d using the new coordinates x1 and x2 from the book. They are in 
% the directions of the stable and unstable eigenvectors in the xy-plane:

figure(1)
clf('reset')
x1vals = (lambdan/sigma -1) *xvals + yvals;
%display(['lamban is', num2str(lambdan)])
x2vals = (lambdap/sigma +1) *xvals - yvals;
%display(['lambap is', num2str(lambdap)])

plot3(x1vals,x2vals,zvals,'-k','linewidth',mywidth) %plot one wing of homoclinic orbit
hold on;
plot3(-x1vals,-x2vals,zvals,'-k','linewidth',mywidth) %plot other wing of homoclinic orbit

x1vals2 = (lambdan/sigma -1) *xvals2 + yvals2;
x2vals2 = (lambdap/sigma +1) *xvals2 - yvals2;

plot3(x1vals2,x2vals2,zvals2,'-r','linewidth',mywidth) %plot first part of a regular orbit, outside box

x1vals3 = (lambdan/sigma -1) *xvals3 + yvals3;
x2vals3 = (lambdap/sigma +1) *xvals3 - yvals3;

plot3(x1vals3,x2vals3,zvals3,'-r','linewidth',mywidth) %plot second part of a regular orbit, still outside box

x1vals4 = (lambdan/sigma -1) *xvals4 + yvals4;
x2vals4 = (lambdap/sigma +1) *xvals4 - yvals4;

plot3(x1vals4,x2vals4,zvals4,'-r','linewidth',mywidth) %plot third part of a regular orbit, inside box

% put dots where the above orbits start and stop:

plot3(x1vals3(end), x2vals3(end), zvals3(end), '.r', 'MarkerSize',30);
plot3(x1vals4(1), x2vals4(1), zvals4(1), '.r', 'MarkerSize',30);
plot3(x1vals4(end), x2vals4(end), zvals4(end), '.r', 'MarkerSize',30);

% put a box in "near" origin:

x1b1=x1box; x2b1=x2box;

%bottom square:
bottomsqx1 = [x1b1; x1b1; -x1b1; -x1b1; x1b1];
bottomsqx2 = [-x2b1; x2b1; x2b1; -x2b1; -x2b1];
bottomsqz  = zeros(5,1);

%top square:
topsqx1 = bottomsqx1;
topsqx2 = bottomsqx2;
topsqz =  bottomsqz+zbox;

%verticals:
v1x1 = [bottomsqx1(1); topsqx1(1)];
v1x2 = [bottomsqx2(1); topsqx2(1)];
v1z = [bottomsqz(1); topsqz(1)];

v2x1 = [bottomsqx1(2); topsqx1(2)];
v2x2 = [bottomsqx2(2); topsqx2(2)];
v2z = [bottomsqz(2); topsqz(2)];

v3x1 = [bottomsqx1(3); topsqx1(3)];
v3x2 = [bottomsqx2(3); topsqx2(3)];
v3z = [bottomsqz(3); topsqz(3)];

v4x1 = [bottomsqx1(4); topsqx1(4)];
v4x2 = [bottomsqx2(4); topsqx2(4)];
v4z = [bottomsqz(4); topsqz(4)];

plot3(bottomsqx1, bottomsqx2, bottomsqz,'-k','linewidth',mywidth);
plot3(topsqx1, topsqx2, topsqz,'-k','linewidth',mywidth);
plot3(v1x1, v1x2, v1z,'-k','linewidth',mywidth);
plot3(v2x1, v2x2, v2z,'-k','linewidth',mywidth);
plot3(v3x1, v3x2, v3z,'-k','linewidth',mywidth);
plot3(v4x1, v4x2, v4z,'-k','linewidth',mywidth);

shortlinex1x = [0; 20]; shortlinex1y = [0; 0]; shortlinex1z = [0; 0];
shortlinex2x = [0; 0]; shortlinex2y = [0; 10]; shortlinex2z = [0; 0];

plot3(shortlinex1x,shortlinex1y,shortlinex1z,'-k','linewidth',mywidth/2)
plot3(shortlinex2x,shortlinex2y,shortlinex2z,'-k','linewidth',mywidth/2)

ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;
xlabel('x_1','FontSize',1.3*defFont); ylabel('x_2','FontSize',1.3*defFont,'Rotation',0.0);
zlabel('z','FontSize',1.3*defFont,'Rotation',0.0);
view([1,1,1]);

hold off;

%=========================================================================

function F=LorenzDE(~,yy)
    
   % Lorenz equations
    x=yy(1);
    y=yy(2);
    z=yy(3);
    
    f1  = sigma*(y-x);
    f2 = (r-z)*x -y;
    f3 = x*y -b*z;
    F=[f1; f2; f3];

end

%=========================================================================

function [value,isterminal,direction]=events(~,yy)
    isterminal=1; % stop
    direction=1; %inreasing through zero is wanted for local min
    x=yy(1);
    y=yy(2);
    z=yy(3);

    value=x*y -b*z; % detects if this product is zero. It is dz/dt. 
                    % For a minimum, this needs to be an increasing
                    % function, so direction is +1
end

%=========================================================================

   function [value,isterminal,direction]=events3(~,yy)
        isterminal=1; % stop
        direction=-1; %decreasing through zero is wanted for z dropping through value zbox
        
        z=yy(3);

        value=z - zbox; % detects if this difference is dropping thru zero. 
    end

%=========================================================================

   function [value,isterminal,direction]=events4(~,yy)
        isterminal=1; % stop
        direction=1; %increasing through zero is wanted for x increasing through value xstart
        
        x=yy(1);
        y=yy(2);       
        
        x1= (lambdan/sigma -1) *x + y;
        value=x1 - x1start; % detects if this difference is increasing thru zero. 
    end

end