function fig5_13
% This generates figure 5.13 in our Chaos book 2019. 
% Slices through the time 
% evolution of a perturbed hamiltonian system - a restricted
% 3-body problem, relevant to resonances in the Sun/Jupiter system

% r is earth's distance from centre of mass, psi = theta-t is angle but 
% it counter-rotates with sun and jupiter system, h is earth's angular
% momentum, E is kinetic energy plus potential energy (only conserved if
% perturbation epsilon is zero)

% for earth, r is 0.19, thetadot is 11.9. Compute J for earth and use that
% with various initial r values, and zero rdot. h for earth is about 0.43.
% 
% to improve the accuracy near r=0, I rescaled time to t = r^3 T 
% where T is the new value of time 

% takes 40 minutes to run if tend is set to 500,000

% Mark McGuinness  Dec 2019


tic

rmax=10.0; %stop if an r value exceeds rmax
eps=0.001; %the perturbation term

options=odeset('RelTol',1e-11,'AbsTol',1e-11, 'Events',@events);
 
defFont=36;
 
%===========================================================================
%
% solve DEs and look at results:
% 
%  earth has r0=0.19:

% dJ=7.79e08;
% GEE=6.673e-11;
% Ms=2e30;

%rearth=1.5e08/7.79e08;

%display(rearth)

% find J for earth:
%tscale = sqrt(dJ^3/(GEE*Ms));
%display(tscale)

%tdotearth=2*pi*tscale;
%display(tdotearth)

% hearth=rearth*rearth*11.86;

%  We are going to use the h that sets
% the minimum of J at rearth:  truncated  to 0.19:

rearth=0.19;

Huse= sqrt(rearth + eps*rearth^4/(1+rearth^2)^(3/2));  %eqn 5.209 in book
display(Huse)

[Jearth,~] = GetJay(0.0,Huse,rearth,eps,pi/2);

display(Jearth)  % this is now the same as the value as in caption to Fig. 5.13

%v0=2.1025; % exhibits narrow secondary tori; 2.10278 hits sun 

% we set different end times for different initial conditions, since some
% ICs require longer run times to see enough of the poincare section:

v0=  [  0.75; 1.225; 1.60;  2;   2.03; 2.05; 2.06; 2.07; 2.08; 2.09; 2.1025;  2.1027];
tend=[   500;  500;  2000; 5000; 5000; 5000; 5000; 5000; 5000; 5000;  500000; 500000];

v0=[v0; 13];
tend=[tend; 500000];
tend=10*tend;  % takes a minute to run
               % SET THIS TO 1000*tend TO GET THE FIGURE IN THE BOOK; TAKES AN HOUR TO RUN

psi0=zeros(size(v0))+ pi/2;
r0=0.19*ones(size(v0));
r0(end)=0.003;


%h0earth=0.44; % earth's h value

Jearth=-3.067 - eps;
disp('but Jearth we are actually using is')
disp(Jearth)

h02=SolveJay2(v0,r0,eps,psi0,Jearth);  % returns two h values from the quadratic for h
 
h0=h02(:,2);  % pick out the second column, the positive root for h

ICs=[r0, v0, psi0, h0]; % each column is one set 

display(ICs)

ps = cell(length(r0));  % a cell array, each item to hold the array which is the poincare sections for each
                         % initial condition, ie values at each time (row)
Ts=ps; Yevents=ps; Ievents=ps; Tevents=ps;

%=========================================

for II=1:length(ICs(:,1))  % run once for each set of initial conditions
    disp(II)
    disp('out of')
    disp(length(ICs(:,1)));
    disp('===')
    IC=ICs(II,:); %each row of ICs is one set of initial conditions     
    
    [Tsol,Ysol, Tev, Yev,IE]=ode45(@R3BDE,[0, tend(II)],IC,options); % I tried stiff solver 15s but accuracy suffered
    
    ps{II} = Ysol;  % this is r, v, psi, h. One row for each time; four columns
    Ts{II} = Tsol;  %save the times that ode45 used to get desired accuracy

    %display(IC);

    if ~isempty(Yev)
      Ievents{II}=IE;  % the code for which event has triggered this output
      Tevents{II}=Tev; % the time of the event
      Yevents{II}=Yev;  %values of r, v, psi, h when this event occurs     
    else
      disp('empty events')
    end
    
    if IE(end) == 3
        disp('escapes')      
    elseif IE(end) == 4
        disp('hits sun')      
    elseif IE(end) == 5
        disp('hits Jupiter')      
    end
    
end

disp('finished solving DEs')

%========================================================================
 
figure(1)  % plot rdot vs r, in a poincare section through psi = pi mod 2pi, event 2
clf('reset')

hold on;
  
numICS=length(ICs(:,1));

for II=1:numICS  % look at first half of the initial conditions. These have the "negative" root for h0
  
  if ~isempty(Yevents{II})
      % look for a section through psi = pi/2 mod 2pi; this is event number 2:
      % and I'm requiring psi to increase through that value pi/2.
      % Otherwise the torus appears to loop around on itself
      
      IEp=Ievents{II};
      
      %exclude escaping orbits or orbits that collide with the sun or Jupiter:      
      
      IEpIS3 = IEp==3 | IEp==4 | IEp==5;  % escaping orbits or collision orbits
      %IEpIS3=[];
      if sum(IEpIS3) == 0 %only plot nonescaping and noncolliding orbits
                          % choose points increasing through the poincare
                          % section psi = pi/2:
          IEpIS6 =   IEp==6;  % this is 1 only for events number 6. Is a column vector, ie one column, chooses rows

          rusep = Yevents{II}(IEpIS6,1); % pick out the rows of column 1 (r values) associated with event number 6
          vusep = Yevents{II}(IEpIS6,2); % pick out the rows of column 2 (v values) associated with event number 6
          yp=vusep;  % this is rdot, rhs of differential equation one
          xp=rusep;  % this is r

          ns=1; %number to skip while plotting
          plot(xp(1:ns:end),yp(1:ns:end), '.r','MarkerSize',5)
      else
          disp(II); disp(ICs(II,:)); 
          if IEp==3
              disp('Not plotting since orbit escapes')
          elseif IEp==4
              disp('Not plotting since orbit hits sun')
          else
              disp('Not plotting since orbit hits Jupiter')
          end
      end
  end
end


ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';   
xlabel('r','FontSize',1.5*defFont); ylabel('v','FontSize',1.5*defFont,'Rotation',0.0);

hold off;

 %========================================================================

figure(2)  % plot r cos psi vs r sin psi, in a poincare section through rdot = 0, event 1
           % to see physical space
clf('reset')

hold on;

for II=1:length(ICs(:,1))  % look at each initial condition in turn

 if ~isempty(Yevents{II})
   IEp=Ievents{II};
   %avoid plotting any escaping orbits:

   IEpIS3 = IEp==3 | IEp==4 | IEp==5;  % escaping orbits or collision orbits
   if sum(IEpIS3) == 0 %only plot closed non-colliding orbits

       % note that orbits starting with r>1 seem to reach max r when
       % decreasing through zero velocity, not increasing like the ones starting with
       % r(0)<1:


          % look for a section through v=0 with increasing v; this is event number 1:
          IEpIS1 =   IEp==1;  % this is 1 only for events number 1. Should be a column vector, ie one column, chooses rows
          rusep = Yevents{II}(IEpIS1,1); % pick out the rows of column 1 (r values) associated with event number 2
          psiusep = Yevents{II}(IEpIS1,3); % pick out the rows of column 3 (psi values) associated with event number 2
          yp=rusep.* sin(psiusep);  % this is the physical y coord of an object at radial position r, psi
          xp=rusep.* cos(psiusep);  % this is the physical x coord of an object at radial position r, psi

          ns=1; %number to skip while plotting
          plot(xp(1:ns:end),yp(1:ns:end), '.r','MarkerSize',5) 
   
   end % orbit is closed
    
 end
 
end
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';
xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont,'Rotation',0.0);
hold off;

%========================================================================
 
 figure(3)  % plot J vs time, to check it is conserved OK
clf('reset')

hold on;
  
for II=1:length(ICs(:,1))  % look at each initial condition in turn
      vee= ps{II}(:,2);
      hh= ps{II}(:,4);
      rr = ps{II}(:,1);
      psiv= ps{II}(:,3);
      [yp,~] = GetJay(vee,hh,rr,eps,psiv);  %plot J values
      xp=Ts{II};  % this is time
      % just plot 200 values:
      nplot=200; ns = floor(length(xp)/nplot);  %number to skip while plotting
      plot(xp(1:ns:end),yp(1:ns:end), '-r','LineWidth',3) 
  
end
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';
xlabel('t','FontSize',1.3*defFont); ylabel('J','FontSize',1.3*defFont,'Rotation',0.0);
hold off;
 

toc

%========================================================================

function F=R3BDE(~,yy)
    
   % Andrew's restricted 3-body problem
   
    r=yy(1);
    v=yy(2);
    psi=yy(3);
    h=yy(4);
    
    rsq=r*r;
    rcbd=rsq*r;
    xsq=1+rsq-2*r*cos(psi);
    if xsq < 1.0e-16
        x=1.0e-16; xsq=x*x;
        disp('!!encountered zero or negative x^2!!')
        disp([r,psi,h,xsq])
    else
        x=sqrt(xsq);
    end
    oneOxsq=1/xsq;
    oneOxcubd=oneOxsq/x;
%    oneOrsq = 1/rsq;
%    oneOrcubd=oneOrsq/r;
    % the following are the original equations before rescaling time:
%     f1 =v;     % r dot
%     f2 = h*h*oneOrcubd - oneOrsq + eps*(cos(psi)*(2*oneOrcubd +oneOxcubd) - r*oneOxcubd);  %v dot, corrected 6 Aug 2017
%     f3 = h*oneOrsq - 1;  % psi dot
%     f4= eps*sin(psi)*( oneOrsq -r*oneOxcubd);    %h dot

% here are the equations RHS after rescaling time on r cubed

    f1 =v*r^3;     % r dot
    f2 = h*h - r + eps*(cos(psi)*(2 +rcbd*oneOxcubd) - rcbd*r*oneOxcubd);  %v dot, corrected 6 Aug 2017
    f3 = h*r - rcbd;  % psi dot
    f4= eps*sin(psi)*r*( 1 -rcbd*oneOxcubd);    %h dot
    F=[f1; f2; f3; f4];

end

%========================================================================

   function [value,isterminal,direction]=events(~,yy)
       % four events will be checked: v = r dot = 0 and is increasing, or is decreasing through zero
       %  psi= some value, and r>rmax:
       % 
       
        isterminal=[ 0; 0; 1; 1; 1; 0]; % 0 is don't stop
        direction=[ 1; 1; 0; 0; 0; -1]; % (1) velocity increasing through zero,
                                  % (2) psi = pi/2 in increasing direction
                                  % (3) stop if r=rmax
                                  % (4) stop if r is too small (a hit on the sun)
                                  % (5) stop if x is near zero (earth hits
                                  % Jupiter
                                  % (6) psi=pi/2 in decreasing direction
        
        r=yy(1);
        v=yy(2);
        psi=yy(3);
        %h=yy(4);
        
        xsq=1+r*r-2*r*cos(psi); 
        
        event1  = v;     % r dot is zero for this event
        
        event2 = mod(psi, 2*pi)-pi/2;  % detect when psi is pi/2 (mod 2pi) with psi increasing
        event6 = mod(psi, 2*pi)-pi/2;  % detect when psi is pi/2 (mod 2pi) with psi decreasing
        
        event3= r-rmax;  % stop if r gets too large; escape orbit
        event4 = r - 1.0e-6; % stop if r gets too small, hitting the sun
        event5 = xsq-1.0e-12; %stop if x gets too small, hitting Jupiter
        
        value=[ event1; event2; event3; event4; event5; event6];
   end

%========================================================================

    function [J,E] = GetJay(rdot,h,r,epsilon,psi)
        %get the value of the Jacobi Integral, Jay. Is OK with vector or
        %matrix inputs for rdot, h, r and psi.
        rdotsq=rdot.*rdot;
        rsq=r.*r;
        E = rdotsq/2.0 + (h.*h/2) ./rsq  -1./r;
        
        xsq=1+rsq-2*r.*cos(psi);
        if xsq < 1.0e-16
            x=1.0e-16;
            disp('!!encountered zero or negative x^2 in GetJay!!')
            display([r,psi,h,xsq])
        else
            x=sqrt(xsq);
        end
        
        W = 1./x - cos(psi)./rsq;
        
        J=E-h-epsilon*W;
            
    end

%========================================================================

    function hout = SolveJay2(rdot,r,epsilon,psi,Jval)
        %Solve for J= a given value, by finding h with
        %given rdot, r, epsilon and psi values. size of hout is same as
        %size of r, psi and rdot. All same or a scalar one is OK. scalar
        %Jval is OK with others being vectors, too
        rsq=r.*r;
        
        xsq=1+rsq-2*r.*cos(psi);
        if xsq < 1.0e-16
            x=1.0e-16; 
            disp('!!encountered zero or negative x^2 in SolveJay!!')
            disp('r,psi,h,xsq:')
            display([r,psi,h,xsq])
        else
            x=sqrt(xsq);
        end
        
        W = 1./x - cos(psi)./rsq;
        
        Disc=1-2./ rsq .*( rdot.*rdot/2 - 1./ r-epsilon*W - Jval);
        hout1 = rsq.*( 1 - sqrt(Disc)); 
        hout2 = rsq.*( 1 + sqrt(Disc));
        negv=Disc<0;  % watch out for negative discriminant values
        hout1(negv) = -999; %replace negative disc with h = -999
        hout2(negv) = -999; %replace negative disc with h = -999
        hout = [hout1, hout2];

        
    end 

  
end