function fig5_15
%% to generate figure 5.15  in Chaos. 

% Slices through the time 
% evolution of a perturbed hamiltonian system - a restricted
% 3-body problem, relevant to resonances in the Sun/Jupiter system

% r is earth's distance from centre of mass, psi = theta-t is angle but 
% it counter-rotates with sun and jupiter system, h is earth's angular
% momentum, E is kinetic energy plus potential energy (only conserved if
% perturbation epsilon is zero)

% The interesting r values lie in the
% range 0.13 to 0.27.

% Constant J= -3.164.
%
% for earth, r is 0.19, thetadot is 11.9. Compute J for earth and use that
% with various initial r values, and zero rdot. h for earth is about 0.43.
%
%  J~-3.164 is calculated using eps=0.1 and psi(0)=pi/2
%
% Mark McGuinness



rmax=10.0; %stop if an r value exceeds rmax
eps=0.1; %the perturbation term

% I am setting fixed J values, by adjusting h values after choosing initial
% values for r and v (and psi=pi/2 and epsilon=0.1).

 
options=odeset('RelTol',1e-7,'AbsTol',1e-7, 'Events',@events);
 
defFont=24;
 

%===========================================================================
%
% solve DEs and look at results:
% 
%  earth has r0=0.19:


% Initial Conditions:

 v01=[0.07; 0.15; 0.2; 0.25; 0.3; 0.35; 0.4; 0.45; 0.5; 0.7; 0.8; 0.9; 1; 1.1]; % 
 r01=0.2*ones(size(v01));
 
 v02=[ -0.4; -0.2; -0.1; 0; 0.1; 0.2; 0.3; 0.35; 0.4; 0.45; 0.7];
 r02=0.165*ones(size(v02));
 
 v03 = [0.43; 0.44; 0.45; 0.46; 0.47; 0.48]; r03 = 0.18+ zeros(size(v03));
 
 r04 = [0.166; 0.1665; 0.167; 0.1675; 0.168; 0.172; 0.174; 0.176]; v04=0.3 + zeros(size(r04));

 r05 = [0.172; 0.1725; 0.173; 0.1735]  ;  v05 = 0.56 + zeros(size(r05));
 
 r06 = [0.1720; 0.17205; 0.1721; 0.17215; 0.1722]  ;  v06 = 0.571 + zeros(size(r06));

 % more points for the biggest blowup plot, with V in [0.5, 0.58] and
 % r in [0.169, 0.173:
%   r07=[0.171; 0.1712; 0.1714; 0.1716; 0.1718; 0.17; 0.17; 0.17; 0.17];   
%   v07=[0.53;0.53;0.53;0.53;0.53; 0.505; 0.51; 0.515;0.52]; 
%   r08=r07+0.002; v08=v07-0.02;
%   r09=[0.173;0.173;0.173;0.173];   v09=[0.505; 0.51; 0.515; 0.54];
%   

%  v0=[v01; v02; v03; v04; v05; v06; v07; v08; v09];
% r0=[r01; r02; r03; r04; r05; r06; r07; r08; r09];

 v0=[v01; v02; v03; v04; v05; v06];
r0=[r01; r02; r03; r04; r05; r06];

tend=5000*ones(size(v0));
%tend=50*ones(size(v0));  % for a quick check
%tend(end-9:end) = 40000;   % need really long times for the last nine values

psi0=zeros(size(v0))+ pi/2;

% rearth=0.19;

% hearth=sqrt(rearth + eps*rearth^4/(1+rearth*rearth)^(1.5)); % the value of h at the centre of the closed loops for J

% use getJ to ensure all orbits have the same J value as earth does for zero r dot and psi=pi/2:
%   [Jearth,~] = GetJay(0,hearth,rearth,eps,pi/2); % gives J = -3.164 approx
% 



Jearth = -3.164;  % when eps is 0.1 and psi is pi/2, using eqn (5.210) gives
                  % h0 = 0.436 and J = -3.1641

                  % now to ensure J is held constant:
                  
h02=SolveJay2(v0,r0,eps,psi0,Jearth);  % returns two h values from the quadratic for h

h0=h02(:,2);  % pick out the second column, the positive root for h

ICs=[r0, v0, psi0, h0]; % each column is one set
nICS=length(ICs(:,1));
noICS=round(linspace(1,nICS,nICS));
numberedICS=[noICS', ICs];
disp('initial conditions: #, r, v, psi, h:')
disp(numberedICS)

ps = cell(length(r0));  % a cell array, each item to hold the array which is the poincare sections for each
                         % initial condition, ie values at each time (row)
Ts=ps; Yevents=ps; Ievents=ps; Tevents=ps;

for II=1:length(ICs(:,1))  % run once for each set of initial conditions
    disp([num2str(II) ' / ', num2str(length(ICs(:,1)))])
    IC=ICs(II,:); %each row of ICs is one set of initial conditions     
    
    [Tsol,Ysol, Tev, Yev,IE]=ode45(@R3BDE,[0, tend(II)],IC,options);
    
    ps{II} = Ysol;  % this is r, v, psi, h. One row for each time; four columns
    Ts{II} = Tsol;  %save the times that ode45 used to get desired accuracy

    display(IC);

    if ~isempty(Yev)
      Ievents{II}=IE;  % the code for which event has triggered this output
      Tevents{II}=Tev; % the time of the event
      Yevents{II}=Yev;  %values of r, v, psi, h when this event occurs     
    else
      disp('empty events')
    end
    
    if IE(end) == 3
        disp('escapes')      
    elseif IE(end) == 4
        disp('hits sun')      
    elseif IE(end) == 5
        disp('hits Jupiter')      
    end
    
end
disp('finished solving DEs')



 %========================================================================

figure(2)  % plot r cos psi vs r sin psi, in a poincare section through rdot = 0, event 1
           % to see physical space
clf('reset')


xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont,'Rotation',0.0);

hold on;

for II=1:length(ICs(:,1))  % look at each initial condition in turn

 if ~isempty(Yevents{II})
   IEp=Ievents{II};
   %avoid plotting any escaping orbits:

   IEpIS3 = IEp==3 | IEp==4 | IEp==5;  % escaping orbits or collision orbits
   if sum(IEpIS3) == 0 %only plot closed non-colliding orbits

      % look for a section through v=0 with increasing v; this is event number 1:
      IEpIS1 =   IEp==1;  % this is 1 only for events number 1. Should be a column vector, ie one column, chooses rows
      rusep = Yevents{II}(IEpIS1,1); % pick out the rows of column 1 (r values) associated with event number 2
      psiusep = Yevents{II}(IEpIS1,3); % pick out the rows of column 3 (psi values) associated with event number 2
      yp=rusep.* sin(psiusep);  % this is the physical y coord of an object at radial position r, psi
      xp=rusep.* cos(psiusep);  % this is the physical x coord of an object at radial position r, psi

      ns=1; %number to skip while plotting
      plot(xp(1:ns:end),yp(1:ns:end), '.r','MarkerSize',5) 
   
   end % orbit is closed
    
 end
 
end
    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.LineWidth=2.0;  ax.Box='on';
%    
 hold off;

%========================================================================
 
figure(3)  % plot J vs time, to check it is conserved OK
clf('reset')


xlabel('t','FontSize',1.3*defFont); ylabel('J','FontSize',1.3*defFont,'Rotation',0.0);

hold on;
  
for II=1:length(ICs(:,1))  % look at each initial condition in turn
      vee= ps{II}(:,2);
      hh= ps{II}(:,4);
      rr = ps{II}(:,1);
      psiv= ps{II}(:,3);
      [yp,~] = GetJay(vee,hh,rr,eps,psiv);  %plot J values
      xp=Ts{II};  % this is time
      % just plot 200 values:
      nplot=200; ns = floor(length(xp)/nplot);
      %ns=1000; %number to skip while plotting
      plot(xp(1:ns:end),yp(1:ns:end), '-r','LineWidth',3) 
  
end
    ax=gca;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.LineWidth=2.0;  ax.Box='on';
 hold off;
 
%========================================================================
 
figure(1)  % plot rdot vs r, in a poincare section through psi = pi mod 2pi, event 2
clf('reset')



hold on;
  
numICS=length(ICs(:,1));

for II=1:numICS  
  
  if ~isempty(Yevents{II})
      % look for a section through psi = pi/2 mod 2pi; this is event number 2:
      % and I'm requiring psi to increase through that value pi/2.
      % Otherwise the torus appears to loop around on itself
      
      IEp=Ievents{II};
      
      %avoid plotting any escaping orbits:
      
      IEpIS3 = IEp==3 | IEp==4 | IEp==5;  % avoid escaping orbits or collision orbits      
      % IEpIS3 =[]; % plot escaping orbits, up to the time they escape
      
      % choose events where psi is increasing through pi/2:
      IEpIS2 =   IEp==2;  % this is 1 only for events number 2. Is a column vector, ie one column, chooses rows

      rusep = Yevents{II}(IEpIS2,1); % pick out the rows of column 1 (r values) associated with event number 2
      vusep = Yevents{II}(IEpIS2,2); % pick out the rows of column 2 (v values) associated with event number 2
      yp=vusep;  % this is rdot, rhs of differential equation one
      xp=rusep;  % this is r

      if sum(IEpIS3) == 0 %only plot nonescaping and noncolliding orbits

          ns=1; %number to skip while plotting
          plot(xp(1:ns:end),yp(1:ns:end), '.r','MarkerSize',5)
      else
          disp(II)
          disp('plotting v against r')
          disp('ICs are')
          disp(ICs(II,:))
          disp('but not plotting since orbit escapes or hits something')
          disp('last few r, v values are')
          disp(xp(end-5:end))
          disp(yp(end-5:end))
      end
  end
end
    ax=gca;

    ax.FontSize=defFont; ax.FontName='Times';

    ax.LineWidth=2.0;  ax.Box='on';
    xlabel('r','FontSize',1.7*defFont); 
    ylabel('v','FontSize',1.7*defFont,'Rotation',0.0);

 hold off;


%========================================================================

function F=R3BDE(~,yy)
    
   %% Restricted 3-body problem
   
    r=yy(1);
    v=yy(2);
    psi=yy(3);
    h=yy(4);
    
    rsq=r*r;
    xsq=1+rsq-2*r*cos(psi);
    if xsq < 1.0e-16
        x=1.0e-16; xsq=x*x;
        disp('!!encountered zero or negative x^2!!')
        display([r,psi,h,xsq])
    else
        x=sqrt(xsq);
    end
    oneOxsq=1/xsq;
    oneOxcubd=oneOxsq/x;
    oneOrsq = 1/rsq;
    oneOrcubd=oneOrsq/r;
    f1 =v;     % r dot
    f2 = h*h*oneOrcubd - oneOrsq + eps*(cos(psi)*(2*oneOrcubd +oneOxcubd) - r*oneOxcubd);  %v dot, corrected 6 Aug 2017
    f3 = h*oneOrsq - 1;  % psi dot
    f4= eps*sin(psi)*( oneOrsq -r*oneOxcubd);    %h dot
    F=[f1; f2; f3; f4];

end

%========================================================================

   function [value,isterminal,direction]=events(~,yy)
       %% six events will be checked: 
       % v = r dot = 0 and is increasing through zero
       %  psi is increasing through pi/2, psi is decreasing through pi/2,
       % r>rmax, r too small, x too small.
       % stop for the too large or too small events only.
       
        isterminal=[ 0; 0; 1; 1; 1; 0]; % 0 is don't stop
        direction=[ 1; 1; 0; 0; 0; -1]; % (1) velocity increasing through zero,
                                  % (2) psi = pi/2 in increasing direction
                                  % (3) stop if r=rmax
                                  % (4) stop if r is too small (a hit on the sun)
                                  % (5) stop if x is near zero (earth hits
                                  % Jupiter
                                  % (6) psi=pi/2 in decreasing direction
        
        r=yy(1);
        v=yy(2);
        psi=yy(3);
        %h=yy(4);
        
        xsq=1+r*r-2*r*cos(psi); 
        
        event1  = v;     % r dot is zero for this event
        
        event2 = mod(psi, 2*pi)-pi/2;  % detect when psi is pi/2 (mod 2pi) with psi increasing
        event6 = mod(psi, 2*pi)-pi/2;  % detect when psi is pi/2 (mod 2pi) with psi decreasing
        
        event3= r-rmax;  % stop if r gets too large; escape orbit
        event4 = r - 1.0e-6; % stop if r gets too small, hitting the sun
        event5 = xsq-1.0e-12; %stop if x gets too small, hitting Jupiter
        
        value=[ event1; event2; event3; event4; event5; event6];
   end

%========================================================================

    function [J,E] = GetJay(rdot,h,r,epsilon,psi)
        %% get the value of the Jacobi Integral, Jay. 
        % Is OK with vector or
        % matrix inputs for rdot, h, r and psi.
        rdotsq=rdot.*rdot;
        rsq=r.*r;
        E = rdotsq/2.0 + (h.*h/2) ./rsq  -1./r;
        
        xsq=1+rsq-2*r.*cos(psi);
        if xsq < 1.0e-16
            x=1.0e-16;
            disp('!!encountered zero or negative x^2 in GetJay!!')
            display([r,psi,h,xsq])
        else
            x=sqrt(xsq);
        end
        
        W = 1./x - cos(psi)./rsq;
        
        J=E-h-epsilon*W;
            
    end

%========================================================================

    function hout = SolveJay2(rdot,r,epsilon,psi,Jval)
        %% Solve for J= a given value, 
        % by finding h with
        % given rdot, r, epsilon and psi values. size of hout is same as
        % size of r, psi and rdot. All same or a scalar one is OK. scalar
        % Jval is OK with others being vectors, too
        rsq=r.*r;
        
        xsq=1+rsq-2*r.*cos(psi);
        if xsq < 1.0e-16
            x=1.0e-16; 
            disp('!!encountered zero or negative x^2 in SolveJay!!')
            disp('r,psi,h,xsq:')
            display([r,psi,h,xsq])
        else
            x=sqrt(xsq);
        end
        
        W = 1./x - cos(psi)./rsq;
        
        Disc=1-2./ rsq .*( rdot.*rdot/2 - 1./ r-epsilon*W - Jval);
        hout1 = rsq.*( 1 - sqrt(Disc)); 
        hout2 = rsq.*( 1 + sqrt(Disc));
        negv=Disc<0;  % watch out for negative discriminant values
        hout1(negv) = -999; %replace negative disc with h = -999
        hout2(negv) = -999; %replace negative disc with h = -999
        hout = [hout1, hout2];

        
    end 


end