function fig6_4
%% Generates the power spectra shown in figure 6.4 of the Chaos book
% Based on 2 periodograms of a solution to the Lorenz equations
% Mark McGuinness

sigma=10; r=28; b=8/3; % parameter values to be used in the Lorenz equations
 
ICs=[5 5 10];  %for Lorenz equations

options=odeset('RelTol',1e-4,'AbsTol',1e-4);

mywidth=1;

% do everything twice, once for time up to 800, and again for times up to
% 6400:

tend=800;  % this is actually 2T in the text of our Chaos book

[Tsol,Ysol]=ode45(@MyDE,[0,tend],ICs,options);   

Ys=Ysol;
xvals=Ys(:,1);

% plot the x component of the solution, versus time, just up to t=100:

fno=1;
defFont=36;

figure(fno);
clf('reset')
 
plot(Tsol,xvals,'-k','linewidth',mywidth)

ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
xlabel('t'); ylabel('x','Rotation',0.0);
grid off;
ax.LineWidth=2.0;  ax.Box='on';
xlim([0 100]) %limit the time range on the x-axis to [0,100]


% get a power of 2 number of points interpolation for the periodogram:

nts=2^12 ; %512*8*8 for T=6400=tend
tspan=linspace(0,tend,nts);
dt = tspan(2)-tspan(1);
Ys =interp1(Tsol,Ysol,tspan)';
xvals=Ys(1,:); 
    
nvals=length(xvals);
    
[pxx,w] = periodogram(xvals,[],nvals,'twosided'); % force two-sided since
                                            % otherwise it will do a
                                            % one-sided, the default for
                                            % real data, and a one-sided
                                            % involves it doubling all
                                            % powers except for the first and
                                            % last frequency..

% plot the power spectrum by using the periodogram results:
fno=fno+1;
figure(fno);
clf('reset')
 
plot(w/dt,2*pi*dt*pxx,'-r','linewidth',mywidth) % correct conversions for two-sided

ax=gca;
defFont=36;
ax.FontSize=defFont; ax.FontName='Times';
xlabel('\omega'); ylabel('P','Rotation',0.0);
xlim([0 15])
grid off;
ax.XTick=[0, 5, 10, 15];
ax.LineWidth=2.0;  ax.Box='on';


tend=800*8;  % this is 2T in the text of our Chaos book

[Tsol,Ysol]=ode45(@MyDE,[0,tend],ICs,options);

% get a power of 2 number of points interpolation for the periodogram:

nts=2^12*8 ; %512*8*8 for T=6400=tend
tspan=linspace(0,tend,nts);
dt = tspan(2)-tspan(1);
Ys =interp1(Tsol,Ysol,tspan)';
xvals=Ys(1,:); 
    
nvals=length(xvals);
    
[pxx,w] = periodogram(xvals,[],nvals,'twosided'); % force two-sided since
                                            % otherwise it will do a
                                            % one-sided, the default for
                                            % real data, and a one-sided
                                            % involves it doubling all
                                            % powers except for the first and
                                            % last frequency..

% plot the power spectrum by using the periodogram results:
fno=fno+1;
figure(fno);
clf('reset')
 
plot(w/dt,2*pi*dt*pxx,'-r','linewidth',mywidth) % correct conversions for two-sided

ax=gca;
defFont=36;
ax.FontSize=defFont; ax.FontName='Times';
xlabel('\omega'); ylabel('P','Rotation',0.0);
xlim([0 15])
grid off;
ax.XTick=[0, 5, 10, 15];
ax.LineWidth=2.0;  ax.Box='on';


function F=MyDE(~,yy)
% the Lorenz equations
    x=yy(1);
    y=yy(2);
    z=yy(3);
    f1  = sigma*(y-x);
    f2 = (r-z)*x -y;
    f3 = x*y -b*z;
    F=[f1; f2; f3];
    
 
end

end