function fig6_7
%% Generates a singular value decomposition as in Fig 6.7 in our book Chaos.
% An svd of the solution to the lorenz equations plus some noise is
% generated. 
% The dimension d_E is set to 37, time step is 0.01, lag
% window is 0.36, and 10,000 time-series points are used.
% The variance of the white noise added to the signal is set to 16, 
% by multiplying the output of randn by sqrt(16)=4. The white noise is filtered out of
% the signal by using the svd as described in the book.
% 
% Three figures are output: 
% fig 1 is the singular values obtained before each decomposition
% fig 2 is fig 6.7, and shows a clean solution x(t) to the Lorenz equation
% together with a lagged plot of x(t), in red; then a raw signal obtained by
% adding white noise to x(t), then the filtered signal obtained by using 
% just the first three singular values, then the result of filtering this 
% filtered signal again by keeping only the first three singular values of 
% its svd. Note that the second filtering has made little difference to the
% plots, suggesting that most of the noise has been filtered out. Note the
% differences and similarities to the red plots of the clean signal.
% fig 3 shows the result of filtering one more time.

% Mark McGuinness

sigma=10; r=28; b=8/3;
dE=37; npt=10000; dt=0.01; variance=16;
stdev=sqrt(variance);

nkeep=3;  %keep this number of singular values
t1=0; t2=10; % plots are to be for times between t1 and t2

mywidth=2; fno=0;
lag=floor(dE/2);
istop=floor(t2/dt); istart=floor(t1/dt)+1;


defFont = 24; ffac=1.5;
    
options=odeset('RelTol',1e-4,'AbsTol',1e-4);


% run for a short while first, to get onto the chaotic attractor:
tend=2;
ICs=[5 5 10];  %for lorenz equations
[~,Ysol]=ode45(@MyDE,[0,tend],ICs,options);  %solve the Lorenz eqns, short run
ICs=Ysol(end,:); % save the final point to be new IC
    
tend=dt*npt;

[Tsol,Ysol]=ode45(@MyDE,[0,tend],ICs,options); %solve the Lorenz equations

tspan=linspace(0,tend,npt);
Ys =interp1(Tsol,Ysol,tspan)'; %sample the solution 10000 times
myseed=floor(2^32-19);  % this (any seed) will give reproducible results. Without this
                          % randn will give different results every time
                          % you run randn. Also need the rng command:
rng(myseed);  % to get reproducible results
xvals=Ys(1,:) + stdev*randn(1,npt);  %add white noise with variance = stdev^2


% get ready to do svd on these x values
% break into rows that are dE long (so there are dE cols) but each row is
% only shifted one along in xvals

M=MatriciseTS(xvals,dE);

[U,S,V]= svd(M,0); %get the  decomp. zero means a simple one, stopping at zero sigma/rank of M

disp('first singular values:')
display(diag(S(1:10, 1:10)))

svals=diag(S(1:nkeep,1:nkeep));  % a col vector with first nkeep singular values
svals=[svals; zeros(dE-nkeep,1)]; %pad svals with zeros
Szerod=diag(svals);  % make a diagonal matrix out of the vector svals

disp('first projection')
display(diag(Szerod(1:8, 1:8)))

M2T = U*Szerod*V';

% each row (or column) of M2 represents one "point" of nearly all of the time series, 

% DO NOT average
% along each row to generate a new, filtered, time series, as this averages 
% over time, too long a time:

X2=DEMatriciseTS(M2T); % get X2, the time series corresp to M2T
M2=MatriciseTS(X2,dE); % make this time series into a matrix

% check the mean of each column is in fact close to zero:

CHECKmean=mean(M2);

disp('checking means of M2, should be zero:')
disp(CHECKmean(1:10))
disp(CHECKmean(end-10:end))
disp('    ')

% do another SVD on this filtered data:

[U2,S2,V2]= svd(M2,0); %get the economy size decomp. S2 is dE x dE

disp('second singular values:')
disp(diag(S2(1:6, 1:6)))

svals=diag(S2(1:nkeep,1:nkeep));  % a col vector with first nkeep singular values
svals=[svals; zeros(dE-nkeep,1)]; %pad svals with zeros
S2zerod=diag(svals);

M3T = U2*S2zerod*V2';
X3 = DEMatriciseTS(M3T);
M3 = MatriciseTS(X3,dE);

[U3,S3,V3]= svd(M3,0); %get the economy size decomp. 

disp('third singular values:')
display(diag(S3(1:6,1:6)))

svals=diag(S3(1:nkeep,1:nkeep));  % a col vector with first nkeep singular values
svals=[svals; zeros(dE-nkeep,1)]; %pad svals with zeros
S3zerod=diag(svals);

M4T = U3*S3zerod*V3';
X4 = DEMatriciseTS(M4T);
M4 = MatriciseTS(X4,dE);

[~,S4,~]= svd(M4,0); %get the economy size decomposition. 

fno=fno+1;
figure(fno);
clf('reset')

plot(diag(S)/sqrt(npt),'-ok','linewidth',mywidth)
hold on;
plot(diag(S2)/sqrt(npt),'-xr','linewidth',mywidth)
plot(diag(S3)/sqrt(npt),'-db','linewidth',mywidth)
plot(diag(S4)/sqrt(npt),'-sg','linewidth',mywidth)

% 
ax=gca; 
ax.FontSize=defFont; ax.FontName='Times';
xlabel('i','FontSize',ffac*defFont); ylabel('\sigma_i ','FontSize',ffac*defFont,'Rotation',0.0);
title('singular values')
ax.LineWidth=2.0;  ax.Box='on';

hold off;

% do lagged plots, lag is dE:

fno=fno+1;
figure(fno);
clf('reset')

subplot(4,3,3)

plot(Ys(1,istart:istop-lag+1), Ys(1,istart+lag-1:istop),'-r','linewidth',mywidth)

ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);

subplot(4,3,6)
plot(xvals(istart:istop-lag+1), xvals(1,istart+lag-1:istop),'-k','linewidth',mywidth)
 
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);

subplot(4,3,9)

plot(X2(istart:istop-lag+1), X2(istart+lag-1:istop),'-k','linewidth',mywidth)
 
% 
ax=gca; 
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);


subplot(4,3,12)
plot(X3(istart:istop-lag+1), X3(istart+lag-1:istop),'-k','linewidth',mywidth)
 
% 
ax=gca; 
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);


subplot(4,3,[1,2])

plot(tspan(istart:istop),Ys(1,istart:istop),'-r','linewidth',mywidth)
 
ax=gca;
grid off;
defFont=24;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';

xlabel(' ','FontSize',ffac*defFont); ylabel('x','Rotation',0.0,'FontSize',ffac*defFont);


subplot(4,3,[4,5])

plot(tspan(istart:istop),xvals(istart:istop),'-k','linewidth',mywidth)
 
ax=gca;
grid off;
defFont=24;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on'; ylim([-20, 20]);

xlabel(' ','FontSize',ffac*defFont); ylabel('x^0','Rotation',0.0,'FontSize',ffac*defFont);

subplot(4,3,[7,8])

plot(tspan(istart:istop),X2(istart:istop),'-k','linewidth',mywidth)
 
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
xlabel(' ','FontSize',ffac*defFont); ylabel('x^1','FontSize',ffac*defFont,'Rotation',0.0);
ax.LineWidth=2.0;  ax.Box='on';ylim([-20, 20]);

subplot(4,3,[10,11])

plot(tspan(istart:istop),X3(istart:istop),'-k','linewidth',mywidth)
 
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
xlabel('t','FontSize',ffac*defFont); ylabel('x^2','FontSize',ffac*defFont,'Rotation',0.0);
ax.LineWidth=2.0;  ax.Box='on';ylim([-20, 20]);


fno=fno+1;
figure(fno);
clf('reset')

subplot(1,3,[1,2])

plot(tspan(istart:istop),X4(istart:istop),'-k','linewidth',mywidth)
 
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
xlabel('t','FontSize',ffac*defFont); ylabel('x^3','FontSize',ffac*defFont,'Rotation',0.0);
ax.LineWidth=2.0;  ax.Box='on';ylim([-20, 20]);


subplot(1,3,3)


plot(X4(istart:istop-lag+1), X4(istart+lag-1:istop),'-k','linewidth',mywidth)
 
% 
ax=gca; 
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);

% combine all lagged plots and data beside them:

fno=fno+1;
figure(fno);
clf('reset')

subplot(5,3,3)

plot(Ys(1,istart:istop-lag+1), Ys(1,istart+lag-1:istop),'-r','linewidth',mywidth)

ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);

subplot(5,3,6)
plot(xvals(istart:istop-lag+1), xvals(1,istart+lag-1:istop),'-k','linewidth',mywidth)
 
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);

subplot(5,3,9)

plot(X2(istart:istop-lag+1), X2(istart+lag-1:istop),'-k','linewidth',mywidth)
 
% 
ax=gca; 
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);


subplot(5,3,12)
plot(X3(istart:istop-lag+1), X3(istart+lag-1:istop),'-k','linewidth',mywidth)
 
% 
ax=gca; 
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);


subplot(5,3,[1,2])

plot(tspan(istart:istop),Ys(1,istart:istop),'-r','linewidth',mywidth)
 
ax=gca;
grid off;
defFont=24;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';

xlabel(' ','FontSize',ffac*defFont); ylabel('x','Rotation',0.0,'FontSize',ffac*defFont);


subplot(5,3,[4,5])

plot(tspan(istart:istop),xvals(istart:istop),'-k','linewidth',mywidth)
 
ax=gca;
grid off;
defFont=24;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on'; ylim([-20, 20]);

xlabel(' ','FontSize',ffac*defFont); ylabel('x^0','Rotation',0.0,'FontSize',ffac*defFont);

subplot(5,3,[7,8])

plot(tspan(istart:istop),X2(istart:istop),'-k','linewidth',mywidth)
 
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
xlabel(' ','FontSize',ffac*defFont); ylabel('x^1','FontSize',ffac*defFont,'Rotation',0.0);
ax.LineWidth=2.0;  ax.Box='on';ylim([-20, 20]);

subplot(5,3,[10,11])

plot(tspan(istart:istop),X3(istart:istop),'-k','linewidth',mywidth)
 
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
xlabel('t','FontSize',ffac*defFont); ylabel('x^2','FontSize',ffac*defFont,'Rotation',0.0);
ax.LineWidth=2.0;  ax.Box='on';ylim([-20, 20]);

subplot(5,3,[13,14])

plot(tspan(istart:istop),X4(istart:istop),'-k','linewidth',mywidth)
 
ax=gca;
ax.FontSize=defFont; ax.FontName='Times';
xlabel('t','FontSize',ffac*defFont); ylabel('x^3','FontSize',ffac*defFont,'Rotation',0.0);
ax.LineWidth=2.0;  ax.Box='on';ylim([-20, 20]);


subplot(5,3,15)


plot(X4(istart:istop-lag+1), X4(istart+lag-1:istop),'-k','linewidth',mywidth)
 
% 
ax=gca; 
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';axis([-20 20 -20 20]);



function F=MyDE(~,yy)
% the Lorenz equations
    x=yy(1);
    y=yy(2);
    z=yy(3);
    f1  = sigma*(y-x);
    f2 = (r-z)*x -y;
    f3 = x*y -b*z;
    F=[f1; f2; f3];
    
 
end


    function MTS=MatriciseTS(X,dim)
        % to set up a matrix with dim cols, using the time series X
        % each row is one vector in the space with dimension dim
        % the first row is X(1) thru X(dim), the second is X(2) thru
        % X(dim+1), ... the last row is X(end-dim+1) thru X(end)
        npts=length(X);
        MTS=zeros(npts-dim+1,dim);
        kMTS=1;
        for iiMTS=1:npts-dim+1
            MTS(iiMTS,:) = X(kMTS:kMTS+dim-1);
            kMTS=kMTS+1;
        end
        %
        % ensure the column averages are zero
        
        meanMS=mean(MTS); %takes means on each col, is a row vector
        MmeanMS=repmat(meanMS,npts-dim+1,1);  %replicate means row vector so there are npts-dim+1 rows of them
        MTS = MTS-MmeanMS;  %subtract mean of each column from each col

    end

    function X=DEMatriciseTS(M)
        % Just take the first column of M and append the last row; return it as a row vector
        % using the first col means no shifting to the right:
        X = M(:,1)';  % start with the first column
       
        X = [X, M(end,2:end)]; %and tack on the last row 
    end

end