function fig1_9
%% Sharpen the cusp in the Lorenz map
% This m-file generates figure 1.9 in our Chaos book, which is of the
% cusp-shaped return map of maxima in z. The cusp is sharpened using 
% interval bisection. 
% Mark McGuinness
% 

tend=500;

sigma=10.0; r=28.0; b=8/3;  % matlab usage makes these accessible to the 
                            % included functions as global variables
    
ICs=[5 5 20];  %initial conditions

RTOL = 1E-10; ATOL = 1E-10;

options=odeset('RelTol',RTOL,'AbsTol',ATOL, 'Events',@events);

% run the equations for a while, saving events where z reaches a local
% maximum value. That is, saving x, y and z values there, in YE.

[~,~,~,YE,~]=ode45(@MyDE,[0,tend],ICs,options);

% plot the maxima in z versus previous maxima, to get a few points on the 
% cusp-shaped return map:

figure(1);

    plot(YE(1:end-1,3),YE(2:end,3),'.k','MarkerSize',20)  %x vs t
    hold on;  % hold on so points close to the cusp can be added later
    ax=gca;
    defFont=24;
    ax.FontSize=defFont; ax.FontName='Times'; ax.LineWidth=2.0;
    xlabel('Zmax(n)','FontSize',1.2*defFont); 
    ylabel('Zmax(n+1)','FontSize',1.2*defFont);

% now to sharpen the cusp:

% First, find a sequence of maxima in z, with a +ve x value followed by a +ve x value
% then a -ve x value. That is, find where the orbit flips from circling one 
% unstable spiral point to circling the other. This flip is when the cusp
% map switches from going up the left side of the cusp to hitting the right
% side. The tip of the cusp corresponds to an orbit that is very close to
% the stable manifold of origin, on the edge between going around one more
% time, and switching to circling the other spiral points.


posx = YE(:,1)>0 ;  % x values >0; posx is a column vector of logicals; 1 for positive x, 0 for negative

% find the first index of posx where we have the pattern 1 1 0 in posx:

ilocn=strfind(posx',[1 1 0]);

% do interval bisection on a line between (x,y,z) at 1st and 2nd indices, to get 
% closer to the cusp of the Zmax return map:

under=ilocn(1);  % this index is an orbit that stays around same spiral point
                 % through the next zmax value, staying to the left of the
                 % cusp
over=under+1;   % this indexed orbit goes around the other spiral point, while 
                 % reaching the next zmax value, so hits to the right of
                 % the cusp
tol = 1e-8;     % tolerance for interval bisection to stop at

aa = YE(under,:); % this is a vector with 3 elements: X, Y, Z. Z is local max
                  % and x is positive there
bb = YE(over,:);   % ditto

nsharps=400;
sharpairs=zeros(nsharps,2);  %to save pairs of sharpened zmax values for plotting

icsharp=0; % a counter for how many sharpened zmax values are saved
while and(norm(aa-bb) > tol, icsharp<nsharps) %stop when distance between aa, bb is less than tol
    icsharp = icsharp+1;  % increment counter
    % find a point halfway between aa and bb:
 
    cc=0.5*(aa+bb); % this is a row vector, coordinates of a point in (x,y,z) space that is 
                    % halfway between aa & bb. Use it as a new initial value:
                 
    % run forward a little bit to get past the current max of z:
    options2=odeset('RelTol',RTOL,'AbsTol',ATOL);  % no catching maxima in z
                                                     % for this bit
    tend= 0.1;  %time of order one gives a full circle at current param values
    [~,Yshort]=ode45(@MyDE,[0,tend],cc,options2);
    ICshort=Yshort(end,:);  %new initial conditions, a little further on than cc

    % now look for the next few maxima in z:
    options3=odeset('RelTol',RTOL,'AbsTol',ATOL, 'Events',@events);
    tend= 10;
    [~,~,~,YE,~]=ode45(@MyDE,[0,tend],ICshort,options3);
    
    sharpairs(icsharp,:) = [cc(3) YE(1,3)];  %[Zmax(n) Zmax(n+1)]
    %save the next two maxima, too, especially to get lower legs of cusp
    icsharp=icsharp+1;
    sharpairs(icsharp,:) = [YE(1,3) YE(2,3)];
    icsharp=icsharp+1;
    sharpairs(icsharp,:) = [YE(2,3) YE(3,3)];
 
    % check this Zmax event for the sign of x at next zmax. If positive x, set aa to
    % cc. Otherwise, set bb to cc. Then go again. This is the bisection process.
 
    if YE(1,1) >0
        aa=cc;
    else
        bb=cc;
    end
 
end

% now add the sharpened maxima in z points to our plot, coloured red:

    plot(sharpairs(1:icsharp,1),sharpairs(1:icsharp,2),'.r','MarkerSize',20)
    
 %   plot([25 50], [25 50], '-k')  % add a 45 degree line for comparison
 %   (commented out)
    hold off;

% now for the functions called by this file:

    function F = MyDE(~,yy)
        % the Lorenz equations (well, the right-hand sides)
        x=yy(1);
        y=yy(2);
        z=yy(3);

        f1  = sigma*(y-x);  %sigma, r, b are passed in as a default globals
        f2 = (r-z)*x -y;
        f3 = x*y -b*z;
        F=[f1; f2; f3];
    end

% the events function finds where z reaches a local maximum, but does not stop:

    function [value,isterminal,direction]=events(~,yy)
        isterminal=0; %don't stop; just zero in to record the event, then move on
        direction=-1; %derivative of z decreasing through zero is wanted for local max
        x=yy(1);
        y=yy(2);
        z=yy(3);

        value=x*y -b*z; % Matlab seeks a value where this product is zero. It is dz/dt. 
                        % For a maximum, this needs to be a decreasing
                        % function, so direction is -1
    end


end