function fig4_4
%% Generates Fig. 4.4, showing the action for the Lorenz equations, versus r
% to generate figure 4.4 in our chaos book 2016. Shoot for an unstable
% periodic orbit, for various r values ranging from 24.74 down to 13.926.
% the first value is where the spiral fixed pts lose stability and gives
% birth to a periodic orbit. The second value
% is near where the homoclinic orbits on origin come into existence. 
% The action is period times amplitude of the unstable periodic orbit,  
% plotted against r.

% shoot using Zmax values. Takes 20 minutes to do the 68 r values that
% generate Fig 4.4
%
% Mark McGuinness


tend=20;
nt=301;
times=linspace(0,tend,nt);  %use these times to get output at

options=odeset('RelTol',1e-13,'AbsTol',1e-13, 'Events',@events);
options2=odeset('RelTol',1e-13,'AbsTol',1e-13);  % no events

mywidth=3;

sigma=10; b=8/3;  %r is the bifurcation parameter

nstraddle=10; % # of times to try to straddle unstable orbit
nbisect=60;  tol=1.0e-015; % for the bisection bit
rcrit=24.74;

sxvals=zeros(nt,nbisect); szvals=sxvals;

rmax=24.738;
rmin=13.927;
finer=0.1;
r1=(1+finer)*rmin; r2 = (1-finer)*rmax;
rvecm=linspace(r1,r2,10);
rvecs=linspace(rmin,r1,20);
rvece=linspace(r2,rmax,40);
rvec = [rvecs(1:end-1) rvecm rvece(2:end)];
display(rvec) 

periods=zeros(length(rvec), 1);
amplitudes=periods; actions=periods;
ntimes=400;
trajectories = zeros(ntimes,3,length(rvec)); % each instance is one r. 
                                        % ntimes time points on each trajectory
                                        % 3 components x,y,z saved


for JJ=1:length(rvec)
    
 r = rvec(JJ);   

 xfx=sqrt(b*(r-1));
 xf=[xfx, xfx, r-1]; % the positive spiral fp


% start up a bit (in z) from the spiral point:
abit=sqrt(rcrit-r)/10;

ICs=[xfx,xfx,r-1+abit];   % we will shoot on Zmax to get unstable  orbit; we run 
                     % an initial condition near the stable spiral point in
                     % the positive quadrant. Run until there are two
                     % maxima in z, and use these to determine if
                     % spiralling in or out from fp.
                   
%     look to straddle the unstable periodic orbit. decreasing vs
%     increasing zmax is one criterion; close to xf we expect to have decreasing zmax;
%     further away on the other side of the unstable orbit, increasing zmax;
%     also, if the x value corresponding to a zmax
%     is negative, we are too far away from xf

% look first to be close enough to approach xf:

success=false; % to say if succeeded in getting inside basin of xf or not

    for II=1:nstraddle
       % now look for next few maxima in z:

        [~,Ysol,~,YE,~]=ode45(@LorenzDE,times,ICs,options);

        maxima= YE;  % the (x,y,z) coords of first three maxima in z

        xvals=Ysol(:,1); yvals=Ysol(:,2); zvals=Ysol(:,3);
        
        if and(maxima(2,3) < maxima(1,3), maxima(2,1)>0) % then zmax values are decreasing OK
            success=true;  %we are inside the basin of xf
            break
        else    % reduce distance to xf to get inside its basin
            ICnew = 0.5*(ICs + xf); %move halfway towards xf and try again
            ICs=ICnew;
        end          
        
    end
    
    if ~success
        DISP('failed to get inside basin of xf, bailing out')
        disp(maxima)
        figure(10)
        plot3(xvals,yvals,zvals,'-k','linewidth',mywidth)
        figure(11)
        plot(times,zvals)
        return
    end
    
  % reach here if found an orbit in basin of xf. Now find one outside this basin
  
  ICinside=ICs; %save start point inside basin of xf
  
  ICs=[xfx,xfx,r-1+ 2*(ICinside(3)-r+1)]; % start with double the z distance from r-1
    
  success=false; % to say if succeeded in getting outside basin of xf or not
 
    for II=1:nstraddle

       % now look for next few maxima in z:

        [~,Ysol,~,YE,~]=ode45(@LorenzDE,times,ICs,options);

        maxima= YE;  % the (x,y,z) coords of first three maxima in z

        xvals=Ysol(:,1); yvals=Ysol(:,2); zvals=Ysol(:,3);
       
        if maxima(2,3) > maxima(1,3) || maxima(2,1)<0 || maxima(1,1)<0 % then zmax values are increasing or we are jumping to other leaf
            success=true;  %we are outside the basin of xf
           
            break
        else    % increase distance to xf to get outside its basin
            ICnew = xf + 2.0*(ICs-xf); %move twice the present distance away from xf and try again
            ICs=ICnew;
        end          
    end
    
        if ~success
            disp('failed to get outside basin of xf, bailing out')
            disp(maxima)
            figure(10)
            plot3(xvals,yvals,zvals,'-k','linewidth',mywidth)
            return
        end

    
    % arrival here means have bracketed the unstable limit cycle
    
    ICoutside=ICs;
    
    % do interval bisection on ICinside, ICoutside to get close to limit
    % cycle
    

    success=false;
    for II=1:nbisect
       ICs = (ICinside + ICoutside)/2.0;

       [~,Ysol,TE,YE,~]=ode45(@LorenzDE,times,ICs,options);

        maxima= YE;  % the (x,y,z) coords of first three maxima in z
 
        xvals=Ysol(:,1); yvals=Ysol(:,2); zvals=Ysol(:,3);
        sxvals(:,II)=xvals; szvals(:,II)= zvals;
    
        if maxima(2,3) > maxima(1,3) || maxima(2,1)<0 || maxima(1,1)<0 
            % then zmax values are increasing or we are jumping to other leaf
            % we are still outside the limit cycle which is the 
            % boundary of the basin of xf
           
            ICoutside=ICs;
            inside=false;
        else    % we are inside the basin
    
            ICinside=ICs;
            inside=true;
        end          
        
        if  norm(ICoutside-ICinside) < tol*norm(xf) % converged OK
            
            %disp('SUCCESS in bisection')
            success=true;
            % but watch out for a final run that flips to the negative
            % quadrant. Simply use the ICinside run:
            if ~inside  %not inside on last run, so set up for the final run:
                ICs=ICinside;
                [~,~,TE,YE,~]=ode45(@LorenzDE,times,ICs,options); %get TE and YE
                %disp('not inside so fixing that')

            end
            
            
            break
        end
        
    end
    
    if ~success
        disp('failed to get close to unstable LC, bailing out')
        disp(maxima)
        figure(10)
        plot3(xvals,yvals,zvals,'-k','linewidth',mywidth)
        return
    end
    
    % get amplitude and period of unstable limit cycle
    
    % xvals, yvals and zvals are column vectors
    % we need to run from one zmax to the next one, for the converged LC:
    
    ICs=YE(1,:); %start at first zmax found; run to the next one
    
    period=TE(2) - TE(1);
    times2=linspace(0, period,ntimes);

    [~,Ysol]=ode45(@LorenzDE,times2,ICs,options2);
    xvals=Ysol(:,1); yvals=Ysol(:,2); zvals=Ysol(:,3);
    
    trajectory=[xvals, yvals, zvals];  % this should be a whole loop around
    
    bxf=repmat(xf,length(times2),1); %set up xf for every row of trajectory (every time)
                                      
    amplitude=max(max(abs(trajectory-bxf))); % this gets the actual amplitude though    

    action=amplitude*period;
    
    actions(JJ) = action;
    amplitudes(JJ)= amplitude;
    periods(JJ) = period;
    trajectories(:,:,JJ) = trajectory; % save the trajectory for plotting, too
    
end % of for JJ loop on r values  
    
%================ first plot, period vs r ===============

fno=1;
figure(fno);
clf('reset')

plot(rvec,periods,'-r','linewidth',mywidth)

ax=gca;
defFont=36;
ax.FontSize=defFont; ax.FontName='Times';
ax.XTickMode='manual';  
ax.YTickMode='manual';
ax.LineWidth=2.0;
xlabel('r','FontSize',1.3*defFont); ylabel('period','FontSize',1.3*defFont,'Rotation',0.0);



%================ second plot, action vs r ===============

fno=fno+1;

figure(fno);
clf('reset')

plot(rvec,actions,'-r','linewidth',mywidth)  % this is the actual fig 4.4

ax=gca;
defFont=36;
ax.FontSize=defFont; ax.FontName='Times';
ax.XTickMode='manual';  
ax.YTickMode='manual';
ax.LineWidth=2.0;
xlabel('r','FontSize',1.3*defFont); ylabel('action','FontSize',1.3*defFont,'Rotation',0.0);

%================ third plot, amplitudes vs r ===============


fno=fno+1;

figure(fno);
clf('reset')

plot(rvec,amplitudes,'-r','linewidth',mywidth) 

ax=gca;
defFont=36;
ax.FontSize=defFont; ax.FontName='Times';
ax.XTickMode='manual';  
ax.YTickMode='manual';
ax.LineWidth=2.0;
xlabel('r','FontSize',1.3*defFont); ylabel('amplitude','FontSize',1.3*defFont); 

%=== plot unstable periodic orbits found, and the spiral point too ====

fno=fno+1;
figure(fno);
clf('reset')

r = rvec(1);  % plot the first one, then add the rest
spoint=[sqrt(b*(r-1)), sqrt(b*(r-1)), r-1];
plot3(spoint(1), spoint(2), spoint(3),'-or','MarkerSize',5)
hold on;
plot3(trajectories(:,1,1),trajectories(:,2,1),trajectories(:,3,1), '-r','linewidth',mywidth)

for JJ=2:length(rvec)
r = rvec(JJ);  % plot the first one, then add the rest
spoint=[sqrt(b*(r-1)), sqrt(b*(r-1)), r-1];
plot3(spoint(1), spoint(2), spoint(3),'-or','MarkerSize',5)
plot3(trajectories(:,1,JJ),trajectories(:,2,JJ),trajectories(:,3,JJ), '-r','linewidth',mywidth)
end

ax=gca;
defFont=36;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;
xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont);
zlabel('z','FontSize',1.3*defFont);
hold off;


%=== plot unstable periodic orbits found, and the spiral point too, y vs x only ====

fno=fno+1;
figure(fno);
clf('reset')

r = rvec(1);  % plot the first one, then add the rest
spoint=[sqrt(b*(r-1)), sqrt(b*(r-1)), r-1];
plot(spoint(1), spoint(2),'-ok','MarkerSize',2,'MarkerFaceColor','black')
hold on;
plot(trajectories(:,1,1),trajectories(:,2,1), '-r','linewidth',2)

for JJ=2:length(rvec)
r = rvec(JJ);  % plot the first one, then add the rest
spoint=[sqrt(b*(r-1)), sqrt(b*(r-1)), r-1];
plot(spoint(1), spoint(2),'-ok','MarkerSize',2,'MarkerFaceColor','black')
plot(trajectories(:,1,JJ),trajectories(:,2,JJ), '-r','linewidth',2)
end

ax=gca;
defFont=36;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;
xlabel('x','FontSize',1.3*defFont); ylabel('y','FontSize',1.3*defFont);
hold off;

function F=LorenzDE(~,yy)
    
   % Lorenz equations
    x=yy(1);
    y=yy(2);
    z=yy(3);
    
    f1  = sigma*(y-x);
    f2 = (r-z)*x -y;
    f3 = x*y -b*z;
    F=[f1; f2; f3];

end

   function [value,isterminal,direction]=events(~,yy)
        isterminal=0; %don't stop
        direction=-1; %decreasing through zero is wanted for local max
        x=yy(1);
        y=yy(2);
        z=yy(3);

        value=x*y -b*z; % detects if this product is zero. It is dz/dt. 
                        % For a maximum, this needs to be a decreasing
                        % function, so direction is -1
    end

end