function fig5_17
%% Plot Poincare sections for the Henon-Heiles system, for various energy values
% Generates figure 5.17 in our chaos book 2018. Slices through the time 
% evolution of a hamiltonian system. Show p2 vs q2 when q1=0 and p1>0
% This is to put all 8 egg-shaped plots on one figure with the same axis scales
% for all, so can see size changing as H changes. Run time is about 8
% minutes.

% Mark McGuinness

options=odeset('RelTol',1e-10,'AbsTol',1e-10, 'Events',@events);

H0array=[1/24, 1/16, 1/12, 7/64, 1/8, 1/7, 7/48, 1/6]; %the hamiltonian energy values desired

% Initial conditions have to be tailored to each energy value. These are
% copied from the original individual plots m-files. All hand-chosen for a
% good-looking section

% Set up the case H0=1/24:
    H0=H0array(1); %the hamiltonian energy value desired

    p2v=[0.0, -0.12,0.12,-0.1907,-0.0925,0.0925,-0.09225, 0.09225,-0.99*sqrt(2*H0)] ; % run a set of values of p2 for q2 zero    
    q2v=[-0.26 -0.2, -0.175,-0.15, 0.0, 0.04, 0.08, 0.11, 0.15, 0.2]; % and a set of values of q2 for p2 zero
    ICs = makeICS(p2v, q2v, H0);
    ICcell{1} = ICs; 
    ps{1} = cell(length(ICs(:,1)),1);  % a cell array, each item to hold the array which is the poincare sections for each
                                   % initial condition, ie values at each time (row)

% Set up H0=1/16:
    H0=H0array(2);
    extr=sqrt(2*H0);

    p2v=[0.0, -extr/3,3*extr/7,0.9*3*extr/7,-0.99*extr,0.99*extr] ; % run a set of values of p2 for q2 zero

    q2v=[0.0, -5*extr/7, -9*extr/14, -4*extr/7,-extr/2, -0.98*extr/3, ...
         -0.145,-0.089, -0.086,-0.04,extr/2,extr/3, 0.99*extr]; % and a set of values of q2 for p2 zero

    ICs = makeICS(p2v, q2v, H0);
    ICcell{2} = ICs; 

    ps{2} = cell(length(ICs(:,1)),1);  % a cell array, each item to hold the array which is the poincare sections for each
                                 % initial condition, ie values at each time (row)

% Set up H0=1/12:
    H0=H0array(3);

    extr=sqrt(2*H0);

    p2v=[0.0, -extr/3,3*extr/7,0.9*3*extr/7,-0.99*extr,0.99*extr] ; % run a set of values of p2 for q2 zero
    q2v=[0.0, -5*extr/7, -9*extr/14, -4*extr/7,-extr/2, -0.98*extr/3, ...
      -0.124,-0.123, -0.105,-extr/6,-0.035,extr/2,extr/3, 0.99*extr,0.4923]; % and a set of values of q2 for p2 zero
    ICs = makeICS(p2v, q2v, H0);
    ICcell{3} = ICs; 

    ps{3} = cell(length(ICs(:,1)),1);  % a cell array, each item to hold the array which is the poincare sections for each

% Set up H0=7/64:
    H0=H0array(4);

    extr=sqrt(2*H0);

    p2v=[0.0, -extr/3,3*extr/7,0.9*3*extr/7,-0.99*extr,0.99*extr] ; % run a set of values of p2 for q2 zero

    q2v=[0.0, -5*extr/7,-0.3, -0.32, -0.29, -0.26,-0.27, -extr/2, -0.98*extr/3,  -0.93*extr/3,...
      -extr/6,-1.3*extr/6, -0.035,extr/2,extr/3, 0.99*extr]; % and a set of values of q2 for p2 zero
    ICs = makeICS(p2v, q2v, H0);
    ICcell{4} = ICs; 

    ps{4} = cell(length(ICs(:,1)),1);  % a cell array, each item to hold the array which is the poincare sections for each

% Set up H0=1/8:
    H0=H0array(5);

    extr=sqrt(2*H0);
    
    p2v=[0.0, -extr/3,3*extr/7,0.9*3*extr/7,-0.99*extr,0.99*extr] ; % run a set of values of p2 for q2 zero

    q2v=[0.0, -5*extr/7,-0.3214, -0.2857, -extr/2, -0.98*extr/3, ...
      -extr/6,-0.035,extr/2,extr/3, 0.99*extr]; % and a set of values of q2 for p2 zero

    ICs = makeICS(p2v, q2v, H0);
    ICcell{5} = ICs; 

    ps{5} = cell(length(ICs(:,1)),1);  % a cell array, each item to hold the array which is the poincare sections for each

% Set up H0=1/7:
    H0=H0array(6);

    extr=sqrt(2*H0);  
    

    p2v=[0.0, -extr/3,3*extr/7,0.9*3*extr/7,-0.99*extr,0.99*extr] ; % run a set of values of p2 for q2 zero

    q2v=[0.0, -5*extr/7,-0.3214, -0.2857, -extr/2, -0.98*extr/3, ...
      -extr/6,-0.035,extr/2,extr/3, 0.99*extr, 1.1*extr, 1.2*extr, 1.3*extr,...
      1.4*extr, 1.425*extr]; % and a set of values of q2 for p2 zero

    ICs = makeICS(p2v, q2v, H0);
 % add one extra row at the bottom, to catch the elusive periodic orbits:
 
    p1Radd= HenonHeiles(0,0.06,3*extr/7,H0); %returns p1arr with shape (length(p2R), 1)

    botrow=[0, 0.06, p1Radd, 3*extr/7];  %this is the extra row I want at the end
    
    ICcell{6} = [ICs; botrow]; 

    ps{6} = cell(length(ICs(:,1)),1);  % a cell array, each item to hold the array which is the poincare sections for each


% Set up H0=7/48:
    H0=H0array(7);

    extr=sqrt(2*H0);  

    p2v=[0.0, -extr/3,3*extr/7,0.9*3*extr/7,-0.99*extr,0.99*extr] ; % run a set of values of p2 for q2 zero

    q2v=[0.0, -5*extr/7,-extr/2, -0.98*extr/3, ...
      -extr/6,-0.035,extr/2,extr/3, 0.99*extr]; % and a set of values of q2 for p2 zero

    ICs = makeICS(p2v, q2v, H0);
    ICcell{7} = ICs; 

    ps{7} = cell(length(ICs(:,1)),1);  % a cell array, each item to hold the array which is the poincare sections for each

% Set up H0=1/6:
    H0=H0array(8);

    extr=sqrt(2*H0);  

    p2v=[0.0, -extr/3,3*extr/7,0.9*3*extr/7,-0.99*extr,0.99*extr] ; % run a set of values of p2 for q2 zero

    q2v=[0.0, -5*extr/7, -9*extr/14, -4*extr/7,-extr/2,-extr/3, -0.98*extr/3, ...
     -0.9*extr/3, -extr/6,-0.035,extr/2,extr/3, 0.99*extr, 0.77, 0.96, 0.98, 0.998]; % and a set of values of q2 for p2 zero

     ICs = makeICS(p2v, q2v, H0);
     
 % add one extra row at the bottom, to catch the elusive periodic orbits:
 
    p1Radd= HenonHeiles(0,-0.3,0.365,H0); %returns p1arr with shape (length(p2R), 1)

    botrow=[0, -0.3, p1Radd, 0.365];  %this is the extra row I want at the end
    
    ICcell{8} = [ICs; botrow]; 

    ps{8} = cell(length(ICs(:,1)),1);  % a cell array, each item to hold the array which is the poincare sections for each

nbanked=0;%number of runs with acceptable data saved; no need to recompute these, starting at number 1.

NJJ=8; %number of runs to compute, starting with nbanked+1:

for JJ= nbanked+1:nbanked+NJJ  % will use 1:8 in the end
    
    % set up ICs. 
    ICs=ICcell{JJ};
    %display(JJ)    
    
    tend=15000;
      
    for II=1:length(ICs(:,1))

        IC=ICs(II,:);

        [~,~,~,YE,~]=ode45(@HHDE,[0, tend],IC,options);

        if ~isempty(YE)        
           ps{JJ}{II}=[YE(:,3), YE(:,2), YE(:,4)];  % this is p1, q2, p2 for this set of IC values. One row for each time; three columns
        end

    end
end
    
    % do the plotting after all of the computations are done
    
    figure(1)  %poincare sections for q2, p2
    clf('reset')
    
for JJ=1:NJJ+nbanked    %cycle through the computed H0 values
    %disp(JJ)
  
    subplot(4,2,JJ)
    hold 'on';
    nII=length(ps{JJ});
        
    for II=1:nII  %cycle through initial conditions
      
      if isempty (ps{JJ}{II})
          continue
      end
      
      x=ps{JJ}{II}(:,2);  % this is q2
      y=ps{JJ}{II}(:,3);  % this is p2
      ipick=ps{JJ}{II}(:,1)>0; %get indices of points with positive p1
      xp=x(ipick);
      yp=y(ipick);

      ns=5; %number to skip while plotting
      if JJ == 4 || JJ==5 || JJ==6
          ns=3;
      elseif JJ==3 
          ns=4;
      elseif JJ==8
          ns=2;
      elseif JJ==7
          ns=2;
      end
      plot(xp(1:ns:end),yp(1:ns:end), '.r','MarkerSize',4) 

      plot(xp(1:ns:end),-yp(1:ns:end), '.r','MarkerSize',4) 

    end

        axis([-0.50, 1.0,-0.58,0.58]);  %I want this to apply to every subplot

        ax=gca;

        grid off;
    
        ax.XTickMode='manual';  
        ax.YTickMode='manual';
        ax.LineWidth=0.5;  ax.Box='off';
        xlabel(' '); ylabel(' ');

      hold off;

end



function F=HHDE(~,yy)
    
   % Henon-Heiles hamiltonian equations. Constant H built-in:
   
    q1=yy(1);
    q2=yy(2);
    p1=yy(3);
    p2local=yy(4);
    
    
    %p1 needs to have a real solution
    % Hamval=H0;
  
    f1  = p1;  %q1 dot
    f2 = p2local;  %q2 dot
    f3= -q1 -2*q1*q2;  % p1 dot
    f4 = -q2-q1*q1+q2*q2; % p2 dot
    F=[f1; f2; f3; f4];

end

   function [value,isterminal,direction]=events(~,yy)
        isterminal=0; % don't stop
        direction=0; % any direction is fine
        
        value=yy(1); % detects if q1 is zero in value.
    end


    function P1 = HenonHeiles(q1,q2,p2,Energy)
    % Henon-Heiles hamiltonian: to solve the expression Hamiltonian=Energy for p1
    % given q1, q2 and p2 which can be scalars or (row or column) vectors;they must be
    % the same length though
    %  P1 has size = (length(q1),1)
                
        q1arr=reshape(q1,1,[]); %ensure this is a row vector. Its length is of course that of q1
        q1sqarr=q1arr.*q1arr; % q1 squared, with length q2 rows and length q1 cols
        
        q2arr=reshape(q2,[],1); %ensure this is a column vector. Its length is of course that of q2
        q2sqarr=q2arr.*q2arr;
        
        p2arr=reshape(p2,[],1); %ensure this is a column vector. Its length is of course that of q2


                 
        sq=2*Energy-(p2arr.*p2arr + q1sqarr + q2sqarr) - 2*q1sqarr.* q2arr + 2* q2arr.*q2sqarr/3;

         
        P1=sqrt(sq);  %complex valued for negative values of sq. Zero these:
        
        indexarr= sq<0;
        P1(indexarr) = -999; %set complex values to -999, in P1  
        
    end

    function ICs = makeICS(p2v, q2v,H0v)
        % assemble the ICs for these p2 and q2 vectors:
        p2R = reshape(p2v,[],1);  %make it a column
        q2zero=zeros(length(p2R), 1);  % a column of zeros to go beside p2R
        q2R=reshape(q2v,[],1);   % make it a column
        p2zero=zeros(length(q2R),1); % a col of zeros for p2 to go next to it

        p2R=[p2R; p2zero];
        q2R=[q2zero; q2R];

        q1R=zeros(length(q2R),1); % zero initial values for q1. one column

        p1R= HenonHeiles(0,q2R,p2R,H0v); %returns p1arr with shape (length(p2R), 1)

        ICs=[q1R, q2R, p1R, p2R];

        %filter out those ICs with complex p1 values, flagged with -999. Just
        %in case.

        ikeep= p1R > -990;  % this is true for good values of p1R
        ICs = ICs(ikeep,:); %keeps only the good rows

    end


end