function fig5_3
%% Poincare section for a Hamiltonian system with Henon-Heiles potential
% This generates figure 5.3 in our book Chaos. A slice through the time 
% evolution of a hamiltonian system. Takes about a minute to run.
% Mark McGuinness

tend=10000;

options=odeset('RelTol',1e-10,'AbsTol',1e-10, 'Events',@events);

defFont=24;

H0=1/24; %the hamiltonian energy value desired
 
% now get some ICs that lie in the valid region for p1 to have a solution:
 
 
p2v=[0.0, -0.12,0.12,-0.1907,-0.0925,0.0925,-0.09225, 0.09225,-0.99/sqrt(12)] ; % run a set of values of p2 for q2 zero
p2R = reshape(p2v,[],1);  %make it a column
q2zero=zeros(length(p2R), 1);  % a column of zeros to go beside p2R

q2v=[-0.26 -0.2, -0.175,-0.15, 0.0, 0.04, 0.08, 0.11, 0.15, 0.2]; % and a set of values of q2 for p2 zero
q2R=reshape(q2v,[],1);   % make it a column
p2zero=zeros(length(q2R),1); % a col of zeros for p2 to go next to it

p2R=[p2R; p2zero; 0.223];
q2R=[q2zero; q2R; 0.1];
q1R=zeros(length(q2R),1); % zero initial values for q1. one column

p1R= HenonHeiles(0,q2R,p2R,H0); % to get p1 values consistent with the 
                                % Hamiltonian having the value H0. 
                                % returns p1arr with shape (length(p2R), 1)

ICs=[q1R, q2R, p1R, p2R];
format long
%display(ICs)


ps = cell(length(q1R));  % a cell array, each item to hold the array which 
                         % is the poincare sections for each
                         % initial condition, ie values at each time (row)

for II=1:length(q1R)
     
   IC=ICs(II,:);     
   [~,~,~,YE,~]=ode45(@HHDE,[0, tend],IC,options);
   
    if ~isempty(YE)        
       ps{II}=[YE(:,3), YE(:,2), YE(:,4)];  % this is p1, q2, p2. One row for each time; three columns
    else
       disp('empty YE')
    end  
 end

% count how many poincare section intersections there are:

npoints = 0;
  
figure(1)  %poincare section for q1, p1
clf('reset')

hold on;
  
for II=1:length(q1R)
  % icol=mod(II,ncols)+1;
  if isempty (ps{II})
      continue
  end
  x=ps{II}(:,2);  % this is q2
  y=ps{II}(:,3);  % this is p2
  ipick=ps{II}(:,1)>0; %get indices of points with positive p1
  xp=x(ipick);
  yp=y(ipick);

  ns=5; %number to skip while plotting
  
  npoints=npoints+ 2* length(xp)/ns;
  
  plot(xp(1:ns:end),yp(1:ns:end), '.r','MarkerSize',6) 

  plot(xp(1:ns:end),-yp(1:ns:end), '.r','MarkerSize',6) % leverage the symmetry for more points

end
 
    ax=gca;
    grid off;
    ax.FontSize=defFont; ax.FontName='Times';
    ax.XTickMode='manual';  
    ax.YTickMode='manual';
    ax.ZTickMode='manual';
    ax.LineWidth=2.0;  ax.Box='off';
    ax.YAxisLocation = 'origin';
    ax.XAxisLocation = 'origin';

    xlabel('q_2','FontSize',1.3*defFont); ylabel('p_2','FontSize',1.3*defFont,'Rotation',0.0);
 
  hold off;

function F=HHDE(~,yy)
   % This provides the right-hand sides of the set of first order 
   % Henon-Heiles hamiltonian equations. Constant H is a consequence:
   
    q1=yy(1);
    q2=yy(2);
    p1=yy(3);
    p2local=yy(4);
  
    f1  = p1;  %q1 dot
    f2 = p2local;  %q2 dot
    f3= -q1 -2*q1*q2;  % p1 dot
    f4 = -q2-q1*q1+q2*q2; % p2 dot
    F=[f1; f2; f3; f4];

end

   function [value,isterminal,direction]=events(~,yy)
        isterminal=0; % don't stop
        direction=0; % any direction is fine
        
        value=yy(1); % detects if q1 is zero in value. 
    end


    function P1 = HenonHeiles(q1,q2,p2,Energy)
    % Henon-Heiles hamiltonian: to solve the expression Hamiltonian=Energy for p1
    % q1, q2 and p2 can be scalars or (row or column) vectors;they must be
    % the same length though
    %  P1 has size = (length(q1),1)
                
        q1arr=reshape(q1,1,[]); %ensure this is a row vector. Its length is of course that of q1
        q1sqarr=q1arr.*q1arr; % q1 squared, with length q2 rows and length q1 cols
        
        q2arr=reshape(q2,[],1); %ensure this is a column vector. Its length is of course that of q2
        q2sqarr=q2arr.*q2arr;
        
        p2arr=reshape(p2,[],1); %ensure this is a column vector. Its length is of course that of q2


                 
        sq=2*Energy-(p2arr.*p2arr + q1sqarr + q2sqarr) - 2*q1sqarr.* q2arr + 2* q2arr.*q2sqarr/3;

         
        P1=sqrt(sq);  %complex valued for negative values of sq. Zero these:
        
        indexarr= sq<0;
        P1(indexarr) = 0.0; %set complex values to zero, in P1  
        
    end
end