function fig6_10
%% To compute the Julia set for the complex quadratic map. Uses Modified Inverse iteration.
% To compute the Julia set for our fig 6.10. The map is z^2-mu, with complex z 
% and mu = 0.999 - 0.251i. Find it by the modified inverse iteration method:
% Considering the iterated functions f^{-1} _+ and f^{-1} _-,  the Julia set
% is the invariant set of f and of f^{-1}
% This suffers from a doubling of the number of points with each iteration.
% One way to deal with this proliferation of points is to randomly (uniformly)
% choose one of the two inverse maps and
% never truncate the points. This does quite a good job. 

% But Adam Majewski's code for maxima (fraktal.republika.pl) does a better job. 
% This modified inverse iteration method (MIIM) starts near 
% an unstable fixed point, not
% from a grid of points.  BOTH inverses are saved (pushed) to a stack which is
% iterated from, and when a point is iterated it is removed from the stack (popped).
% MIIM counts hits on pixels, and only saves a point for plotting if the hit count is
% zero and it has just been hit. It only saves (the accurate value) to the stack 
% if the hit count is
% less than some maximum value. Hence after an initial rapid growth, 
% eventually the stack is emptied and
% iteration stops. Saved points are fully accurate for plotting. This
% process seems to avoid the known issues with the doubling of the number
% of inverse points each iteration. I have written Majewski's algorithm in
% matlab code here.

% Mark McGuinness

% initial grid to set the pixel resolution on, WIDTHxHEIGHT. 2000x1000 takes 6 minutes to
% run, if MaxHits is set to 20. The stack gets as large as nearly 128,000
% points, and the number of points plotted is about 74,500.

tic

WIDTH = 2000; %number of pixels in x direction ("# of pixels")
HEIGHT = 1000; %number of points in y direction to be ouput
npmax=1e07; %max number of points to plot
MaxHits=20;  %max number of hits allowed on a pixel

% x and y ranges
Ymin = -0.9;
Ymax = 0.9;
Xmin = -1.8;
Xmax = 1.8;

PixW = (Xmax-Xmin)/WIDTH;  %the width of one pixel 
PixH = (Ymax-Ymin)/HEIGHT;  %the height of one pixel

mu=0.999-0.251i; %pretty cool-looking version, saved to MIIM7. 
% This is the one going in the book

%find fixed points and their stability
discr=1+4*mu;
x1=(1+sqrt(discr))/2;
x2=(1-sqrt(discr))/2;
lam1=2*x1;

disp('fixed point 1 is')
disp(x1)
disp('with eigenvalue')
disp(lam1)
disp('and its conjugate is the other eigenvalue. Magnitude is')
disp(abs(lam1))
disp('fixed point 2 is')
display(x2)
disp('with eigenvalue')
lam2=2*x2;
disp(lam2)
disp('and its conjugate is the other eigenvalue. Magnitude is')
disp(abs(lam2))

% find a period-two orbit and see if it is a saddle:

%ff=x^4-2*mu*x^2+mu*mu-mu;  %f(f(x)), which is represented by the vector:
ffp=[1 0 -2*mu -1 mu*mu-mu];
rootsff=roots(ffp);
display(rootsff)

%eliminate the fixed points from this set:
fp = [1 0 -mu]; % represent the polynomial z^2-mu in matlab format
elim=abs(rootsff - polyval(fp,rootsff)); % calculate the difference between 
                                         % f(rootsff) and rootsff to find fixed points
chooz=elim > 1e-14;                                        
disp('two-cycle is:')
twocyc=rootsff(chooz);
disp(twocyc)

%the jacobean matrix is the product of two for f itself:
Jac1 = 2*[real(twocyc(1)),  -imag(twocyc(1)); imag(twocyc(1)), real(twocyc(1))];  
Jac2 = 2*[real(twocyc(2)),  -imag(twocyc(2)); imag(twocyc(2)), real(twocyc(2))];

disp(Jac1); display(Jac2);

Jacff=Jac1*Jac2;

disp('eigenvalues of 2-cycle are')
disp(eig(Jacff))
disp('with magnitudes')
disp(abs(eig(Jacff)))
disp('with 1 subtracted:')
display(abs(eig(Jacff))-1)

% use the unstable fixed point to iterate on, not the 2-cycle
% (starting from the 2-cycle did not look any different really)

Xp = zeros(1,npmax); Yp=zeros(1,npmax); %to hold accurate values of x, y to be plotted as dots
Hits=zeros(WIDTH+1,HEIGHT+1);  % to count the number of hits on each pixel
MyStack=zeros(npmax,1); % to hold the accurate values of z iterated backwards onto
MyStack(1:end-1)=[];  %the above line was just to preallocate space for MyStack; now empty it

%x1 was the most unstable fixed point; try starting from it:
            
MyStack(1)=x1;  %push x1 onto the stack
Xp(1)=real(x1); Yp(1)=imag(x1); % save the x, y coords of x1, our starting 
                                % point, which is in the Julia set since it
                                % is an unstable fixed point
%locate x1's index on an imagined pixel map, to pixel resolution
ix=1+ceil((real(x1)-Xmin)/PixW); iy=1+ceil((imag(x1)-Ymin)/PixH);
Hits(ix,iy) = 1;

ipcount=1; %count the number of points saved for plotting
maxstacksz=1;
for itcount=1:npmax %limit the max number of iterations to npmax
    %pop a z value from the stack:
    z=MyStack(1); MyStack(1)=[];
    
    %iterate it backwards, positive branch:
    zp = Finv(z, mu);
    %locate zp's index 
    ix=1+ceil((real(zp)-Xmin)/PixW); iy=1+ceil((imag(zp)-Ymin)/PixH);
    
    %check the hit count for this pixel:
    if Hits(ix,iy) < MaxHits
        %push the new z onto the stack:
        MyStack = [zp, MyStack]; %#ok<AGROW>
        
        if Hits(ix,iy) <1  %this is the first time hit this pixel, save for plotting
            ipcount = ipcount+1;
            if ipcount > npmax %watch out for exceeding array size before saving
                disp('jumping out of main loop, ipcount too big')
                break
            end
            Xp(ipcount)=real(zp); Yp(ipcount)=imag(zp);
        end
       
        %add one to the hit count for this pixel:
        Hits(ix,iy) = Hits(ix,iy)+1;
    end
    
    %repeat the above section for the negative branch:
    zp=-zp;
    ix=1+ceil((real(zp)-Xmin)/PixW); iy=1+ceil((imag(zp)-Ymin)/PixH); 
    
    %check the hit count for this pixel:
    if Hits(ix,iy) < MaxHits
        %push the new z onto the stack:
        MyStack = [zp, MyStack]; %#ok<AGROW>
        if Hits(ix,iy) <1  %this is the first time hit this pixel, save for plotting
            ipcount = ipcount+1;
            if ipcount > npmax %watch out for exceeding array size before saving
                disp('jumping out of main loop, ipcount too big')
                break
            end
            Xp(ipcount)=real(zp); Yp(ipcount)=imag(zp);
        end
        %add one to the hit count for this pixel:
        Hits(ix,iy) = Hits(ix,iy)+1;
    end

% stop if MyStack is empty:
    if isempty(MyStack)
        disp('stack is empty, stopping, this is normal')
        break
    end
    
    maxstacksz=max(maxstacksz,length(MyStack));
end

disp(ipcount)
disp(maxstacksz)

figure(1)
clf('reset')

plot(Xp,Yp,'.k','MarkerSize',3)   

axis([Xmin Xmax  Ymin, Ymax]);
xlabel(' '); ylabel(' ');
ax=gca;
defFont=24;
ax.FontSize=defFont; ax.FontName='Times';
grid off;
ax.XTickMode='manual';  
ax.YTickMode='manual';
ax.XTick=[];  
ax.YTick=[]; 
ax.Box='off';
ax.Color='white';   
title(['\mu=', num2str(mu)])

toc
 
   function Zout = Finv(Zin, mu)
        % Finv iterates once on the positive branch of
        % the inverse function for the complex quadratic map z^2-mu
        % Zin is in general complex valued. So is mu. Both can be matrices.
        
        Zout = sqrt(Zin+mu);  %mu can be a scalar even if Zin is a matrix.
               
   end

end
