function fig6_12
%% This generates the Mandelbrot set for figure 6.12 in "Chaos" book
% The Mandelbrot set is calculated and plotted as an image, by counting the
% number of iterations of origin required to escape towards infinity.

% the iterates of the quadratic map z_{n+1} = z_n ^2 -mu
% ouput image resolution, WIDTHxHEIGHT
%
% 1000x1000 only takes a few seconds to run.
%
% the image used for the figure was 10,000 by 10,000, which takes about 12
% hours to run on a macbook pro in 2017. Hence we also provide a matlab
% figure file containing the output from this m-file.
% Mark McGuinness

WIDTH = 1000; %number of points in x axis
HEIGHT = 1000; %number of points in y axis

% fractal x y range
Y_MIN = -1.26;
Y_MAX = 1.26;
X_MIN = -0.6;
X_MAX = 2;

xbounds=max([X_MAX^2, X_MIN^2]);
ybounds=max([Y_MAX^2, Y_MIN^2]);
Ilimit=1*(xbounds^2+ybounds^2);

% maximum number of iterations
MAX_ITERATION = 10000;

% Generate linearly spaced points from X_MIN to X_MAX
x = linspace(X_MIN, X_MAX, WIDTH);
% Generate linearly spaced points from Y_MIN to Y_MAX
y = linspace(Y_MIN, Y_MAX, HEIGHT);

% Create a linear 2d grid
% X is 2d a array, column value are varying and row values are constant
% Y is 2d a array, row value are varying and column values are constant
[X, Y] = meshgrid(x, y);

% Allocate space for output
zval = zeros(HEIGHT, WIDTH);

nPoints = WIDTH * HEIGHT;

for m = 1:nPoints
    a = X(m);
    b = Y(m);
    xn = 0;
    yn = 0;
    k = 0;
    while (k < MAX_ITERATION) && ((xn^2+yn^2) < Ilimit)
        xnew = xn^2 - yn^2 - a;
        ynew = 2*xn*yn - b;
        xn = xnew;
        yn = ynew;
        k = k+1;
    end
    zval(m) = k; 
    %
    % note that an interesting refinement of using k to color on, would be
    % to use the "real distance", which may be less susceptible to the 
    % appearance of bands outside the set:
    % zval(m) = k - log(log(xn^2+yn^2)/log(Ilimit))/log(2)
    % This is described for example in:
    % https://en.wikipedia.org/wiki/Julia_set#Quadratic_polynomials

end
minz=min(min(zval)); maxz=max(max(zval));

%rescale zval to be on (0,1);
zval= (zval-minz)/(maxz-minz);

minz=min(min(zval)); maxz=max(max(zval));


inside=zval==maxz;  % identify points that survive max_iteration iterations so have max iterations

% offset the colours so the inside is dead black but the outside is not
% quite so black:
zval(inside)=minz - (maxz-minz)*0.01; %set inside points to be blacker not white, for contrast


% Try arctan as a
% localised stretching to emphasis the light bits: the following values
% have been hand-adjusted to get a good-looking image

width1=0.0008; shift1=4*width1; A1=0.8;  % colour from black to yellow
width2=0.5; shift2=0.7;             % colour from yellow to white
zval = morph(zval,shift1,width1,shift2,width2,A1);

figure(1)
clf('reset')

cmap = hot(MAX_ITERATION);
colormap(cmap);

display(cmap(1:10,1:3))
 
%show image
image(zval,'CDataMapping','scaled');

ax=gca; ax.XTickMode='manual';   ax.YTickMode='manual';
colorbar;

% plot the transformation used to crowd the colour changes onto 
% the boundary:

figure(2)  
clf('reset')

x=linspace(0,1,200);
morphx=morph(x,shift1,width1,shift2,width2,A1);

plot(x,morphx,'LineWidth',2)

% reverse the colour map of fig 1:

figure(3)
clf('reset')

cmap = flipud(hot(MAX_ITERATION));
colormap(cmap);

image(zval,'CDataMapping','scaled');

ax=gca; ax.XTickMode='manual';   ax.YTickMode='manual';
colorbar;


function outpt=morph(zval,shift1,width1,shift2,width2,A1)
    %MORPH is to distort the colour map and concentrate colour changes onto
    %the interesting boundary of the Mandelbrot set

    outpt = A1*(1+2/pi*atan((zval-shift1)/width1))+ ...
        (1-A1)*(1+2/pi*atan((zval-shift2)/width2)); % zval - place where it changes/ width of change region
end

end