function fig6_6
%% Generates figure 6.6 in Chaos.
% Figure 6.6 shows the normalised singular values 
% from a singular value decomposition of the
% solution to the lorenz equations with some white noise added to it; 
% d_E is 37, time step is 0.01, lag window is 0.36, 10,000 points wanted.
% The variance of the white noise is to be 16, so we multiply randn by
% sqrt(16)=4.
% Mark McGuinness

sigma=10; r=28; b=8/3;
dE=37; npt=10000; dt=0.01; variance=16;
stdev=sqrt(variance); 
   
options=odeset('RelTol',1e-4,'AbsTol',1e-4);

% run for a short while to get onto the chaotic attractor:
tend=2;
ICs=[5 5 10];  %for lorenz equations
[~,Ysol]=ode45(@MyDE,[0,tend],ICs,options);  %solve the Lorenz eqns, short run
ICs=Ysol(end,:); % save the final point to be new IC
    
tend=dt*npt;

[Tsol,Ysol]=ode45(@MyDE,[0,tend],ICs,options); %solve the Lorenz equations

tspan=linspace(0,tend,npt);
Ys =interp1(Tsol,Ysol,tspan)'; %sample the solution 10000 times
xvals=Ys(1,:) + stdev*randn(1,npt);  %add white noise with variance = stdev^2

mywidth=2;

fno=1;
figure(fno);
clf('reset')
subplot(2,1,1)
plot(tspan(1:10:end),Ys(1,1:10:end),'-r','linewidth',mywidth)
 
ax=gca;
grid off;
defFont=24;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';
xlabel('t'); ylabel('x','Rotation',0.0);

subplot(2,1,2)
plot(tspan(1:10:end),xvals(1:10:end),'-r','linewidth',mywidth)

ax=gca;
grid off;
defFont=24;
ax.FontSize=defFont; ax.FontName='Times';
ax.LineWidth=2.0;  ax.Box='on';
xlabel('t'); ylabel('x','Rotation',0.0);


% get ready to do svd on these x values
% break into rows that are dE long (so there are dE cols) but each row is
% only shifted one along in xvals

M=zeros(npt-dE+1,dE);
kk=1;
for ii=1:npt-dE+1
   M(ii,:) = xvals(kk:kk+dE-1);
   kk=kk+1;
end

% usually one would remove the average from each column of M, but x already has a mean near zero...

svals= svd(M)/sqrt(npt);


fno=fno+1;
figure(fno);
clf('reset')

plot(svals,'-ok','linewidth',mywidth)
  
ax=gca; 
ax.FontSize=defFont; ax.FontName='Times';
xlabel('i','FontSize',1.6*defFont); ylabel('\sigma_i ','FontSize',1.6*defFont,'Rotation',0.0);
 ax.LineWidth=2.0;  ax.Box='on';



function F=MyDE(~,yy)
% the Lorenz equations
    x=yy(1);
    y=yy(2);
    z=yy(3);
    f1  = sigma*(y-x);
    f2 = (r-z)*x -y;
    f3 = x*y -b*z;
    F=[f1; f2; f3];
    
 
end


end