% p_4_13_1.m
% Copyright
% T. Schmitz
% January 1, 2018

close all
clear
clc

% Define cutting force coefficients
Ks = 2250e6;                        % N/m^2
beta = 75;                          % deg
kt = Ks*sin(beta*pi/180);
kn = Ks*cos(beta*pi/180);

% Define parameters for x direction
kx = 9e6;                           % N/m
zetax = 0.02;
wnx = 900*2*pi;                     % rad/s
mx = kx/(wnx^2);                    % kg
cx = 2*zetax*sqrt(mx*kx);           % N-s/m

% Define parameters for y direction
ky = 1e7;                           % N/m
zetay = 0.01;
wny = 950*2*pi;                     % rad/s
my = ky/(wny^2);                    % kg
cy = 2*zetay*sqrt(my*ky);           % N-s/m

% Define cutting parameters
Nt = 3;
phis = 126.9;                       % deg
phie = 180;                         % deg
omega = 13000;                      % rpm
b = 4e-3;                           % m
ft = 0.2e-3;                        % m/tooth

% Simulation specifications
wnmax = max([wnx wny]);
fnmax = wnmax/2/pi;
DT = 1/(50*fnmax);                  % integration time step, s
steps_rev = 60/(omega*DT);          % steps per revolution
steps_tooth = steps_rev/Nt;         % steps per tooth
steps_tooth = round(steps_tooth);   % set steps_tooth as an integer
steps_rev = steps_tooth*Nt;
dt = 60/(steps_rev*omega);          % reset dt, s
dphi = 360/steps_rev;               % deg
rev = 40;
steps = rev*steps_rev;

% Initialize vectors
for cnt = 1:Nt
    teeth(cnt) = (cnt-1)*steps_rev/Nt + 1;
end
for cnt = 1:steps_rev
    phi(cnt) = (cnt - 1)*dphi;
end
for cnt = 1:steps_rev
    surf(cnt) = 0;
end

% Define initial conditions
x = 0;
dx = 0;
y = 0;
dy = 0;

for cnt1 = 1:steps
    time(cnt1) = dt*cnt1;
    for cnt2 = 1:Nt
        teeth(cnt2) = teeth(cnt2) + 1;
        if teeth(cnt2) > steps_rev
            teeth(cnt2) = 1;
        end
    end
    Fx = 0;
    Fy = 0;
    for cnt3 = 1:Nt
        if ((phi(teeth(cnt3)) >= phis) && (phi(teeth(cnt3)) <= phie))
            n = (x*sin(phi(teeth(cnt3))*pi/180) - y*cos(phi(teeth(cnt3))*pi/180));  % m
            h = ft*sin(phi(teeth(cnt3))*pi/180) + surf(teeth(cnt3)) - n;            % m
            Ft = kt*b*h;    % N
            if Ft < 0
                Ft = 0;
                surf(teeth(cnt3)) = surf(teeth(cnt3)) + ft*sin(phi(teeth(cnt3))*pi/180);
            else
                surf(teeth(cnt3)) = n;
            end
        else
            Ft = 0;
        end

        Fn = kn/kt*Ft;
        Fx = Fx + Ft*cos(phi(teeth(cnt3))*pi/180) + Fn*sin(phi(teeth(cnt3))*pi/180);
        Fy = Fy + Ft*sin(phi(teeth(cnt3))*pi/180) - Fn*cos(phi(teeth(cnt3))*pi/180);
    end
    Forcex(cnt1) = Fx;
    Forcey(cnt1) = Fy;

    % Numerical integration for position
    ddx = (Fx - cx*dx - kx*x)/mx;
    dx = dx + ddx*dt;
    x = x + dx*dt;
    xpos(cnt1) = x;

    ddy = (Fy - cy*dy - ky*y)/my;
    dy = dy + ddy*dt;
    y = y + dy*dt;
    ypos(cnt1) = y;
end

% Remove initial transients
start = round(length(time)*0.67);
time = time(start:length(time));
xpos = xpos(start:length(xpos));
ypos = ypos(start:length(ypos));
Forcex = Forcex(start:length(Forcex));
Forcey = Forcey(start:length(Forcey));

% Complete periodic sampling
time_s = time(1:steps_tooth:length(time));
xpos_s = xpos(1:steps_tooth:length(xpos));
ypos_s = ypos(1:steps_tooth:length(ypos));
Forcex_s = Forcex(1:steps_tooth:length(Forcex));
Forcey_s = Forcey(1:steps_tooth:length(Forcey));

% Calculate metric values
metricxpos = sum(abs(diff(xpos_s*1e6)))/length(xpos_s)   % micrometers
metricypos = sum(abs(diff(ypos_s*1e6)))/length(ypos_s)

figure(1)
subplot(211)
plot(time, Forcex, 'b', time_s, Forcex_s, 'ro')
axis([min(time) max(time) -1200 0]) 
set(gca,'FontSize', 14)
ylabel('F_x (N)')
subplot(212)
plot(time, xpos*1e6, 'b', time_s, xpos_s*1e6, 'ro')
axis([min(time) max(time) -300 300]) 
set(gca,'FontSize', 14)
xlabel('t (s)')
ylabel('x (\mum)')

figure(2)
subplot(211)
plot(time, Forcey, 'b', time_s, Forcey_s, 'ro')
axis([min(time) max(time) 0 2000]) 
set(gca,'FontSize', 14)
ylabel('F_y (N)')
subplot(212)
plot(time, ypos*1e6, 'b', time_s, ypos_s*1e6, 'ro')
axis([min(time) max(time) -300 300]) 
set(gca,'FontSize', 14)
xlabel('t (s)')
ylabel('y (\mum)')
