% p_4_16_1.m
% Copyright
% T. Schmitz
% June 1, 2008

clear
close all
clc

% Simulation specifications
kt = 710;                       % tangential specific force, N/mm^2
kte = 4;                        % tangential edge coefficient, N/mm
kn = 190;                       % normal specific force, N/mm^2
kne = 5;                        % normal edge coefficient, N/mm
ka = 95;                        % axial specific force, N/mm^2
kae = 2;                        % axial edge coefficient, N/mm
b = 5;                          % axial depth, mm
FT = [0.05 0.1 0.15 0.2 0.25];  % feed per tooth values, mm/tooth
phis = 0;                       % cut start angle, deg
phie = 180;                     % cut exit angle, deg
gamma = 30;                     % helix angle, deg
d = 19;                         % cutter diameter, mm
Nt = 2;                         % number of teeth, int
omega = 5000;                   % spindle speed, rpm
steps = 2^9;                    % steps for one cutter revolution, int
dt = 60/(steps*omega);          % time step, s
dphi = 360/steps;               % angular step, deg
if gamma == 0                   % straight teeth
    db = b;                     % discretized axial depth, mm
else                            % nonzero helix angle
    db = d*(dphi*pi/180)/2/tan(gamma*pi/180);
end
steps_axial = round(b/db);      % number of steps along tool axis
tooth_angles = 0:360/Nt:(360-360/Nt);  % angles of Nt cutter teeth starting from zero, deg

mean_Fx = zeros(1, length(FT));
mean_Fy = zeros(1, length(FT));
mean_Fz = zeros(1, length(FT));

for loop = 1:length(FT)
    ft = FT(loop);

    % Initialize vectors
    teeth = round(tooth_angles/dphi) + 1;
    phi = linspace(0, (steps-1)*dphi, steps);
    Force_x = zeros(1, steps);
    Force_y = zeros(1, steps);
    Force_z = zeros(1, steps);

    for cnt1 = 1:10*steps           % time steps, s
        for cnt2 = 1:Nt
            teeth(cnt2) = teeth(cnt2) + 1;      % index teeth one position (rotate cutter by dphi)
            if teeth(cnt2) > steps
                teeth(cnt2) = 1;
            end
        end

        Fx = 0;
        Fy = 0;
        Fz = 0;
        for cnt3 = 1:Nt                         % sum forces over all teeth
            for cnt4 = 1:steps_axial            % sum forces along axial depth of helical endmill
                phi_counter = teeth(cnt3) - (cnt4-1);
                if phi_counter < 1              % helix has wrapped through phi = 0 deg
                    phi_counter = phi_counter + steps;
                end
                phia = phi(phi_counter);        % angle for given axial disk, deg

                if (phia >= phis) && (phia <= phie)          % verify that tooth angle is in specified range for current disk, deg
                    h = ft*sin(phia*pi/180);                % chip thickness, mm
                    Ft = kt*db*h + kte*db;                  % tangential force, N
                    Fn = kn*db*h + kne*db;                  % normal force, N
                    Fa = ka*db*h + kae*db;                  % axial force, N
                else    % tooth angle is outside range bounded by radial immersion
                    Ft = 0;
                    Fn = 0;
                    Fa = 0;
                end

                Fx = Fx + Fn*sin(phia*pi/180) + Ft*cos(phia*pi/180);
                Fy = Fy - Fn*cos(phia*pi/180) + Ft*sin(phia*pi/180);
                Fz = Fz - Fa;
            end
        end

        Force_x(cnt1) = Fx;
        Force_y(cnt1) = Fy;
        Force_z(cnt1) = Fz;
        t(cnt1) = (cnt1-1)*dt;
        tooth_angle(cnt1) = (cnt1-1)*dphi;
    end
    time = tooth_angle/omega*60/360;

    Force_x = Force_x + max(Force_x)*0.1*randn(1, length(Force_x));
    Force_y = Force_y + max(Force_y)*0.1*randn(1, length(Force_y));
    Force_z = Force_z + max(Force_z)*0.1*randn(1, length(Force_z));

    mean_Fx(loop) = mean(Force_x);
    mean_Fy(loop) = mean(Force_y);
    mean_Fz(loop) = mean(Force_z);
end

figure(1)
plot(tooth_angle, Force_x, 'r:', tooth_angle, Force_y, 'b--', tooth_angle, Force_z, 'k')
axis([0 360 -600 1200])
set(gca,'FontSize', 14)
xlabel('\phi (deg)')
ylabel('Force (N)')
legend('F_x', 'F_y', 'F_z')

figure(2)
plot(t, Force_x, 'r:', t, Force_y, 'b--', t, Force_z, 'k')
axis([0 0.012 -600 1200])
set(gca,'FontSize', 14)
xlabel('t (s)')
ylabel('Force (N)')
legend('F_x', 'F_y', 'F_z')

% linear regression
n = length(FT);
a1x = (n*sum(FT.*mean_Fx) - sum(FT)*sum(mean_Fx))/(n*sum(FT.^2) - (sum(FT))^2);
a0x = mean(mean_Fx) - a1x*mean(FT);
kn_fit = 4*a1x/(Nt*b)
kne_fit = pi*a0x/(Nt*b)

a1y = (n*sum(FT.*mean_Fy) - sum(FT)*sum(mean_Fy))/(n*sum(FT.^2) - (sum(FT))^2);
a0y = mean(mean_Fy) - a1y*mean(FT);
kt_fit = 4*a1y/(Nt*b)
kte_fit = pi*a0y/(Nt*b)

a1z = (n*sum(FT.*mean_Fz) - sum(FT)*sum(mean_Fz))/(n*sum(FT.^2) - (sum(FT))^2);
a0z = mean(mean_Fz) - a1z*mean(FT);
ka_fit = -pi*a1z/(Nt*b)
kae_fit = -2*a0z/(Nt*b)

figure(3)
plot(FT, mean_Fx, 'ro', FT, mean_Fy, 'bs', FT, mean_Fz, 'k^')
hold on
axis([0 0.3 -125 525])
set(gca,'FontSize', 14)
xlabel('f_t (mm/tooth)')
ylabel('Mean Force (N)')
legend('x', 'y', 'z')
% best fit lines
plot(FT, a0x+a1x*FT, 'r:', FT, a0y+a1y*FT, 'b--', FT, a0z+a1z*FT, 'k')

% coefficients of determination
rx2 = (sum((mean_Fx - mean(mean_Fx)).^2) - sum((mean_Fx - a0x - a1x*FT).^2))/(sum((mean_Fx - mean(mean_Fx)).^2))
ry2 = (sum((mean_Fy - mean(mean_Fy)).^2) - sum((mean_Fy - a0y - a1y*FT).^2))/(sum((mean_Fy - mean(mean_Fy)).^2))
rz2 = (sum((mean_Fz - mean(mean_Fz)).^2) - sum((mean_Fz - a0z - a1z*FT).^2))/(sum((mean_Fz - mean(mean_Fz)).^2))
