% p_5_3_1.m
% Copyright
% T. Schmitz
% June 1, 2008

clc
close all
clear

% Variables
% Specific cutting force values
kt = 700e6;                     % tangential cutting force coefficient, N/m^2
kn = 210e6;                     % radial cutting force coefficient, N/m^2
kte = 0e3;                      % tangential edge constant, N/m
kne = 0e3;                      % radial edge constant, N/m

% Tool dynamics
kx = [1e7];                     % N/m
zetax = [0.01];
wnx = [500]*2*pi;               % rad/s
mx = kx./(wnx.^2);              % kg
cx = 2*zetax.*(mx.*kx).^0.5;    % N-s/m
x_modes = length(kx);           % number of modes in x-direction, integer

ky = [1e7];                     % N/m
zetay = [0.01];
wny = [500]*2*pi;               % rad/s
my = ky./(wny.^2);              % kg
cy = 2*zetay.*(my.*ky).^0.5;    % N-s/m
y_modes = length(ky);           % number of modes in y-direction, integer

% Cutting direction: 1 = down milling, 2 = up milling
cut_dir = 2;
if (cut_dir ~= 1) && (cut_dir ~= 2)
    disp('Must select 1 (down) or 2 (up) to designate cutting direction.')
    return
end

% Cutting tool specifications
Nt = 4;                         % number of teeth, integer
d = 12.7e-3;                    % nominal tool diameter, m
gamma = 30;                     % helix angle, deg
r = d/2;                        % tool radius, m
teeth = [0 90 180 270];         % teeth angles, deg

% Machining specifications
omega = 7500;                   % spindle speed, rpm
a = d/2;                        % radial depth of cut, m
ytest = -(d/2 - a);             % y-direction coordinate that bounds cut, m
b = 4e-3;                       % axial depth of cut, m
ft = 0.1e-3;                    % feed/tooth, m

% Simulation specifications
steps_tooth = 750;                              % number of steps between teeth
steps_rev = Nt*steps_tooth;                     % steps per revolution
rev = 60;                                       % number of revolutions
rev = rev + 2;                                  % add two initialization revolutions
steps = rev*steps_rev;                          % total number of steps
dt = 60/(steps_rev*omega);                      % integration time step, s
dphi = 360/steps_rev;                           % angular steps size per time step, deg
dfeed = Nt*ft/steps_rev;                        % linear feed between time steps, m
steps_axial = 10;                                % number of axial steps
db = b/steps_axial;                             % axial depth increment, m
delta_phi = 2*db*tan(gamma*pi/180)/d*180/pi;    % angle delay between axial depths, deg

% Initialize vectors
% surface vectors: each column represents a tooth, while each "page" represents an axial depth
surfx = zeros(steps, Nt, steps_axial);          % x direction surface vector
surfy = zeros(steps, Nt, steps_axial);          % y direction surface vector
for cnt3 = 1:steps_axial
    for cnt2 = 1:Nt
        for cnt1 = 1:2*steps_rev
            phi = (cnt1 - 1)*dphi - teeth(cnt2) - delta_phi*(cnt3 - 1);
            if phi < 0
                phi = phi + 360;
            end
            if phi >= 360
                phi = phi - 360;
            end
            % Tooth cnt1 surface
            surfx(cnt1, cnt2, cnt3) = r*sin(phi*pi/180) + dfeed*(cnt1-1);   % x component of old surface
            surfy(cnt1, cnt2, cnt3) = r*cos(phi*pi/180);                    % y component
        end     % cnt1 -- row
    end         % cnt2 -- column
end             % cnt3 -- page
Forcex = zeros(1, steps);
Forcey = zeros(1, steps);
xpos = zeros(1, steps);
ypos = zeros(1, steps);
xvel = zeros(1, steps);
yvel = zeros(1, steps);
time = (0:steps-1)*dt;

% Plot first two revolutions of cutter (at selected axial slice) used to initialize simulation
% Select axial depth to plot using plot_depth variable (1 corresponds to free end of cutter)
plot_depth = 1;
figure(1)
hold on
for cnt = 1:Nt
    eval(['plot(surfx(:,' num2str(cnt) ',plot_depth)*1e3, surfy(:,' num2str(cnt) ',plot_depth)*1e3);'])
end
set(gca,'FontSize', 14)
xlabel('x (mm)')
ylabel('y (mm)')

% Euler integration initial conditions
x = 0;
dx = 0;
y = 0;
dy = 0;
dp = zeros(1, x_modes);         
p = zeros(1, x_modes);          % x direction modal displacements, m
dq = zeros(1, y_modes);
q = zeros(1, y_modes);          % y direction modal displacements, m
feed = 2*Nt*ft-dfeed;           % linear feed during two initialization revolutions, m
phi = -dphi;

% Function to keep track of simulation progress
handle = waitbar(0, 'Please wait... simulation in progress.');

for cnt1 = (2*steps_rev + 1):steps
    waitbar((cnt1 - 2*steps_rev)/(steps - 2*steps_rev), handle)
    phi = phi + dphi;           % cutter angle, deg
    if phi >= 360
        phi = phi - 360;
    end
    feed = feed + dfeed;        % m
    cenx = feed + x;            % x direction center of cutter, m
    ceny = y;                   % y direction center of cutter, m
    
    Fx = 0;
    Fy = 0;
    
    for cnt3 = 1:Nt              % sum forces over all teeth
        for cnt4 = 1:steps_axial
            % Specify angle for current tooth
            phi_tooth = phi - teeth(cnt3) - delta_phi*(cnt4 - 1);
            if phi_tooth < 0
                phi_tooth = phi_tooth + 360;
            end
            
            % Calculate coordinates of current tooth
            Cx = r*sin(phi_tooth*pi/180) + cenx;     % x coordinate of point C, m
            Cy = r*cos(phi_tooth*pi/180) + ceny;     % y coordinate of point C, m
            
            % Verify that point cutting can occur using selected radial immersion
            % Must first calculate D coordinates by linear interpolation
            % Search to find point B
            search_cnt = cnt1 - ceil(1.1*steps_tooth);  % number of steps to look back, integer
            tooth_cnt = cnt3 - 1;   % Tooth number to search in surf(x,y) vectors, integer
            if tooth_cnt < 1
                tooth_cnt = Nt;     % Wrap around for tooth 1
            end
            flag = -1;              % Initialize flag before entering 'while' loop to search for point B
            while flag < 0
                test_Bx = surfx(search_cnt, tooth_cnt, cnt4);
                test_By = surfy(search_cnt, tooth_cnt, cnt4);
                test_Ax = surfx(search_cnt-1, tooth_cnt, cnt4);
                test_Ay = surfy(search_cnt-1, tooth_cnt, cnt4);
                phi_B = atan2((test_Bx - cenx), (test_By - ceny))*180/pi;
                if phi_B < 0        % Correct for quadrant-based result of atan2
                    phi_B = phi_B + 360;
                end
                phi_A = atan2((test_Ax - cenx), (test_Ay - ceny))*180/pi;
                if phi_A < 0
                    phi_A = phi_A + 360;
                end
                if abs(phi_B - phi_A) > 180
                    if phi_tooth < 180
                        phi_A = phi_A - 360;
                    else
                        phi_B = phi_B + 360;
                    end
                end
                if (phi_tooth > phi_A) && (phi_tooth <= phi_B)    % Exit condition
                    flag = 1;
                else    % Continue searching
                    search_cnt = search_cnt + 1;
                end
                if search_cnt > cnt1 + ceil(0.5*steps_tooth) || search_cnt > steps
                    flag = 1;                           % Artificially end while loop
                    search_cnt = cnt1 - steps_tooth;    % Look back nominally one tooth period
                end
            end
            % 'While' loop condition is satisfied, define point B
            Bx = surfx(search_cnt, tooth_cnt, cnt4);
            By = surfy(search_cnt, tooth_cnt, cnt4);
            
            % Find point A (previous point)
            Ax = surfx(search_cnt-1, tooth_cnt, cnt4);
            Ay = surfy(search_cnt-1, tooth_cnt, cnt4);
            
            % Interpolate to find D
            C = (Ay - By)/(Ax - Bx);
            Dx = (tan(phi_tooth*pi/180)*Ax*C - tan(phi_tooth*pi/180)*Ay + tan(phi_tooth*pi/180)*Cy - Cx)/(tan(phi_tooth*pi/180)*C - 1);
            Dy = Ay - Ax*C + Dx*C;
            
            % The test to determine if cutting is occurring follows, must account
            % for different cut entry and chip thinning conditions depending on radial immersion
            if cut_dir == 1     % down milling
                if a <= d/2     % radial immersion is less than or equal to 50%
                    test = Cy;
                else            % radial immersion is greater than 50%
                    test = Dy;
                end
            else                % up milling
                if a <= d/2     % radial immersion is less than or equal to 50%
                    test = -Cy;
                else            % radial immersion is greater than 50%
                    test = -Dy;
                end
            end
            
            if test > ytest     % No cutting can occur
                % Set forces equal to zero
                Ft = 0;
                Fn = 0;
                % Define surface using current tooth position
                surfx(cnt1, cnt3, cnt4) = Cx;
                surfy(cnt1, cnt3, cnt4) = Cy;
            else    % Cutting can occur, but is not guaranteed
                % Calculate vector lengths used to determine if cutting is occurring      
                OC = sqrt((Cx - cenx)^2 + (Cy - ceny)^2);
                OD = sqrt((Dx - cenx)^2 + (Dy - ceny)^2);
                
                if OC >= OD     % Tooth is cutting
                    surfx(cnt1, cnt3, cnt4) = Cx;
                    surfy(cnt1, cnt3, cnt4) = Cy;
                    % Check to see if chip thinning at cut entry is occurring
                    if cut_dir == 1 % down milling
                        if a <= d/2     % radial immersion is less than or equal to 50%
                            test = Dy;
                        else            % radial immersion is greater than 50%
                            test = Cy;
                        end
                    else            % up milling
                        if a <= d/2     % radial immersion is less than or equal to 50%
                            test = -Dy;
                        else            % radial immersion is greater than 50%
                            test = -Cy;
                        end
                    end
                    
                    if test < ytest     % No chip thinning
                        % Calculate chip thickness, h
                        h = sqrt((Cx - Dx)^2 + (Cy - Dy)^2);
                        % Determine forces
                        Ft = kt*db*h + kte*db;
                        Fn = kn*db*h + kne*db;
                    else    % Chip thinning is occurring
                        if cut_dir == 1     % down milling
                            Dxp = (ytest - ceny)*tan(phi_tooth*pi/180) + cenx;  % new point D x coordinate
                            Dyp = ytest;                                        % new point D y coordinate
                            if a <= d/2     % radial immersion is less than or equal to 50%
                                h = sqrt((Cx - Dxp)^2 + (Cy - Dyp)^2);
                            else            % radial immersion is greater than 50%
                                h = sqrt((Dxp - Dx)^2 + (Dyp - Dy)^2);
                            end
                        else                % up milling
                            % Must use correct ytest value (changed sign to allow single test/ytest
                            % comparison for up and down milling)
                            Dxp = (-ytest - ceny)*tan(phi_tooth*pi/180) + cenx;  % new point D x-coordinate
                            Dyp = -ytest;                                        % new point D y-coordinate
                            if a <= d/2     % radial immersion is less than or equal to 50%
                                h = sqrt((Cx - Dxp)^2 + (Cy - Dyp)^2);
                            else            % radial immersion is greater than 50%
                                h = sqrt((Dxp - Dx)^2 + (Dyp - Dy)^2);
                            end
                        end
                        % Determine forces
                        Ft = kt*db*h + kte*db;
                        Fn = kn*db*h + kne*db;
                    end
                else        % Tooth is not cutting (h < 0)
                    % Set forces equal to zero
                    Ft = 0;
                    Fn = 0;
                    % Define surface using current surface
                    surfx(cnt1, cnt3, cnt4) = Cx;
                    surfy(cnt1, cnt3, cnt4) = Cy;
                end
            end     % end of 'if' statement to compare test and ytest
            
            Fx = Fx - Ft*cos(phi_tooth*pi/180) - Fn*sin(phi_tooth*pi/180);
            Fy = Fy + Ft*sin(phi_tooth*pi/180) - Fn*cos(phi_tooth*pi/180);
        end     % end of cnt4 'for' loop
    end         % end of cnt3 'for' loop
    
    Forcex(cnt1) = Fx;
    Forcey(cnt1) = Fy;

    % Euler integration for position
    x = 0;
    y = 0;
    dx = 0;
    dy = 0;
    
    % x direction
    for cnt5 = 1:x_modes
        ddp = (Fx - cx(cnt5)*dp(cnt5) - kx(cnt5)*p(cnt5))/mx(cnt5);
        dp(cnt5) = dp(cnt5) + ddp*dt;
        p(cnt5) = p(cnt5) + dp(cnt5)*dt;
        x = x + p(cnt5);        % m
        dx = dx + dp(cnt5);     % m/s
    end
    xpos(cnt1) = x;
    xvel(cnt1) = dx;
       
    % y direction
    for cnt5 = 1:y_modes
        ddq = (Fy - cy(cnt5)*dq(cnt5) - ky(cnt5)*q(cnt5))/my(cnt5);
        dq(cnt5) = dq(cnt5) + ddq*dt;
        q(cnt5) = q(cnt5) + dq(cnt5)*dt;
        y = y + q(cnt5);        % m
        dy = dy + dq(cnt5);     % m/s
    end
    ypos(cnt1) = y;
    yvel(cnt1) = dy;
end     % end of cnt1 'for' loop

close(handle);

% Plot force and displacement data
% Trim initialization portion of data
time = time((2*steps_rev + 1):length(time));
time = time - time(1);  % Normalize to zero starting time
Forcex = Forcex((2*steps_rev + 1):length(Forcex));
xpos = xpos((2*steps_rev + 1):length(xpos));
xvel = xvel((2*steps_rev + 1):length(xvel));
Forcey = Forcey((2*steps_rev + 1):length(Forcey));
ypos = ypos((2*steps_rev + 1):length(ypos));
yvel = yvel((2*steps_rev + 1):length(yvel));

figure(2)
subplot(211)
plot(time, Forcex)
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
ylabel('F_x (N)')
subplot(212)
plot(time, xpos*1e6)
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
xlabel('t (s)')
ylabel('x (\mum)')

figure(3)
subplot(211)
plot(time, Forcey)
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
ylabel('F_y (N)')
subplot(212)
plot(time, ypos*1e6)
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
xlabel('t (s)')
ylabel('y (\mum)')

% Plot final simulated surface for particular axial slice of cutter
scale_factor = 0.1*d/2;        % scales y-direction axis range
if cut_dir == 1                 % down milling
    nominal = -d/2;             % nominal value for axis scaling
    xmax = sqrt((d/2)^2 - (nominal + scale_factor)^2);
    xmin = -xmax;
    xmax = xmax + rev*ft*Nt;
else                            % up milling
    nominal = d/2;
    xmax = sqrt((d/2)^2 - (nominal - scale_factor)^2);
    xmin = -xmax;
    xmax = xmax + rev*ft*Nt;
end

xsurf = [];
ysurf = [];

plot_depth = 5;     % Select axial slice for Ra and SLE calculation
figure(4)
hold on
for cnt = 1:Nt
    eval(['toothx = surfx(:,' num2str(cnt) ',plot_depth)*1e3;'])
    toothx = toothx((2*steps_rev + 1):length(toothx));
    xsurf = [xsurf toothx'];
    eval(['toothy = surfy(:,' num2str(cnt) ',plot_depth)*1e3;'])
    toothy = toothy((2*steps_rev + 1):length(toothy));
    ysurf = [ysurf toothy'];
    plot(toothx, toothy)
end
set(gca,'FontSize', 14)
xlabel('x (mm)')
ylabel('y (mm)')
axis([xmin xmax nominal-scale_factor nominal+scale_factor]*1e3)

% Trim to portion of path near machined surface
if cut_dir == 1                 % down milling
    select = -0.95*(d/2*1e3);
    index = find(ysurf < select);
else                            % up milling
    select = 0.95*(d/2*1e3);
    index = find(ysurf > select);
end
xsurf = xsurf(index);
ysurf = ysurf(index);

% Sort arrays in ascending order of xsurf (feed) direction
[xsurf, index] = sort(xsurf);
ysurf = ysurf(index);

% Trim to second half of data files
total_length = length(xsurf);
xsurf = xsurf(round(total_length/2):total_length);
ysurf = ysurf(round(total_length/2):total_length);

xorig = xsurf;
yorig = ysurf;

iterations = 100;
handle = waitbar(0, 'Please wait... sorting data.');
for counter = 1:iterations
    waitbar(counter/iterations, handle)
    x2 = xsurf(1);
    y2 = ysurf(1);
    
    if cut_dir == 1     % down milling
        for cnt = 1:(length(xsurf)-2)
            dy1 = ysurf(cnt+1) - ysurf(cnt);
            dy2 = ysurf(cnt+2) - ysurf(cnt);
            if dy2 < dy1
                x2 = [x2 xsurf(cnt+2)];
                y2 = [y2 ysurf(cnt+2)];
            else
                x2 = [x2 xsurf(cnt+1)];
                y2 = [y2 ysurf(cnt+1)];
            end
        end
    else                % up milling
        for cnt = 1:(length(xsurf)-2)
            dy1 = ysurf(cnt+1) - ysurf(cnt);
            dy2 = ysurf(cnt+2) - ysurf(cnt);
            if dy2 > dy1
                x2 = [x2 xsurf(cnt+2)];
                y2 = [y2 ysurf(cnt+2)];
            else
                x2 = [x2 xsurf(cnt+1)];
                y2 = [y2 ysurf(cnt+1)];
            end
        end
    end
    
    xsurf = x2;
    ysurf = y2;
end
close(handle);

% Select portion of path in steady-state (for stable cuts) and remove end of path (where tool radius is seen)
total_length = length(x2);
x2 = x2(round(1/5*total_length):round(4/5*total_length));
y2 = y2(round(1/5*total_length):round(4/5*total_length));

figure(5)
plot(xorig, yorig, 'r*', x2, y2, 'b-')
set(gca,'FontSize', 14)
xlabel('x (mm)')
ylabel('y (mm)')

% Calculate SLE
if cut_dir == 1                     % down milling
    SLE = (d/2*1e3 + mean(y2))*1e3  % undercut is positive, micrometers
else                                % up milling
    SLE = -(d/2*1e3 - mean(y2))*1e3 % overcut is positive, micrometers
end

% Calculate roughness average
y2 = y2*1e3;                        % micrometers
y2 = y2 - mean(y2);
y2abs = abs(y2);
Ra = sum(y2abs)/length(y2abs)       % roughness average, micrometers
