% p_6_9_1.m
% Copyright
% T. Schmitz
% June 1, 2008

clc
close all
clear

% Variables
% Specific cutting force values
kt = 644e6;                     % N/m^2
kn = 238e6;                     % N/m^2
kte = 0e3;                      % tangential edge coefficient, N/m
kne = 0e3;                      % radial edge coefficient, N/m

% Define modal parameters for x direction
kx = [4.1e5];                   % N/m
zetax = [0.009];
wnx = [720]*2*pi;               % rad/s
mx = kx./(wnx.^2);              % kg
cx = 2*zetax.*(mx.*kx).^0.5;    % N-s/m
x_modes = length(kx);           % number of modes in x direction

% Define modal parameters for y direction
ky = [4.1e5];                   % N/m
zetay = [0.009];
wny = [720]*2*pi;               % rad/s
my = ky./(wny.^2);              % kg
cy = 2*zetay.*(my.*ky).^0.5;    % N-s/m
y_modes = length(ky);           % number of modes in y direction

% Cutting direction: 1 = down milling, 2 = up milling
cut_dir = 2;
if (cut_dir ~= 1) && (cut_dir ~= 2)
    disp('Must select 1 (down) or 2 (up) to designate cutting direction.')
    return
end

% Cutting tool specifications
Nt = 1;                         % number of teeth, integer
d = 8e-3;                       % nominal tool diameter, m
gamma = 45;                     % helix angle, deg
RO = [0]*1e-6;                  % runout, m
r = d/2 + RO;                   % tool radius, m
teeth = [0];                    % teeth angles, deg

% Machining specifications
omega = 28000;                  % spindle speed, rpm
a = 0.05*d;                     % radial depth of cut, m
ytest = -(d/2 - a);             % y-direction coordinate that bounds cut, m
b = 2e-3;                       % axial depth of cut, m
ft = 0.1e-3;                    % feed/tooth, m

% Simulation specifications
steps_tooth = 800;                              % number of steps between teeth
steps_rev = Nt*steps_tooth;                     % steps per revolution
rev = 75;                                       % number of revolutions
rev = rev + 2;                                  % add two initialization revolutions
steps = rev*steps_rev;                          % total number of steps
dt = 60/(steps_rev*omega);                      % integration time step, s
dphi = 360/steps_rev;                           % angular steps size per time step, deg
dfeed = Nt*ft/steps_rev;                        % linear feed between time steps, m
steps_axial = 8;                                % number of axial steps
db = b/steps_axial;                             % axial depth increment, m
delta_phi = 2*db*tan(gamma*pi/180)/d*180/pi;    % angle delay between axial depths, deg

% Initialize vectors
% surface vectors: each column represents a tooth, while each "page" represents an axial depth
surfx = zeros(steps, Nt, steps_axial);          % x direction surface vector
surfy = zeros(steps, Nt, steps_axial);          % y direction surface vector
for cnt3 = 1:steps_axial
    for cnt2 = 1:Nt
        for cnt1 = 1:2*steps_rev
            phi = (cnt1 - 1)*dphi - teeth(cnt2) - delta_phi*(cnt3 - 1);
            if phi < 0
                phi = phi + 360;
            end
            if phi >= 360
                phi = phi - 360;
            end
            % Tooth cnt1 surface
            surfx(cnt1, cnt2, cnt3) = r(cnt2)*sin(phi*pi/180) + dfeed*(cnt1-1);   % x component of old surface
            surfy(cnt1, cnt2, cnt3) = r(cnt2)*cos(phi*pi/180);                    % y component
        end     % cnt1 -- row
    end         % cnt2 -- column
end             % cnt3 -- page
Forcex = zeros(1, steps);
Forcey = zeros(1, steps);
xpos = zeros(1, steps);
ypos = zeros(1, steps);
xvel = zeros(1, steps);
yvel = zeros(1, steps);
time = (0:steps-1)*dt;

% Euler integration initial conditions
x = 0;
dx = 0;
y = 0;
dy = 0;
dp = zeros(1, x_modes);         
p = zeros(1, x_modes);          % x direction modal displacements, m
dq = zeros(1, y_modes);
q = zeros(1, y_modes);          % y direction modal displacements, m
feed = 2*Nt*ft-dfeed;           % linear feed during two initialization revolutions, m
phi = -dphi;

% Function to keep track of simulation progress
handle = waitbar(0, 'Please wait... simulation in progress.');

for cnt1 = (2*steps_rev + 1):steps
    waitbar((cnt1 - 2*steps_rev)/(steps - 2*steps_rev), handle)
    phi = phi + dphi;           % cutter angle, deg
    if phi >= 360
        phi = phi - 360;
    end
    feed = feed + dfeed;        % m
    cenx = feed + x;            % x direction center of cutter, m
    ceny = y;                   % y direction center of cutter, m
    
    Fx = 0;
    Fy = 0;
    
    for cnt3 = 1:Nt              % sum forces over all teeth
        for cnt4 = 1:steps_axial
            % Specify angle for current tooth
            phi_tooth = phi - teeth(cnt3) - delta_phi*(cnt4 - 1);
            if phi_tooth < 0
                phi_tooth = phi_tooth + 360;
            end
            
            % Calculate coordinates of current tooth
            Cx = r(cnt3)*sin(phi_tooth*pi/180) + cenx;     % x coordinate of point C, m
            Cy = r(cnt3)*cos(phi_tooth*pi/180) + ceny;     % y coordinate of point C, m
            
            % Verify that point cutting can occur using selected radial immersion
            % Must first calculate D coordinates by linear interpolation
            % Search to find point B
            search_cnt = cnt1 - ceil(1.1*steps_tooth);  % number of steps to look back, integer
            tooth_cnt = cnt3 - 1;   % Tooth number to search in surf(x,y) vectors, integer
            if tooth_cnt < 1
                tooth_cnt = Nt;     % Wrap around for tooth 1
            end
            flag = -1;              % Initialize flag before entering 'while' loop to search for point B
            while flag < 0
                test_Bx = surfx(search_cnt, tooth_cnt, cnt4);
                test_By = surfy(search_cnt, tooth_cnt, cnt4);
                test_Ax = surfx(search_cnt-1, tooth_cnt, cnt4);
                test_Ay = surfy(search_cnt-1, tooth_cnt, cnt4);
                phi_B = atan2((test_Bx - cenx), (test_By - ceny))*180/pi;
                if phi_B < 0        % Correct for quadrant-based result of atan2
                    phi_B = phi_B + 360;
                end
                phi_A = atan2((test_Ax - cenx), (test_Ay - ceny))*180/pi;
                if phi_A < 0
                    phi_A = phi_A + 360;
                end
                if abs(phi_B - phi_A) > 180
                    if phi_tooth < 180
                        phi_A = phi_A - 360;
                    else
                        phi_B = phi_B + 360;
                    end
                end
                if (phi_tooth > phi_A) && (phi_tooth <= phi_B)    % Exit condition
                    flag = 1;
                else    % Continue searching
                    search_cnt = search_cnt + 1;
                end
                if search_cnt > cnt1 + ceil(0.5*steps_tooth) || search_cnt > steps
                    flag = 1;                           % Artificially end while loop
                    search_cnt = cnt1 - steps_tooth;    % Look back nominally one tooth period
                end
            end
            % 'while' loop condition is satisfied, define point B
            Bx = surfx(search_cnt, tooth_cnt, cnt4);
            By = surfy(search_cnt, tooth_cnt, cnt4);
            
            % Find point A (previous point)
            Ax = surfx(search_cnt-1, tooth_cnt, cnt4);
            Ay = surfy(search_cnt-1, tooth_cnt, cnt4);
            
            % Interpolate to find D
            C = (Ay - By)/(Ax - Bx);
            Dx = (tan(phi_tooth*pi/180)*Ax*C - tan(phi_tooth*pi/180)*Ay + tan(phi_tooth*pi/180)*Cy - Cx)/(tan(phi_tooth*pi/180)*C - 1);
            Dy = Ay - Ax*C + Dx*C;
            
            % The test to determine if cutting is occurring follows, must account
            % for different cut entry and chip thinning conditions depending on radial immersion
            if cut_dir == 1     % down milling
                if a <= d/2     % radial immersion is less than or equal to 50%
                    test = Cy;
                else            % radial immersion is greater than 50%
                    test = Dy;
                end
            else                % up milling
                if a <= d/2     % radial immersion is less than or equal to 50%
                    test = -Cy;
                else            % radial immersion is greater than 50%
                    test = -Dy;
                end
            end
            
            if test > ytest     % No cutting can occur
                % Set forces equal to zero
                Ft = 0;
                Fn = 0;
                % Define surface using current tooth position
                surfx(cnt1, cnt3, cnt4) = Cx;
                surfy(cnt1, cnt3, cnt4) = Cy;
            else    % Cutting can occur, but is not guaranteed
                % Calculate vector lengths used to determine if cutting is occurring      
                OC = sqrt((Cx - cenx)^2 + (Cy - ceny)^2);
                OD = sqrt((Dx - cenx)^2 + (Dy - ceny)^2);
                
                if OC >= OD     % Tooth is cutting
                    surfx(cnt1, cnt3, cnt4) = Cx;
                    surfy(cnt1, cnt3, cnt4) = Cy;
                    % Check to see if chip thinning at cut entry is occurring
                    if cut_dir == 1 % down milling
                        if a <= d/2     % radial immersion is less than or equal to 50%
                            test = Dy;
                        else            % radial immersion is greater than 50%
                            test = Cy;
                        end
                    else            % up milling
                        if a <= d/2     % radial immersion is less than or equal to 50%
                            test = -Dy;
                        else            % radial immersion is greater than 50%
                            test = -Cy;
                        end
                    end
                    
                    if test < ytest     % No chip thinning
                        % Calculate chip thickness, h
                        h = sqrt((Cx - Dx)^2 + (Cy - Dy)^2);
                        % Determine forces
                        Ft = kt*db*h + kte*db;
                        Fn = kn*db*h + kne*db;
                    else    % Chip thinning is occurring
                        if cut_dir == 1     % down milling
                            Dxp = (ytest - ceny)*tan(phi_tooth*pi/180) + cenx;  % new point D x coordinate
                            Dyp = ytest;                                        % new point D y coordinate
                            if a <= d/2     % radial immersion is less than or equal to 50%
                                h = sqrt((Cx - Dxp)^2 + (Cy - Dyp)^2);
                            else            % radial immersion is greater than 50%
                                h = sqrt((Dxp - Dx)^2 + (Dyp - Dy)^2);
                            end
                        else                % up milling
                            % Must use correct ytest value (changed sign to allow single test/ytest
                            % comparison for up and down milling)
                            Dxp = (-ytest - ceny)*tan(phi_tooth*pi/180) + cenx;  % new point D x-coordinate
                            Dyp = -ytest;                                        % new point D y-coordinate
                            if a <= d/2     % radial immersion is less than or equal to 50%
                                h = sqrt((Cx - Dxp)^2 + (Cy - Dyp)^2);
                            else            % radial immersion is greater than 50%
                                h = sqrt((Dxp - Dx)^2 + (Dyp - Dy)^2);
                            end
                        end
                        % Determine forces
                        Ft = kt*db*h + kte*db;
                        Fn = kn*db*h + kne*db;
                    end
                else        % Tooth is not cutting (h < 0)
                    % Set forces equal to zero
                    Ft = 0;
                    Fn = 0;
                    % Define surface using current surface
                    surfx(cnt1, cnt3, cnt4) = Cx;
                    surfy(cnt1, cnt3, cnt4) = Cy;
                end
            end     % end of 'if' statement to compare test and ytest
            
            Fx = Fx - Ft*cos(phi_tooth*pi/180) - Fn*sin(phi_tooth*pi/180);
            Fy = Fy + Ft*sin(phi_tooth*pi/180) - Fn*cos(phi_tooth*pi/180);
        end     % end of cnt4 'for' loop
    end         % end of cnt3 'for' loop
    
    Forcex(cnt1) = Fx;
    Forcey(cnt1) = Fy;

    % Euler integration for position
    x = 0;
    y = 0;
    dx = 0;
    dy = 0;
    
    % x direction
    for cnt5 = 1:x_modes
        ddp = (Fx - cx(cnt5)*dp(cnt5) - kx(cnt5)*p(cnt5))/mx(cnt5);
        dp(cnt5) = dp(cnt5) + ddp*dt;
        p(cnt5) = p(cnt5) + dp(cnt5)*dt;
        x = x + p(cnt5);        % m
        dx = dx + dp(cnt5);     % m/s
    end
    xpos(cnt1) = x;
    xvel(cnt1) = dx;
       
    % y direction
    for cnt5 = 1:y_modes
        ddq = (Fy - cy(cnt5)*dq(cnt5) - ky(cnt5)*q(cnt5))/my(cnt5);
        dq(cnt5) = dq(cnt5) + ddq*dt;
        q(cnt5) = q(cnt5) + dq(cnt5)*dt;
        y = y + q(cnt5);        % m
        dy = dy + dq(cnt5);     % m/s
    end
    ypos(cnt1) = y;
    yvel(cnt1) = dy;
end     % end of cnt1 'for' loop

close(handle);

% Plot force and displacement data
% Trim initialization portion of data
time = time((2*steps_rev + 1):length(time));
time = time - time(1);  % Normalize to zero starting time
Forcex = Forcex((2*steps_rev + 1):length(Forcex));
xpos = xpos((2*steps_rev + 1):length(xpos));
xvel = xvel((2*steps_rev + 1):length(xvel));
Forcey = Forcey((2*steps_rev + 1):length(Forcey));
ypos = ypos((2*steps_rev + 1):length(ypos));
yvel = yvel((2*steps_rev + 1):length(yvel));

% Perform once-per-tooth samping
tsample = time(1:steps_tooth:length(time));
xsample = xpos(1:steps_tooth:length(xpos));
ysample = ypos(1:steps_tooth:length(ypos));
Fxsample = Forcex(1:steps_tooth:length(Forcex));
Fysample = Forcey(1:steps_tooth:length(Forcey));

figure(1)
subplot(211)
plot(time, Forcex, 'g', tsample, Fxsample, 'r+')
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
ylabel('F_x (N)')
subplot(212)
plot(time, xpos*1e6, 'g', tsample, xsample*1e6, 'r+')
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
xlabel('t (s)')
ylabel('x (\mum)')

figure(2)
subplot(211)
plot(time, Forcey, 'g', tsample, Fysample, 'r+')
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
ylabel('F_y (N)')
subplot(212)
plot(time, ypos*1e6, 'g', tsample, ysample*1e6, 'r+')
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
xlabel('t (s)')
ylabel('y (\mum)')

figure(3)
subplot(211)
plot(time, xpos*1e6, 'g', tsample, xsample*1e6, 'r+')
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
ylabel('x (\mum)')
subplot(212)
plot(time, ypos*1e6, 'g', tsample, ysample*1e6, 'r+')
xlim([0 max(time)]) 
set(gca,'FontSize', 14)
xlabel('t (s)')
ylabel('y (\mum)')

figure(4)
plot(xpos*1e6, ypos*1e6, 'g-', xsample*1e6, ysample*1e6, 'r+')
set(gca,'FontSize', 14)
xlabel('x (\mum)')
ylabel('y (\mum)')
