%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%Inductive Links for Wireless Power Transfer%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%Pablo Pérez-Nicoli, Fernando Silveira and Maysam Ghovanloo%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%% CHAPTER 2, SECTION 2.1.7 %%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%% Frequency Splitting Effect %%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

close all 
clear all
clc

figure_config                       % Set figures style

%% PARAMETERS %%

QTX=200;                            % Tx quality factor
QRX=200;                            % Rx quality factor
QL=50;                              % Load quality factor QL=wLRX/Re{ZMN}
QRXL= QRX*QL/(QRX+QL);
RTX=5;                              % Tx coil series parasitic resistance, in ohms
kTXRX=(0.0001:0.002:0.06);          % Tx-Rx coupling coefficient
w=(12.5e6:20e3:14.5e6)*2*pi;        % Carrier signal frequency
wres=13.56e6*2*pi;                  % Coils resonance frequency set by the resonance capacitors

VS=1;                               % Tx voltage peak amplitude (when voltage source is used in the Tx), in volts
IS=50e-3;                           % Tx current peak amplitude (when current source is used in the Tx), in amperes

%% CALCULATIONS %%

etaRX=QRX/(QRX+QL);                                                             % Rx efficiency
ZRXTXref=(kTXRX.^2*QTX*QRX*QL)'*(1./(QRX+QL+j*QRX*QL*(1-(wres./w).^2)))*RTX;    % Normalized reflected impedance in the Tx (ZRx-Rxref/RTx)
etaTX=real(ZRXTXref)./(RTX+real(ZRXTXref));                                     % Tx efficiency
eta_tot=etaRX*etaTX;                                                            % Power Transfer Efficiency (PTE)
Zs=RTX+ZRXTXref+ones(length(kTXRX),1)*j*RTX*QTX*(1-(wres./w).^2);               % Tx driver load impedance 
PTX_voltage=0.5*(VS./abs(Zs)).^2.*(real(Zs));                                   % Voltage driver output power
PTX_current=0.5*(IS^2)*(real(Zs));                                              % Current driver output power
PMN_voltage=PTX_voltage.*eta_tot;                                               % Power Delivered to the Load (PDL) with a Voltage Tx driver 
PMN_current=PTX_current.*eta_tot;                                               % Power Delivered to the Load (PDL) with a Current Tx driver 

%% PLOTS %%

%Received power with a Voltage Tx driver%
figure(1)
h=surf(kTXRX,(w/(2*pi))*1e-6,PMN_voltage'*1e3);
hold on
xlabel('k_{Tx-Rx}')
ylabel('f (MHz)')
zlabel('P_{MN} (mW)')
ylim([12.5 14.5])
xlim([0 0.05])
view(135,25)
set(gca,'XTick',[0 0.01 0.02 0.03 0.04 0.05])
set(gca,'YTick',[12.5 13 13.56 14 14.5])

%Received power with a Current Tx driver%
figure(2)                                       
h=surf(kTXRX,(w/(2*pi))*1e-6,PMN_current'*1e3);
hold on
xlabel('k_{Tx-Rx}')
ylabel('f (MHz)')
zlabel('P_{MN} (mW)')
ylim([12.5 14.5])
xlim([0 0.05])
view(135,25)
set(gca,'XTick',[0 0.01 0.02 0.03 0.04 0.05])
set(gca,'YTick',[12.5 13 13.56 14 14.5])

%Link efficiency (PTE)%
figure(3)
h=surf(kTXRX,(w/(2*pi))*1e-6,eta_tot'*100);
hold on
xlabel('k_{Tx-Rx}')
ylabel('f (MHz)')
zlabel('\eta_{Link} (%)')
ylim([12.5 14.5])
xlim([0 0.05])
view(135,25)
set(gca,'XTick',[0 0.01 0.02 0.03 0.04 0.05])
set(gca,'YTick',[12.5 13 13.56 14 14.5])


%% Highlight selected points %%

LTX=RTX*QTX/wres;
CTX=1/(LTX*wres^2);
LRX=LTX;
MTXRX=kTXRX*sqrt(LTX*LRX);

%%%%I%%%%%
disp('-----------I-----------')
kTXRX_I=0.05;                                                                                           % Tx-Rx coupling coefficient, selected in over-coupled region
w_I=1/sqrt(CTX*(LTX));                                                                                  % Carrier frequency, selected equal to the resonant frequency of each individual coil
Zs=RTX*(1+(kTXRX_I^2*QTX*QRX*QL)*(1/(QRX+QL+j*QRX*QL*(1-(wres/w_I)^2)))+j*QTX*(1-(wres/w_I)^2))         % Tx driver load impedance
PMN_voltage_I = interp2(kTXRX,w,PMN_voltage',kTXRX_I,w_I)                                               % PDL in case I
eta_tot_I= interp2(kTXRX,w,eta_tot',kTXRX_I,w_I)                                                        % PTE in case I
%Highlight I in PDL figure%
figure(1)
plot3(kTXRX_I,(w_I/(2*pi))*1e-6,PMN_voltage_I*1e3,'*r','markersize',10)
h=text(kTXRX_I,(w_I/(2*pi))*1e-6,PMN_voltage_I*1e3-1.5,'I');
set(h, 'fontweight','bold','fontsize',30,'color', [1 0 0])
%Highlight I in PTE figure%
figure(3)
plot3(kTXRX_I,(w_I/(2*pi))*1e-6,eta_tot_I*100,'*r','markersize',10)
h=text(kTXRX_I,(w_I/(2*pi))*1e-6,eta_tot_I*100+5,'I=III');
set(h, 'fontweight','bold','fontsize',30,'color', [1 0 0])

%%%%I%%%%%
disp('-----------II----------')
kTXRX_II=0.05;                                                                                          % Tx-Rx coupling coefficient, selected in over-coupled region
MTXRX=kTXRX_II*sqrt(LTX*LRX);                                                                           % Mutual inductance used to calculate the carrier frequency that maximizes PDL
w_II=1/sqrt(CTX*(LTX+MTXRX));                                                                           % Carrier frequency, selected equal to one of the natural frequencies of the system
Zs=RTX*(1+(kTXRX_II^2*QTX*QRX*QL)*(1/(QRX+QL+j*QRX*QL*(1-(wres/w_II)^2)))+j*QTX*(1-(wres/w_II)^2))      % Tx driver load impedance
PMN_voltage_II = interp2(kTXRX,w,PMN_voltage',kTXRX_II,w_II)                                            % PDL in case II
eta_tot_II= interp2(kTXRX,w,eta_tot',kTXRX_II,w_II)                                                     % PTE in case II
%Highlight I in PDL figure%
figure(1)
plot3(kTXRX_II,(w_II/(2*pi))*1e-6,PMN_voltage_II*1e3,'*r','markersize',10)
h=text(kTXRX_II,(w_II/(2*pi))*1e-6,PMN_voltage_II*1e3+1.5,'II');
set(h, 'fontweight','bold','fontsize',30,'color', [1 0 0])
%Highlight I in PTE figure%
figure(3)
plot3(kTXRX_II,(w_II/(2*pi))*1e-6,eta_tot_II*100,'*r','markersize',10)
h=text(kTXRX_II,(w_II/(2*pi))*1e-6,eta_tot_II*100+5,'II');
set(h, 'fontweight','bold','fontsize',30,'color', [1 0 0])

%%%%III%%%%%
disp('-----------III---------')
kTXRX_III=0.05;                                                                                         % Tx-Rx coupling coefficient, selected in over-coupled region
w_III=1/sqrt(CTX*(LTX));                                                                                % Carrier frequency, selected equal to the resonant frequency of each individual coil
VS_III=1.74;                                                                                            % Tx voltage
Zs=RTX*(1+(kTXRX_III^2*QTX*QRX*QL)*(1/(QRX+QL+j*QRX*QL*(1-(wres/w_III)^2)))+j*QTX*(1-(wres/w_III)^2))   % Tx driver load impedance
eta_tot_III= interp2(kTXRX,w,eta_tot',kTXRX_III,w_III)                                                  % PTE in case III
PTX_voltage_III=0.5*(VS_III./abs(Zs)).^2.*(real(Zs));                                                   % Voltage driver output power
PMN_voltage_III=PTX_voltage_III.*eta_tot_III                                                            % PDL in case III
%Highlight III in PDL figure%
figure(1)
plot3(kTXRX_III,(w_III/(2*pi))*1e-6,PMN_voltage_III*1e3,'*r','markersize',10)
h=text(kTXRX_III,(w_III/(2*pi))*1e-6,PMN_voltage_III*1e3+1.5,'III');
set(h, 'fontweight','bold','fontsize',30,'color', [1 0 0])
%Highlight III in PTE figure%
%This is commented because I=III, and I is already highlighted
%figure(3)
%plot3(kTXRX_III,(w_III/(2*pi))*1e-6,eta_tot_III*100,'*r','markersize',10)
%h=text(kTXRX_III,(w_III/(2*pi))*1e-6,eta_tot_III*100+5,'III');
%set(h, 'fontweight','bold','fontsize',30,'color', [1 0 0])

