###############################################################################
## Set-up
###############################################################################

## First, copy path to "~/btw". Then:
# setwd(gsub("\\\\", "/", readClipboard()))

## Load packages
## Save package names as a vector of strings
pkgs <- c("plyr", "DescTools")

## Install uninstalled packages
lapply(pkgs[!(pkgs %in% installed.packages())], install.packages,
       repos = 'https://cran.uni-muenster.de/')

## Load all packages to library and adjust options
lapply(pkgs, library, character.only = TRUE)


###############################################################################
## Source functions
###############################################################################

source("fun/mround.R")

###############################################################################
## Data
###############################################################################

## Load pre-processed GLES data 
load("data/gles.RData")


###############################################################################
## Auxiliary objects
###############################################################################

## True election results
vts13 <- c(.047, .415, .015, .048, .084, .086, .022 + .013 + .028 - .015, .257)
vts13 <- vts13[c(1, 3, 2, 8, 6, 4, 5, 7)]
vts13 <- c(vts13, 1 - .715)
vts17 <- c(.126, .329, .004, .107, .089, .092, .05 - .004, .205)
vts17 <- vts17[c(1, 3, 2, 8, 6, 4, 5, 7)]
vts17 <- c(vts17, 1 - .762)
lbls <- c("AfD", "Rechte", "CDU/CSU", "SPD", "Linke", "FDP", "Grne",
          "Sonstige", "Nichtwahl")

## Party colors
col.afd <- "#009ee0"
col.cdu <- "#000000"
col.frp <- "#8b4726"
col.fdp <- "#ffff00"
col.gre <- "#1EA12D"
col.lef <- "#BE3075"
col.spd <- "#E3000F"
col.oth <- "#aaaaaa"
col.non <- "#ffffff"
cols <- c(col.afd, col.frp, col.cdu, col.spd, col.lef, col.fdp, col.gre,
          col.oth, col.non)
g.col.afd <- ColToGray("#009ee0")
g.col.cdu <- ColToGray("#000000")
g.col.frp <- ColToGray("#8b4726")
g.col.fdp <- ColToGray("#ffff00")
g.col.gre <- ColToGray("#1EA12D")
g.col.lef <- ColToGray("#BE3075")
g.col.spd <- ColToGray("#E3000F")
g.col.oth <- ColToGray("#aaaaaa")
g.col.non <- ColToGray("#ffffff")
g.cols <- ColToGray(cols)

# x-axis values
tt <- seq(0, 38, 2)
log.tt <- log(tt + 1)
tt.lab <- c(0, 1, 5, 10, 20, 30, 40)
tt.loc <- log(tt.lab + 1)


###############################################################################
## Figures
###############################################################################

## Figure 1 (Grayscale)
load("est/m0.RData")
tiff(filename = "fig/fig1_gsc.tiff",
     width = 9, 
     height = 6, 
     units = "in",
     pointsize = 12,
     compression = "lzw" ,
     res = 1200,
     family = "serif", 
     restoreConsole = TRUE)
par(family = 'serif',
    mar = c(5.1, 5.1, 1.1, 1.1))
plot(x = tt,
     y = tt,
     type = 'n',
     ylim = c(0, 1),
     xlim = range(log.tt),
     main = "",
     ylab = "Wahrscheinlichkeit der Angst vor Stellen-/Betriebsverlust",
     xlab = "konomisches Risiko (Log-Skala)",
     axes = F)
axis(2)
axis(1, labels = tt.lab, at = tt.loc) 
abline(h = seq(0, 1, .2),
       lwd = .5,
       col = "gray70")
polygon(c(log.tt, rev(log.tt)),
        c(m0.0[2, ], rev(m0.0[3, ])),
        col = adjustcolor("gray40", alpha.f = .4), 
        border = NA)
polygon(c(log.tt, rev(log.tt)),
        c(m0.1[2, ], rev(m0.1[3, ])),
        col = adjustcolor("gray40", alpha.f = .4), 
        border = NA)
lines(log.tt, m0.0[1, ], lty = 1)
lines(log.tt, m0.1[1, ], lty = 2)
abline(h = 0,
       lwd = .5,
       col = "black")
legend("topright",
       bty = 'n',
       legend = c("Erwerbsttige", "Arbeitslose"),
       lty = c(1, 2))
dev.off()


## Figure 2 (color)
load("est/m2.RData")
m2 <- m2$pr.yx
tiff(filename = "fig/fig2_col.tiff",
     width = 12, 
     height = 8, 
     units = "in",
     pointsize = 12,
     compression = "lzw" ,
     res = 1200,
     family = "serif", 
     restoreConsole = TRUE)
par(family = 'serif',
    mfrow = c(2, 2))
titles <- c("Erwerbsttig", "Arbeitslos", "In Ausbildung", "Inaktiv")
for (d in 1:4) {
  plot(x = vts17,
       y = vts17,
       type = 'n',
       ylim = c(0, .35),
       xlim = c(0, 9),
       main = titles[d],
       ylab = "",
       xlab = "",
       axes = F)
  axis(2)
  axis(1, labels = rep("", 10), at = 0:9) 
  text(x = seq(.5, 8.5, 1),
       y = rep(- .025, 9),
       xpd = TRUE,
       labels = lbls, 
       cex = 0.8)
  abline(h = seq(0, .35, .05),
         lwd = .5,
         col = "gray85")
  for (j in seq_along(lbls)) {
    rect(j - .7,
         m2[2, j, d],
         j - .3,
         m2[3, j, d],
         col = adjustcolor(cols[j], alpha.f = .45),
         border = ifelse(cols[j] == col.non, "gray50", 
                         ifelse(cols[j] == col.fdp, adjustcolor("orange", alpha.f = .5),
                                adjustcolor(cols[j], alpha.f = .5))))
    points(j - .5,
           m2[1, j, d],
           pch = 21,
           bg = ifelse(cols[j] == col.fdp,
                       adjustcolor("orange", alpha.f = .5),
                       adjustcolor(cols[j], alpha.f = 1)),
           col = ifelse(cols[j] == col.non, "gray50", col.non),
           cex = 1.25)
  }
  abline(h = 0,
         lwd = .5,
         col = "black") 
}
dev.off()

## Figure 2 (grayscale)
tiff(filename = "fig/fig2_gsc.tiff",
     width = 12, 
     height = 8, 
     units = "in",
     pointsize = 12,
     compression = "lzw" ,
     res = 1200,
     family = "serif", 
     restoreConsole = TRUE)
par(family = 'serif',
    mfrow = c(2, 2))
titles <- c("Erwerbsttig", "Arbeitslos", "In Ausbildung", "Inaktiv")
for (d in 1:4) {
  plot(x = vts17,
       y = vts17,
       type = 'n',
       ylim = c(0, .35),
       xlim = c(0, 9),
       main = titles[d],
       ylab = "",
       xlab = "",
       axes = F)
  axis(2)
  axis(1, labels = rep("", 10), at = 0:9) 
  text(x = seq(.5, 8.5, 1),
       y = rep(- .025, 9),
       xpd = TRUE,
       labels = lbls, 
       cex = 0.8)
  abline(h = seq(0, .35, .05),
         lwd = .5,
         col = "gray85")
  for (j in seq_along(lbls)) {
    rect(j - .7,
         m2[2, j, d],
         j - .3,
         m2[3, j, d],
         col = adjustcolor(g.cols[j], alpha.f = .45),
         border = ifelse(g.cols[j] == g.col.non, "gray50", 
                         ifelse(g.cols[j] == g.col.fdp, 
                                ColToGray(adjustcolor("orange", alpha.f = .5)),
                                adjustcolor(g.cols[j], alpha.f = .5))))
    points(j - .5,
           m2[1, j, d],
           pch = 21,
           bg = ifelse(g.cols[j] == g.col.fdp,
                       ColToGray(adjustcolor("orange", alpha.f = .5)),
                       adjustcolor(g.cols[j], alpha.f = 1)),
           col = ifelse(g.cols[j] == g.col.non, "gray50", g.col.non),
           cex = 1.25)
  }
  abline(h = 0,
         lwd = .5,
         col = "black") 
}
dev.off()


## Figure 3 (color)
m2by2 <- cbind(replicate(4, c(rep(1, 4), 2)), c(rep(3, 4), 4))
load("est/m3.RData")
tt <- seq(0, 38, 2)
log.tt <- log(tt + 1)
tt.lab <- c(0, 1, 5, 10, 20, 30, 40)
tt.loc <- log(tt.lab + 1)

tiff(filename = "fig/fig3_col.tiff",
     width = 8, 
     height = 5, 
     units = "in",
     pointsize = 14,
     compression = "lzw" ,
     res = 1200,
     family = "serif", 
     restoreConsole = TRUE)
## Layout
layout(m2by2)

## Stacked Probabilities
par(mar = c(0.1, 5.1, 0.1, 0.1), 
    family = "serif")
plot(tt, tt, 
     type = 'n', 
     main = "", 
     axes = F,
     xlab = "",
     ylab = "Vorhergesagte kumulative Stimmanteile",
     ylim = c(0, 1),
     xlim = range(log.tt))
axis(2, outer = FALSE)
box()
for (j in 1:dim(m3$pr.stk)[2]) {
  if (j == dim(m3$pr.stk)[2]) {
    polygon(c(log.tt, rev(log.tt)),
            c(rep(1, length(log.tt)),
              rev(m3$pr.stk[1, j, ])),
            col = adjustcolor(cols[j], alpha.f = .25), 
            border = NA)
  } else {
    polygon(c(log.tt, rev(log.tt)),
            c(m3$pr.stk[1, j + 1, ],
              rev(m3$pr.stk[1, j, ])),
            col = adjustcolor(cols[j], alpha.f = .5), 
            border = NA)
  }
}
for (j in 1:dim(m3$pr.stk)[2]) {
  polygon(c(log.tt, rev(log.tt)),
          c(m3$pr.stk[2, j, ],
            rev(m3$pr.stk[3, j, ])),
          col = adjustcolor("white", alpha.f = 0.25), 
          border = NA)
  lines(log.tt, m3$pr.stk[1, j, ], 
        col = adjustcolor("white", alpha.f = 0.75),
        lwd = 0.5)
}
rect(min(log.tt),
     0,
     max(log.tt),
     1,
     col = NA,
     border = "black")

## Histogram
par(mar = c(5.1, 5.1, 0.1, 0.1))
hist(gles.act[[1]]$risk, 
     axes = F, 
     xlab = "konomisches Risiko (Log-Skala)", 
     ylab = "", 
     main = "", 
     xlim = c(0, max(gles.act[[1]]$risk)),
     breaks = 100,
     border = adjustcolor("gray40", alpha.f = 0.5),
     col = adjustcolor("gray40", alpha.f = 0.25))
box()
axis(1, labels = tt.lab, at = tt.loc, outer = FALSE)
axis(2, outer = FALSE)

## Marginal Effects
par(mar = c(0.1, 0.1, 0.1, 5.1))
plot(1:9, 1:9, 
     type = 'n', 
     main = "", 
     axes = F,
     xlab = "",
     ylab = "",
     ylim = c(0, 1),
     xlim = mround(range(m3$dy.dx), .01))
box()
axis(1)
abline(v = 0,
       col = "gray50",
       lwd = .5)
pos <- NULL
for (k in 2:dim(m3$pr.stk)[2]) {
  pos <- c(pos, mean(c(m3$pr.stk[1, k,     length(log.tt)],
                       m3$pr.stk[1, k - 1, length(log.tt)])))
}
pos <- c(pos, mean(c(pos[length(pos)], 1)))
abline(h = pos,
       lty = 3,
       lwd = .5,
       col = adjustcolor(cols, alpha.f = .5))
segments(m3$dy.dx[2, ],
         pos,
         m3$dy.dx[3, ],
         pos,
         col = "black")
points(m3$dy.dx[1, ],
       pos,
       pch = 21,
       bg = adjustcolor(cols, alpha.f = .75),
       col = "black",
       cex = 1.25)
axis(2,
     side = 4,
     labels = lbls,
     at = pos,
     las = 2,
     cex = .75,
     outer = FALSE)
dev.off()

## Figure 3 (grayscale)
tiff(filename = "fig/fig3_gsc.tiff",
     width = 8, 
     height = 5, 
     units = "in",
     pointsize = 14,
     compression = "lzw" ,
     res = 1200,
     family = "serif", 
     restoreConsole = TRUE)
## Layout
layout(m2by2)

## Stacked Probabilities
par(mar = c(0.1, 5.1, 0.1, 0.1), 
    family = "serif")
plot(tt, tt, 
     type = 'n', 
     main = "", 
     axes = F,
     xlab = "",
     ylab = "Vorhergesagte kumulative Stimmanteile",
     ylim = c(0, 1),
     xlim = range(log.tt))
axis(2, outer = FALSE)
box()
for (j in 1:dim(m3$pr.stk)[2]) {
  if (j == dim(m3$pr.stk)[2]) {
    polygon(c(log.tt, rev(log.tt)),
            c(rep(1, length(log.tt)),
              rev(m3$pr.stk[1, j, ])),
            col = adjustcolor(g.cols[j], alpha.f = .25), 
            border = NA)
  } else {
    polygon(c(log.tt, rev(log.tt)),
            c(m3$pr.stk[1, j + 1, ],
              rev(m3$pr.stk[1, j, ])),
            col = adjustcolor(g.cols[j], alpha.f = .5), 
            border = NA)
  }
}
for (j in 1:dim(m3$pr.stk)[2]) {
  polygon(c(log.tt, rev(log.tt)),
          c(m3$pr.stk[2, j, ],
            rev(m3$pr.stk[3, j, ])),
          col = adjustcolor("white", alpha.f = 0.25), 
          border = NA)
  lines(log.tt, m3$pr.stk[1, j, ], 
        col = adjustcolor("white", alpha.f = 0.75),
        lwd = 0.5)
}
rect(min(log.tt),
     0,
     max(log.tt),
     1,
     col = NA,
     border = "black")

## Histogram
par(mar = c(5.1, 5.1, 0.1, 0.1))
hist(gles.act[[1]]$risk, 
     axes = F, 
     xlab = "konomisches Risiko (Log-Skala)", 
     ylab = "", 
     main = "", 
     xlim = c(0, max(gles.act[[1]]$risk)),
     breaks = 100,
     border = adjustcolor("gray40", alpha.f = 0.5),
     col = adjustcolor("gray40", alpha.f = 0.25))
box()
axis(1, labels = tt.lab, at = tt.loc, outer = FALSE)
axis(2, outer = FALSE)

## Marginal Effects
par(mar = c(0.1, 0.1, 0.1, 5.1))
plot(1:9, 1:9, 
     type = 'n', 
     main = "", 
     axes = F,
     xlab = "",
     ylab = "",
     ylim = c(0, 1),
     xlim = mround(range(m3$dy.dx), .01))
box()
axis(1)
abline(v = 0,
       col = "gray50",
       lwd = .5)
pos <- NULL
for (k in 2:dim(m3$pr.stk)[2]) {
  pos <- c(pos, mean(c(m3$pr.stk[1, k,     length(log.tt)],
                       m3$pr.stk[1, k - 1, length(log.tt)])))
}
pos <- c(pos, mean(c(pos[length(pos)], 1)))
abline(h = pos,
       lty = 3,
       lwd = .5,
       col = adjustcolor(g.cols, alpha.f = .5))
segments(m3$dy.dx[2, ],
         pos,
         m3$dy.dx[3, ],
         pos,
         col = "black")
points(m3$dy.dx[1, ],
       pos,
       pch = 21,
       bg = adjustcolor(g.cols, alpha.f = .75),
       col = "black",
       cex = 1.25)
axis(2,
     side = 4,
     labels = lbls,
     at = pos,
     las = 2,
     cex = .75,
     outer = FALSE)
dev.off()


## Figure 4 (color)
load("est/med_risk.RData")
load("est/med_unemp.RData")
tiff(filename = "fig/fig4_col.tiff",
     width = 8, 
     height = 5, 
     units = "in",
     pointsize = 12,
     compression = "lzw" ,
     res = 1200,
     family = "serif", 
     restoreConsole = TRUE)
par(family = 'serif',
    mfrow = c(2, 1),
    mar = c(3.1, 5.1, 1.1, 1.1))
y.lim <- mround(range(risk.ed[1:5]), .05)
plot(x = 1:9,
     y = 1:9,
     type = 'n',
     ylim = y.lim,
     xlim = c(0, 9),
     main = "",
     ylab = "",
     xlab = "",
     axes = F)
title(ylab = "Effekte log. konomischen Risikos", cex.lab = .7)
axis(2)
axis(1, labels = rep("", 10), at = 0:9) 
text(x = seq(.5, 8.5, 1),
     y = rep(y.lim[1] - .2 * abs(y.lim[1]), 9),
     xpd = TRUE,
     labels = lbls, 
     cex = 0.8)
abline(h = seq(y.lim[1], y.lim[2], .025),
       lwd = .5,
       col = "gray85")
abline(v = 0:9,
       lwd = .5,
       col = "gray85")
abline(h = 0,
       lwd = .5,
       lty = 2,
       col = "gray20")
for (j in seq_along(lbls)) {
  rect(j - .9,
       risk.ed$te[2, j],
       j - .1,
       risk.ed$te[3, j],
       col = adjustcolor(cols[j], alpha.f = .25),
       border = ifelse(cols[j] == col.non, "gray50", 
                       ifelse(cols[j] == col.fdp, adjustcolor("orange", alpha.f = .5),
                              adjustcolor(cols[j], alpha.f = .5))))
  segments(j - .9,
           risk.ed$te[1, j],
           j - .1,
           risk.ed$te[1, j],
           col = ifelse(cols[j] == col.non, "gray50", 
                        ifelse(cols[j] == col.fdp, adjustcolor("orange", alpha.f = .75),
                               adjustcolor(cols[j], alpha.f = .75))))
  stubs <- c("DE", "IM", "UV", "UZ")
  hspace <- c(.8, .6, .4, .2)
  for (e in 1:4) {
    segments(j - hspace[e],
             risk.ed[[e + 1]][2, j],
             j - hspace[e],
             risk.ed[[e + 1]][3, j])
    points(j - hspace[e],
           risk.ed[[e + 1]][1, j],
           pch = 19,
           cex = .6)
  }
}
abline(h = y.lim[1],
       lwd = .5,
       col = "black")

y.lim <- mround(range(unemp.ed[1:5]), .05)
plot(x = 1:9,
     y = 1:9,
     type = 'n',
     ylim = y.lim,
     xlim = c(0, 9),
     main = "",
     ylab = "",
     xlab = "",
     axes = F)
axis(2)
title(ylab = "Effekte akuter Arbeitslosigkeit", cex.lab = .8)
axis(1, labels = rep("", 10), at = 0:9) 
text(x = seq(.5, 8.5, 1),
     y = rep(y.lim[1] - .2 * abs(y.lim[1]), 9),
     xpd = TRUE,
     labels = lbls, 
     cex = 0.8)
abline(h = seq(y.lim[1], y.lim[2], .05),
       lwd = .5,
       col = "gray85")
abline(v = 0:9,
       lwd = .5,
       col = "gray85")
abline(h = 0,
       lwd = .5,
       lty = 2,
       col = "gray20")
for (j in seq_along(lbls)) {
  rect(j - .9,
       unemp.ed$te[2, j],
       j - .1,
       unemp.ed$te[3, j],
       col = adjustcolor(cols[j], alpha.f = .25),
       border = ifelse(cols[j] == col.non, "gray50", 
                       ifelse(cols[j] == col.fdp, adjustcolor("orange", alpha.f = .5),
                              adjustcolor(cols[j], alpha.f = .5))))
  segments(j - .9,
           unemp.ed$te[1, j],
           j - .1,
           unemp.ed$te[1, j],
           col = ifelse(cols[j] == col.non, "gray50", 
                        ifelse(cols[j] == col.fdp, adjustcolor("orange", alpha.f = .75),
                               adjustcolor(cols[j], alpha.f = .75))))
  stubs <- c("DE", "IM", "UV", "UZ")
  hspace <- c(.8, .6, .4, .2)
  for (e in 1:4) {
    segments(j - hspace[e],
             unemp.ed[[e + 1]][2, j],
             j - hspace[e],
             unemp.ed[[e + 1]][3, j])
    points(j - hspace[e],
           unemp.ed[[e + 1]][1, j],
           pch = 19,
           cex = .6)
  }
}
abline(h = y.lim[1],
       lwd = .5,
       col = "black")
dev.off()

## Figure 4 (grayscale)
tiff(filename = "fig/fig4_gsc.tiff",
     width = 8, 
     height = 5, 
     units = "in",
     pointsize = 12,
     compression = "lzw" ,
     res = 1200,
     family = "serif", 
     restoreConsole = TRUE)
par(family = 'serif',
    mfrow = c(2, 1),
    mar = c(3.1, 5.1, 1.1, 1.1))
y.lim <- mround(range(risk.ed[1:5]), .05)
plot(x = 1:9,
     y = 1:9,
     type = 'n',
     ylim = y.lim,
     xlim = c(0, 9),
     main = "",
     ylab = "",
     xlab = "",
     axes = F)
title(ylab = "Effekte log. konomischen Risikos", cex.lab = .7)
axis(2)
axis(1, labels = rep("", 10), at = 0:9) 
text(x = seq(.5, 8.5, 1),
     y = rep(y.lim[1] - .2 * abs(y.lim[1]), 9),
     xpd = TRUE,
     labels = lbls, 
     cex = 0.8)
abline(h = seq(y.lim[1], y.lim[2], .025),
       lwd = .5,
       col = "gray85")
abline(v = 0:9,
       lwd = .5,
       col = "gray85")
abline(h = 0,
       lwd = .5,
       lty = 2,
       col = "gray20")
for (j in seq_along(lbls)) {
  rect(j - .9,
       risk.ed$te[2, j],
       j - .1,
       risk.ed$te[3, j],
       col = adjustcolor(g.cols[j], alpha.f = .25),
       border = ifelse(g.cols[j] == col.non, "gray50", 
                       ifelse(g.cols[j] == col.fdp, adjustcolor("orange", alpha.f = .5),
                              adjustcolor(g.cols[j], alpha.f = .5))))
  segments(j - .9,
           risk.ed$te[1, j],
           j - .1,
           risk.ed$te[1, j],
           col = ifelse(g.cols[j] == col.non, "gray50", 
                        ifelse(g.cols[j] == col.fdp, adjustcolor("orange", alpha.f = .75),
                               adjustcolor(g.cols[j], alpha.f = .75))))
  stubs <- c("DE", "IM", "UV", "UZ")
  hspace <- c(.8, .6, .4, .2)
  for (e in 1:4) {
    segments(j - hspace[e],
             risk.ed[[e + 1]][2, j],
             j - hspace[e],
             risk.ed[[e + 1]][3, j])
    points(j - hspace[e],
           risk.ed[[e + 1]][1, j],
           pch = 19,
           cex = .6)
  }
}
abline(h = y.lim[1],
       lwd = .5,
       col = "black")

y.lim <- mround(range(unemp.ed[1:5]), .05)
plot(x = 1:9,
     y = 1:9,
     type = 'n',
     ylim = y.lim,
     xlim = c(0, 9),
     main = "",
     ylab = "",
     xlab = "",
     axes = F)
axis(2)
title(ylab = "Effekte akuter Arbeitslosigkeit", cex.lab = .8)
axis(1, labels = rep("", 10), at = 0:9) 
text(x = seq(.5, 8.5, 1),
     y = rep(y.lim[1] - .2 * abs(y.lim[1]), 9),
     xpd = TRUE,
     labels = lbls, 
     cex = 0.8)
abline(h = seq(y.lim[1], y.lim[2], .05),
       lwd = .5,
       col = "gray85")
abline(v = 0:9,
       lwd = .5,
       col = "gray85")
abline(h = 0,
       lwd = .5,
       lty = 2,
       col = "gray20")
for (j in seq_along(lbls)) {
  rect(j - .9,
       unemp.ed$te[2, j],
       j - .1,
       unemp.ed$te[3, j],
       col = adjustcolor(g.cols[j], alpha.f = .25),
       border = ifelse(g.cols[j] == col.non, "gray50", 
                       ifelse(g.cols[j] == col.fdp, adjustcolor("orange", alpha.f = .5),
                              adjustcolor(g.cols[j], alpha.f = .5))))
  segments(j - .9,
           unemp.ed$te[1, j],
           j - .1,
           unemp.ed$te[1, j],
           col = ifelse(g.cols[j] == col.non, "gray50", 
                        ifelse(g.cols[j] == col.fdp, adjustcolor("orange", alpha.f = .75),
                               adjustcolor(g.cols[j], alpha.f = .75))))
  stubs <- c("DE", "IM", "UV", "UZ")
  hspace <- c(.8, .6, .4, .2)
  for (e in 1:4) {
    segments(j - hspace[e],
             unemp.ed[[e + 1]][2, j],
             j - hspace[e],
             unemp.ed[[e + 1]][3, j])
    points(j - hspace[e],
           unemp.ed[[e + 1]][1, j],
           pch = 19,
           cex = .6)
  }
}
abline(h = y.lim[1],
       lwd = .5,
       col = "black")
dev.off()