      SUBROUTINE CFSPTR (N, XN, FN, W, MS, PX, PY, A, B, C, D, PHIN, R, &
      PHIR, AUXF, IERR)                                                 
!                                                                       
!*****************************************************************      
!                                                                *      
!  CFSPTR computes the coefficients A(I), B(I), C(I), D(I) for   *      
!  I=0, 1, ..., N-1 of a transformed parametric cubic fitting    *      
!  spline for a closed smooth curve K.                           *      
!  The program initially uses the given points XN(I),FN(I) to    *      
!  compute the transformed coordinates PHIN(I),R(I) for I=0,...,N*      
!  which serve as nodes for computing a nonparametric periodic   *      
!  fitting spline S(PHI).                                        *      
!  S(PHI) is represented in the form:                            *      
!                                                                *      
!  S(PHI) = A(I) + B(I)(PHI-PHIN(I)) + C(I)(PHI-PHIN(I))**2 +    *      
!                                    + D(I)(PHI-PHIN(I))**3      *      
!                                                                *      
!  for PHI in the interval [PHIN(I),PHIN(I+1)], I=0,1,...,N-1.   *      
!                                                                *      
!  Since the transformed nodes PHIN(I) must be ordered strictly  *      
!  monotonically, we must shift the origin to a point P = (PX,PY)*      
!  and rotate the plane by an angle PHIR depending on the        *      
!  original location of the points XN(I),FN(I) in order to       *      
!  compute PHIN(I).                                              *      
!  In order to be able to achieve this strict momotonicity of the*      
!  PHIN(I) the following conditions must be satisfied by the     *      
!  original points:                                              *      
!   - the new origin P must be situated on the surface F         *      
!     generated by the points XN(I),FN(I) so that every polar    *      
!     ray from P cuts the boundary curve of F only once.         *      
!     (The coordinates (PX,PY) of P can be specified by the      *      
!     user, refer to input parameter MS.)                        *      
!   - the points XN(I),FN(I) must be ordered counterclockwise    *      
!     on the boundary curve of F, beginning with XN(0),FN(0).    *      
!     If periodicity of S(PHI) is desired, the end point         *      
!     XN(N),FN(N) must be equal to the starting point XN(0),FN(0)*      
!  The coordinates of PHIN(I),R(I) are computed as follows:      *      
!         PHIN(0) = 0,                                           *      
!         PHIN(I) = DATAN(Y'(I)/X'(I)) - PHIR , I=1, ..., N-1,   *      
!         PHIN(N) = 2*PI,                                        *      
!         R(I) = DSQRT(X'(I)**2 + Y'(I)**2), I=0, 1, ..., N,     *      
!         where:  PHIR = DATAN(FN(0)/XN(0)),                     *      
!                 X'(I) = XN(I) - PX,  Y'(I) = FN(I) - PY.       *      
!                                                                *      
!                                                                *      
!  REMARK:  The curve K can be found using subroutine TSPANA.    *      
!  =======  To find a table of values for K one can use the      *      
!           subroutine TSPTAB.                                   *      
!           Both subroutines use the values for PX, PY and       *      
!           PHIR as computed in this program in order to         *      
!           recompute the coordinates.                           *      
!                                                                *      
!                                                                *      
!  ASSUMPTIONS:    1.         N > 5                              *      
!  ============    2.     XN(0) = XN(N)                          *      
!                  3.     FN(0) = FN(N)                          *      
!                  4.      W(I) > 0.0   , I=0, 1, ..., N         *      
!                  5.      W(0) = W(N)                           *      
!                                                                *      
!                                                                *      
!  INPUT PARAMETERS:                                             *      
!  =================                                             *      
!  N  :  Index of the last node                                  *      
!  XN :  vector XN(0:N); the nodes XN(I) , I = 0, 1, ..., N      *      
!  FN :  vector FN(0:N); the data points FN(I) at XN(I)          *      
!  W  :  vector W(0:N);  the weights W(I) of the data points     *      
!                        FN(I)                                   *      
!                                                                *      
!  MS :  label for the shift or translation of the origin:       *      
!        MS > 0 : the user specifies the coordinates PX,PY       *      
!        MS = 0 : no translation, i.e., PX = PY = 0              *      
!        MS < 0 : the translation coordinates PX,PY are computed *      
!                 by CFSPTR as follows:                          *      
!                    PX = (XMAX+XMIN)/2                          *      
!                    PY = (YMAX+YMIN)/2                          *      
!                    where : XMAX = MAX(XN(I)) ]                 *      
!                            XMIN = MIN(XN(I)) ] for I=0, 1, ...,*      
!                            YMAX = MAX(FN(I)) ]       N         *      
!                            YMIN = MIN(FN(I)) ]                 *      
!                 NOTE: This does not insure that P will satisfy *      
!                       the conditions mentioned earlier, i.e.,  *      
!                       IERR = -4 might occur.                   *      
!                                                                *      
!  PX : ]  coordinaten of P                                      *      
!  PY : ]  (for MS > 0)                                          *      
!                                                                *      
!                                                                *      
!  AUXILIARY VARIABLES:                                          *      
!  ====================                                          *      
!  AUXF :  vector AUXF(1:14*N-10)                                *      
!                                                                *      
!                                                                *      
!  OUTPUT PARAMETERS:                                            *      
!  ==================                                            *      
!  A    :  Vector A(0:N) ]  The entries in positions 0 to N-1    *      
!  B    :  Vector B(0:N) ]  contain the spline coefficients for  *      
!  C    :  Vector C(0:N) ]  S(PHI). The entries in A(N), B(N),   *      
!  D    :  Vector D(0:N) ]  C(N) and D(N) are auxiliary          *      
!                        ]  variables.                           *      
!                                                                *      
!  PHIN :  vector PHIN(0:N)   PHIN(I) measures the polar angle of*      
!                             the point (XN(I),FN(I)) from the   *      
!                             center P=(PX,PY)                   *      
!  R    :  vector R(0:N);  R(I) is the length of the vector      *      
!                          joining P and PHIN(I)                 *      
!  PX   : ] coordinates of the center P                          *      
!  PY   : ]                                                      *      
!  PHIR :  angle in radian measure of the rotation used          *      
!  IERR :  error parameter                                       *      
!          =  0 :  All is o.k.                                   *      
!          = -1 :  N < 6                                         *      
!          = -3 :  Inadmissable weight W                         *      
!          = -4 :  The nodes PHIN(I)are not monotonically ordered*      
!                  PHIN(I) >= PHIN(I+1) for some I=0,1,...,N-1   *      
!          = -5 :  XN(0) not equal to XN(N)  or                  *      
!                  FN(0) not equal to FN(N)  or                  *      
!                   W(0) not equal to W(N)                       *      
!          =  1 :  default in NCYFSY (system matrix is           *      
!                  numerically singular)                         *      
!                                                                *      
!----------------------------------------------------------------*      
!                                                                *      
!  subroutines required: CFSPPE                                  *      
!                                                                *      
!*****************************************************************      
!                                                                *      
!  Author  : Gnter Palm                                         *      
!  Date    : 04.18.1988                                          *      
!  Source  : FORTRAN 77                                          *      
!                                                                *      
!*****************************************************************      
!                                                                       
      IMPLICIT DOUBLEPRECISION (A - H, O - Z) 
      DOUBLEPRECISION XN (0:N), FN (0:N), W (0:N), PHIN (0:N), R (0:N), &
      A (0:N), B (0:N), C (0:N), D (0:N), AUXF (1:14 * N - 10)          
!                                                                       
!-----Checking the input                                                
!                                                                       
      IERR = - 1 
      IF (N.LT.6) RETURN 
      IF (XN (0) .NE.XN (N) ) RETURN 
      IF (FN (0) .NE.FN (N) ) RETURN 
      IF (W (0) .NE.W (N) ) RETURN 
      DO 10 I = 0, N, 1 
         IF (W (I) .LE.0.0D0) THEN 
            IERR = - 3 
            RETURN 
         ENDIF 
   10 END DO 
!                                                                       
!-----Initializing                                                      
!                                                                       
      PI = 4.0D0 * DATAN (1.0D0) 
      PI2 = 2.0D0 * PI 
!                                                                       
!-----Transformation of the coordinates                                 
!                                                                       
!     Translation of the origin to P = (PX,PY),                         
!     in case MS is not equal to 0, i.e., we subtract                   
!     PX and PY from X(I) and Y(I)                                      
!                                                                       
      IF (MS.EQ.0) THEN 
!                                                                       
!       No shift                                                        
!                                                                       
         PX = 0.0D0 
         PY = 0.0D0 
         DO 20 I = 0, N, 1 
            B (I) = XN (I) 
            C (I) = FN (I) 
   20    END DO 
      ELSE 
!                                                                       
!       Shift the origin to (PX,PY)                                     
!                                                                       
         IF (MS.LT.0) THEN 
!                                                                       
!         Compute PX and PY                                             
!                                                                       
            XMAX = XN (0) 
            XMIN = XN (0) 
            YMAX = FN (0) 
            YMIN = FN (0) 
            DO 30 I = 1, N, 1 
               XMAX = DMAX1 (XMAX, XN (I) ) 
               XMIN = DMIN1 (XMIN, XN (I) ) 
               YMAX = DMAX1 (YMAX, FN (I) ) 
               YMIN = DMIN1 (YMIN, FN (I) ) 
   30       END DO 
            PX = (XMAX + XMIN) / 2.0D0 
            PY = (YMAX + YMIN) / 2.0D0 
         ENDIF 
         DO 40 I = 0, N, 1 
            B (I) = XN (I) - PX 
            C (I) = FN (I) - PY 
   40    END DO 
      ENDIF 
!                                                                       
!-----Compute the transformed nodes                                     
!                                                                       
!     1. Compute R(I); Stop if R(I)=0, for then the center              
!        (PX,PY) is one of the nodes                                    
!                                                                       
      IERR = - 4 
      DO 50 I = 0, N, 1 
         R (I) = DSQRT (B (I) * B (I) + C (I) * C (I) ) 
         IF (R (I) .EQ.0.0D0) RETURN 
   50 END DO 
!                                                                       
!     2. Compute the coordinates X', Y' after a rotation by ALPHA       
!        using the equations                                            
!                                                                       
!           [ X']   [ COS(ALPHA)   -SIN(ALPHA) ] [ X ]                  
!           [   ] = [                          ] [   ]                  
!           [ Y']   [ SIN(ALPHA)    COS(ALPHA) ] [ Y ]                  
!                                                                       
!        for ALPHA = -PHIR                                              
!                                                                       
      PHIR = DACOS (B (0) / R (0) ) 
      IF (C (0) .LT.0.0D0) PHIR = PI2 - PHIR 
      CA = B (0) / R (0) 
      SA = - C (0) / R (0) 
      DO 60 I = 0, N, 1 
         A (I) = B (I) * CA - C (I) * SA 
         D (I) = B (I) * SA + C (I) * CA 
   60 END DO 
!                                                                       
!     3. Compute the angular coordinates PHIN(I); Stop if the           
!        angles are not stictly monotonically increasing                
!                                                                       
      PHIN (0) = 0.0D0 
      DO 70 I = 1, N - 1 
         PHIN (I) = DACOS (A (I) / R (I) ) 
         IF (D (I) .LT.0.0D0) PHIN (I) = PI2 - PHIN (I) 
         IF (PHIN (I) .LE.PHIN (I - 1) ) RETURN 
   70 END DO 
      PHIN (N) = PI2 
!                                                                       
!-----Compute the spline coefficients                                   
!                                                                       
      CALL CFSPPE (N, PHIN, R, W, 1, A, B, C, D, AUXF (1), AUXF (N + 2),&
      AUXF (2 * N + 3), AUXF (3 * N + 4), AUXF (4 * N + 5), AUXF (5 * N &
      + 5), IERR)                                                       
      RETURN 
      END SUBROUTINE CFSPTR                         
