##########################################################################
##            
#A  minrep.g                               Meinolf Geck and Goetz Pfeiffer
##
#Y  Copyright (C) 1995      Equipe des groupes finis, Universite Paris VII
#Y                          and University of St. Andrews
##
##  This file contains functions for computing representatives of minimal
##  length in the conjugacy classes of finite Coxeter groups. You need 
##  CHEVIE to work with them. We also provide the programs which are used
##  to verify Theorem 1.1 in the article 'On the irreducible characters
##  of Hecke algebras' (Advances in Math. 102, 79--94, 1993).
##
##  We denote by -> the binary relation on the finite Coxeter group W 
##  which is generated by the relation: x->y if there exists some 
##  generator s in S such that y=sxs and l(y) <= l(x). It is checked 
##  that, for each element w in a conjugacy class C there exists an 
##  element x of minimal length in C such that w->y. Moreover, if C is 
##  a cuspidal class (i.e., all generators appear in reduced expressions 
##  for elements of minimal length in C) then the set of elements of 
##  minimal length in C is just one cyclic shift class.
##  
##  Example:    gap> W := CoxeterGroup( \"E\", 6 );
##              gap> SmallSet( W );
##  
##  The result is a set of representatives of cyclic shift classes for 
##  those elements w in W for which there exist no x in W with w->x and 
##  l(x)<l(w).
##
##  The program has to run, in some way, through all elements of W. The
##  efficiency of doing this depends a bit on a choice of a chain of 
##  parabolic subgroups in W. This can be controlled by choosing different
##  labellings of the nodes of the Dynkin diagram. For example, for type
##  E_8 one should choose the following labelling:
##
##              gap> c := CartanMat( \"E\", 8 );
##              gap>  c{[4,3,2,5,1,6,7,8]}{[4,3,2,5,1,6,7,8]};
##              gap> PrintDynkinDiagram( CartanType( last ) );
##              E8          3             
##                          |
##              5 - 2 - 1 - 4 - 6 - 7 - 8
##
#H  written MG, GP, 1992/1995
##
#F TestCyclicShift( <W>, <w> ) . . . . . . . . . . . . . minimality test 
#F CyclicShifts( <W>, <w> ) . . . . . . . . . . . .   cyclic shift class
#F SmallSet( <W> ) . . . . .  a set of cyclic shift class representatives
##

#############################################################################
##
#F TestCyclicShift( <W>, <w> ) . . . . . . . . . . . . . . .  minimality test 
##
## 'TestCyclicShift' returns 'true' if there exist no $y \in W$ such that
## $w \rightarrow y$ and $l(y)<l(w)$. Otherwise, such an element $y$ is
## returned, where $y=sws$ for some generator $s \in S$.
##
TestCyclicShifts:=function(W,w)
  local bahn,s,j,y,yy;
  bahn:=[w];
  for j in bahn do
    for s in [1..W.semisimpleRank] do
       y:=W.generators[s]*j;
       if s^j>W.N then
         if s^(y^-1)>W.N then 
           return y*W.generators[s];
         else
           yy:=y*W.generators[s];
           if not yy in bahn then
             Add(bahn,yy);
           fi;
         fi;
       else
         yy:=y*W.generators[s];
         if s^(y^-1)>W.N and not yy in bahn then
           Add(bahn,yy);
         fi;
       fi; 
    od;
  od;
  return true;
end;

############################################################################
##
#F CyclicShifts( <W>, <w> ) . . . . . . . . . . . . . . . cyclic shift class
##
## 'CyclicShifts' returns the orbit of $w \in W$ under the relation 
## generated by: $x->y$ where $l(x)=l(y)$.
##
CyclicShifts:=function(W,w)
  local orbit,x,y,i;
  orbit:=[w];
  for x in orbit do
    for i in [1..W.semisimpleRank] do
      y:=W.generators[i]*x;
      if (i^x<=W.N and i/y>W.N) or (i^x>W.N and i/y<=W.N) then
        y:=y*W.generators[i];
        if not y in orbit then
          Add(orbit,y);
        fi;
      fi;
    od;
  od;
  return orbit;
end;

############################################################################
##
#F SmallSet( <W> ) . . . . . . a small set of conjugacy class representatives
##
## 'Smallset' returns a set of representatives of cyclic shift 
## classes of the elements in the corresponding class for which 
## 'TestCyclicShifts' is true. 
##
SmallSet:=function(W)
  local SelectedCosets,I,J,i,j,cos,min,new,m,w,y,cc,rest,z;
  SelectedCosets:=function(W,I,J,cos)
    local fertig,j,c,x,w,new;
    new:=[];
    for w in ReducedRightCosetRepresentatives(ReflectionSubgroup(W,I),
                                                  ReflectionSubgroup(W,J)) do
      for c in cos do
        x:=w*c;
        fertig:=true;
        j:=1;
        while fertig and j<=Length(J) do
          if J[j]/x>W.N then
            fertig:=false;
          else
            j:=j+1;
          fi;
        od;
        if j>Length(J) then
          Add(new,x);
        fi;
      od;
    od;
    return new;
  end;
  cos:=[()];
  I:=[1..W.semisimpleRank];
  for i in Reversed([1..W.semisimpleRank]) do
    J:=Difference(I,[i]);
    cos:=SelectedCosets(W,I,J,cos); 
    I:=Copy(J);
  od;
  InfoChevie("#I cos: " ,Length(cos)," \c");
  min:=[];
  z:=1;
  for w in cos do
    if IsInt(z/100) then
      InfoChevie(".\c");
      z:=1;
    else
      z:=z+1;
    fi;
    y:=TestCyclicShifts(W,w);
    if y=true then
      AddSet(min,w);
#    else
#      if not y in min and not y in cos then
#        Add(cos,y);
#      fi;
    fi;
  od;
  InfoChevie("  min: ",Length(min)," \c");
  rest:=[];
  while min<>[] do
    Add(rest,min[1]);
    min:=Difference(min,CyclicShifts(W,min[1]));
  od;
  InfoChevie("rest: ",Length(rest),"    \n");
  return List(rest,i->CoxeterWord(W,i));
end;

#  cc:=ConjugacyClasses(Group(W.generators,()));
#  InfoChevie("#I computed ",Length(cc)," conjugacy classes \n");
#  new:=List(cc,i->[]);
#  for m in rest do
#    Add(new[Position(cc,ConjugacyClass(W,m))],m);
#  od;
#  return List(new,i->List(i,j->CoxeterWord(W,j)));

