
#############################################################################
##
#A  coxeter.g            CHEVIE library          Meinolf Geck, Frank Luebeck, 
#A                                            Jean Michel and G\"otz Pfeiffer
##
#A  $Id: coxeter.g,v 1.2 1997/02/20 11:09:42 werner Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##
##  This file contains general functions that deal with Coxeter groups.
##

#############################################################################
##
#F  CartanMat( <type>, <rk>, ... )  
#F    returns a Cartan matrix of given type(s) and rank(s)
##
##  E.g., 'CartanMat( "F", 4 );'
##  For type I2(n), a third argument is needed: 'CartanMat( "I", 2, n )'.
##  If several couples <type>, <rk> are given, the direct sum of the
##  corresponding Cartan matrices is returned.
##
CartanMat := function ( arg ) local t, n, An, r, a, i, j, res;
  An := function ( n ) local  a, i, v;
	a := [  ];
	for i  in [ 1 .. n ]  do
	    a[i] := [1..n]*0;
	    a[i][i] := 2;
	    if i<n then a[i][i+1] := -1; fi;
	    if i>1 then a[i][i-1] := -1; fi;
	od;
	return a;
  end;
  res:=[];
  while Length(arg)>0 and IsString(arg[1]) do
    t:=arg[1];n:=arg[2];arg:=arg{[3..Length(arg)]};
    if t = "A" or t="~A" then a:=An(n);
    elif t = "B" and n>=2 then a:=An(n); a[1][2]:=-2;
    elif t = "C" and n>=2 then a:=An(n); a[2][1]:=-2;
    elif t = "D" and n>=3 then
	a:=An(n); a{[1..3]}{[1..3]}:=[[2,0,-1],[0,2,-1],[-1,-1,2]];
    elif t = "G" and n=2 then a:=An(2); a[2][1]:=-3;
    elif t = "F" and n=4 then a:=An(4); a[3][2]:=-2;
    elif t = "E" and n in [6..8] then
        a:=An(n);
        a{[1..4]}{[1..4]}:=[[2,0,-1,0],[0,2,0,-1],[-1,0,2,-1 ],[0,-1,-1, 2]];
    elif t = "H" and n in [2..4] then
	r := -(1 + ER( 5 )) / 2;
	a := [[2,r,0,0],[r,2,-1,0 ],[0,-1,2,-1],[0,0,-1,2]];
	a := a{[1..n]}{[1..n]};
    elif t = "I" and n=2 and Length(arg)>0 then
      n:=arg[1];arg:=arg{[2..Length(arg)]};
      if n=2 then         a:= [[2,0],[0,2]];
      elif n mod 2=0 then a:= [[2,-1],[-2-E(n)-E(n)^-1,2]];
      else                a:= [[2,-E(2*n)-E(2*n)^-1],[-E(2*n)-E(2*n)^-1,2]];
      fi;
    else Error("illegal arguments");
    fi;
    Add(res,a);
  od;
  return DirectSumMat(res);
end;

#############################################################################
##
#F  CartanType( <C> ) . . . . . . . . . . . . . . . type of a Cartan matrix
#F  CartanType( <rec> ) . . . type of a Coxeter Datum or Hecke algebra, or...
##  
##  For    a  Cartan matrix  <C>   argument,  returns  a  list  of couples
##  [type,indices] which  describe  irreducible components   of the Cartan
##  matrix <C>, such that
##    C{indices}{indices}=CartanMat(type,Length(indices))
##  (a triple is actually returned for type "I").
##  
##  If <C> is not a Cartan matrix this function returns 'false'. So it can
##  be used to check a matrix for being a Cartan matrix.
##  
##  This function has been made generic since it is overloaded for Coxeter 
##  cosets.
##  
CartanType:=function(C)
  local c, i, type, s, t, m, r, x, list, branch, l, p, tmp;
  
  # dispatcher function, if not called with Cartan matrix:
  if IsRec(C) then
    if not IsBound(C.type) then
      if IsBound(C.operations) and IsBound(C.operations.CartanType) then
        C.type:=C.operations.CartanType(C);
      else Error("argument has no method CartanType\n");
      fi;
    fi;
    return C.type;
  fi;
  
  # else C is a Cartan matrix:
  type:=[];
  for s in DecomposedMat(C) do
    m:=C{s}{s};
    r:=Length(m);
    list:=[1];
    repeat
      x:=Filtered([1..r],i->m[list[Length(list)]][i]<>0 and not i in list);
      if Length(x)>0 then Add(list,x[1]);fi;
    until Length(x)=0;
    repeat
      x:=Filtered([1..r],i->m[list[1]][i]<>0 and not i in list);
      if Length(x)>0 then list:=Concatenation([x[1]],list);fi;
    until Length(x)=0;
    l:=Length(list);
    if l=r then # types A,B,C,F,G,H,I
      if not ForAll(Concatenation(m),IsInt) then
	 if r>2 then 
	   if m[list[1]][list[2]]=-1 then list:=Reversed(list);fi;
	   t:=["H",list];
	 else
	   x:=Maximum(List(Concatenation(m),NofCyc));
	   if x mod 2<>0 and m[list[1]][list[2]]<>m[2][1] then x:=2*x; fi;
	   if m[list[1]][list[2]]<>m[list[2]][list[1]] and
	      m[list[1]][list[2]]<>-1 then list:=Reversed(list);fi;
	   #if x=5 then t:=["H",list];
	   #else t:= ["I",list,x];
	   #fi;
	   t:= ["I",list,x];
	 fi;
      elif -3 in Concatenation(m) then
	   if m[list[1]][list[2]]<>-1 then list:=Reversed(list);fi;
	   t:= ["G",list];
      elif -2 in Concatenation(m) then
	   if m[list[1]][list[2]]<>-2 and m[list[2]][list[1]]<>-2 then
	     list:=Reversed(list);
	   fi;
	   if m[list[1]][list[2]]<>-2 and m[list[2]][list[1]]<>-2 then
	     if m[list[3]][list[2]]<>-2 then list:=Reversed(list);fi;
	     t:=["F",list];
	   elif m[list[1]][list[2]]=-2 then t:= ["B",list];
	   else t:= ["C",list];
	   fi;
      else t:= ["A",list];
      fi;
    else # types D,E
      branch:=Filtered(list,
		       x->Number([1..r],i->m[x][i]<>0 and not i in list)<>0);
#      if Length(branch)<>1 then Error("not a Cartan matrix");fi;
      if Length(branch)<>1 then return false; fi;
      repeat
       x:=Filtered([1..r],i->m[branch[Length(branch)]][i]<>0
			     and not i in branch and not i in list);
       if Length(x)>0 then Add(branch,x[1]);fi;
      until Length(x)=0;
#      if Length(branch)+l<>r+1 then Error("not a Cartan matrix");fi;
      if Length(branch)+l<>r+1 then return false; fi;
      p:=Position(list,branch[1]);
      if Length(branch)>=l+1-p then 
	x:=list{[p..l]};
	list:=Concatenation(list{[1..p-1]},branch);
	branch:=x;
	l:=Length(list);
      fi;
      if Length(branch)>=p then
	x:=list{[1..p]};
	list:=Concatenation(Reversed(branch),list{[p+1..l]});
	p:=Length(branch);
	branch:=Reversed(x);
	l:=Length(list);
      fi;
      if p-1 > l-p then list:=Reversed(list);p:=l+1-p;fi;
      if p=2 then t:=["D",Concatenation([branch[2]],list)];
      else  
	t:=["E",Concatenation([list[1]],[branch[2]],list{[2..Length(list)]})];
      fi;
    fi;
    t[2]:=s{t[2]};
    Add(type,t);
  od;
  
  # Adjustment such that type B_2=C_2 is always considered as C_2:
  for t in [1..Length(type)] do
    if type[t][1]="B" and Length(type[t][2])=2 then
      type[t]:=["C",[type[t][2][2],type[t][2][1]]];
    fi;
  od;
  
  # Sorting via rank and alphabetically:
  tmp:=List(type,a->[Length(a[2]),a[1],Minimum(a[2])]);
  SortParallel(tmp,type,function(a,b)
                          return a[1]>b[1] or (a[1]=b[1] and a[2]<b[2])
                                or (a[1]=b[1] and a[2]=b[2] and a[3]<b[3]); 
                        end);
  
  # countercheck if we get C from type:                      
  tmp:=IdentityMat(Length(C));
  for t in type do
    if (t[1]="E" and not Length(t[2]) in [6..8]) or
       (t[1]="F" and Length(t[2])<>4) or
       (t[1]="G" and Length(t[2])<>2) or
       (t[1]="H" and not Length(t[2]) in [3,4]) then
      return false;
    fi;
    if Length(t)=2 then
      tmp{t[2]}{t[2]}:=CartanMat(t[1],Length(t[2]));
    else
      tmp{t[2]}{t[2]}:=CartanMat(t[1],Length(t[2]),t[3]);
    fi;
  od;
  if tmp=C then
    return type;
  else
    return false;
  fi;
end;

#############################################################################
##
#F  CartanName(<type>) . . . . . . . . . . . . . . . . name of a CartanType
#F  CartanName(<W>) . . . .  name of a Coxeter Datum or Hecke algebra, etc...
##  
##  Gives a string which describes the isomorphism type of W (like "A2B3" for
##  the direct product of a root system of type A2 by one of type B3, or like
##  "I2(7)"). 
##  The first form allows to use as CartanName(CartanType(<cartan matrix>));
##  This function has been made generic since it is overloaded for Coxeter 
##  cosets.
##

CartanName:=function(W)local t,name;
  if IsRec(W) then
    if not IsBound(W.cartanName) then
      if IsBound(W.operations) and IsBound(W.operations.CartanName) then
        W.cartanName:=W.operations.CartanName(W);
      else Error("argument has no method CartanName");
      fi;
    fi;
    return W.cartanName;
  else 
    name:=[];
    for t in W do
      Append(name,t[1]);
      Append(name,String(Length(t[2])));
      if Length(t)=3 then
	 Append(name,"(");
	 Append(name,String(t[3]));
	 Append(name,")");
      fi;
      Append(name,"x");
    od;
    return name{[1..Length(name)-1]};
  fi;
end;

#############################################################################
##
#F  PrintDynkinDiagram( <type> ) . . . prints Dynkin diagram of a CartanType
#F  PrintDynkinDiagram( <rec> ) . . prints Dynkin diagram of a Coxeter Coset
##
##  This is a purely descriptive routine which shows the labeling of
##  the Dynkin diagram corresponding to the given <type>, which should
##  conform to the format of CartanType.
##  The first form allows PrintDynkinDiagram(CartanType(<cartan matrix>))
##  This function has been made generic since it is overloaded for Coxeter 
##  cosets.
##
PrintDynkinDiagram:=function(W)
  local i, v, r, n, t, a, o;
  if IsRec(W) then
    if IsBound(W.operations) and
       IsBound(W.operations.PrintDynkinDiagram)
    then W.operations.PrintDynkinDiagram(W);
    else Error("argument has no method PrintDynkinDiagram\n");
    fi;
  else 
    for a in W do
      t:=a[1];v:=a[2];r:=Length(v);
##        if Length(a)=3 then Print("I2(",a[3],")    ",v[1]," - ",v[2],"\n");
      if Length(a)=3 then 
        if a[3] mod 2 = 1 then
          Print("I2(",a[3],")    ",v[1]," - ",v[2],"\n");
        else
          Print("I2(",a[3],")    ",v[1]," > ",v[2],"\n");
        fi;
      elif t="B" then
        Print("B",r,"    ",v[1]," < ",v[2]);
        for i in [3..r] do Print(" - ",v[i]);od;Print("\n");
      elif t="C" then
        Print("C",r,"    ",v[1]," > ",v[2]);
        for i in [3..r] do Print(" - ",v[i]);od;Print("\n");
      elif t="A" or t="~A" then
        Print(t,r,"    ",v[1]);
        for i in [2..r] do Print(" - ",v[i]);od;Print("\n");
      elif t="D" then
##          Print("D",r,String("",Length(String(v[1]))-1),"      ",v[2],"\n");
##          Print(String("",Length(String(r))+Length(String(v[1]))-1));
##          Print("       |\n");
##          Print(String("",Length(String(r))));Print("   ",v[1]);
##          for i in [3..r] do Print(" - ",v[i]);od;Print("\n");
        
##          Print("D",r,String("",Length(String(v[2]))-1),"      ",v[1],"\n");
##          Print(String("",Length(String(r))+Length(String(v[2]))-1));
##          Print("       |\n");
##          Print(String("",Length(String(r))));Print("   ",v[2]);
##          for i in [3..r] do Print(" - ",v[i]);od;Print("\n");
        
        Print("D",r,"   ",v[1],"\n");
        Print(String("\\",Length(String(r))+6));
        Print(String("",Length(String(r))+6),"\n",
                      String("",Length(String(r))+6),v[3]);
        for i in [4..r] do Print(" - ",v[i]);od;Print("\n");
        Print(String("/",Length(String(r))+6),"\n");
        Print(String("",Length(String(r))+4),v[2],"\n");
      elif t="E" then
        Print("E",r,String("",Length(String(v[1]))+Length(String(v[3]))-2),
              "          ",v[2],"             \n");
        Print(String("",Length(String(v[1]))+Length(String(v[3]))-2),
              "            |\n");
        Print("    ",v[1]);for i in [3..r] do Print(" - ",v[i]);od;Print("\n");
      elif t="G" then
        Print("G2    ",v[1]," > ",v[2]," \n");
      elif t="F" then
        Print("F4    ",v[1]," - ",v[2]," > ",v[3]," - ",v[4],"\n");
      elif t="H" then
        Print(String("",Length(String(v[1]))-1),"        5 \n");
        Print("H",r,"    ",v[1]," - ",v[2]," - ",v[3]);
        if r=4 then Print(" - ",v[4]);fi;
        Print("\n");
      fi;
    od;
  fi;
end;


#############################################################################
##
#F  RootsIntCartan( <C> ) . . . . . . . . root system from Cartan matrix with
#F  integer entries
##  
##  
##  Returns the set of roots determined by  <C>, as linear combinations of
##  the simple  ones.  The first half  are the  positive roots ordered  by
##  their height.  The second half   contains  the corresponding  negative
##  roots in the same order.
##  
##  The simple roots come first and they are ordered as given by <C>.
##  
RootsIntCartan:=function(C)
  local RootsIntCartan1, a, l, n, p, v, i, x, tmp, erg;
  
  # returns list l where l[i] is the list of positive roots of height i:
  RootsIntCartan1:=function(C)
    local erg, I, l, a, pmq, i, j;
    l:=Length(C);
    I:=IdentityMat(l);
    erg:=[I];
    erg[2]:=[];
    for i in [1..l] do
      for j in [i+1..l] do
        if C[i][j]<>0 then
          Add(erg[2],erg[1][i]+erg[1][j]);
        fi;
      od;
    od;
    i:=2;
    while erg[i]<>[] do
      erg[i+1]:=[];
      for a in erg[i] do
        for j in [1..l] do
          pmq:=C[j]*a;
          if pmq<0 or i-pmq-1>0 and  a[j]>pmq and 
                                          a-(pmq+1)*I[j] in erg[i-pmq-1] then 
            if not (a+I[j] in erg[i+1]) then
              Add(erg[i+1],a+I[j]);
            fi;  
          fi;
        od;
      od;
      i:=i+1;
    od;
    Unbind(erg[i]);

    return erg;  
  end;

  l:=Length(C);
  n:=0*[1..l];
  
  # to compute the irreducible subsytems separately:
  p:=DecomposedMat(C);

  # simple roots must be ordered as given (erg[i] becomes the 
  # set of roots of height i): 
  erg:=[IdentityMat(l)];
  for a in p do
    tmp:=RootsIntCartan1(C{a}{a});
    for i in [2..Length(tmp)] do
      if not IsBound(erg[i]) then
        erg[i]:=[];
      fi;
      for x in tmp[i] do
        v:=ShallowCopy(n);
        v{a}:=x;
        Add(erg[i],v);
      od;
    od;
  od;
  
  # we get now the reflection degrees very cheap:
  p:=List(erg,Length);
  tmp:=0*[1..Maximum(p)];
  for i in p do
    for a in [1..i] do
      tmp[a]:=tmp[a]+1;
    od;
  od;
  
  erg:=Concatenation(erg);
  return rec(roots:=Concatenation(erg,-erg)
     # , degrees:=Reversed(tmp)+1
        );
end;

    
#############################################################################
##
#F  RootsAndMatgens( <C> ) . . the root system of the Cartan Matrix <C>
##
##  Returns a record with fields .cartan, .matgens, .roots, .N, 
##    .semisimpleRank
##  cartan         : the Cartan matrix <C>,
##  semisimpleRank : the size of <C>
##  roots          : the root vectors
##  N              : the number of positive roots
##  matgens        : the matrices of the reflections w.r.t. the simple roots
##  
RootsAndMatgens := function(C)local W,R, a, i, v, k, e;
  W:=rec(cartan:=C);
  W.semisimpleRank:=Length(W.cartan);
  if Length(W.cartan)=0 then W.matgens:=[];W.roots:=[];
  else 
    e := W.cartan ^ 0;
    W.matgens := List([1..W.semisimpleRank], x->Copy(e));
    for k in [1..W.semisimpleRank] do 
      W.matgens[k]{[1..W.semisimpleRank]}[k] := e[k]-W.cartan[k]; 
    od;
    if ForAll( Concatenation( W.cartan ), IsInt )  then
        R:=RootsIntCartan(W.cartan);
#       W.degrees:=R.degrees;
        R:=R.roots;
    else
        R := W.cartan ^ 0;
        for a  in R  do
            for i  in [1..W.semisimpleRank] do
                if a <> R[i] then
                    v := a * W.matgens[i];
                    if not v in R  then
                        Add( R, v );
                    fi;
                fi;
            od;
        od;
        R:=Concatenation(R,-R);
    fi;
    W.roots:=R;
  fi;
  W.N := Length( W.roots )/2;
  return W;
end;

########################################################################
##
#F  HighestShortRoot( <W> ) . . . . . . . . . . . . . highest short root 
##
##  If <W> is irreducible then 'HighestShortRoot' computes the unique 
##  short root of maximal height. Note that if all roots have the same 
##  length then this is the  unique root of maximal height. The latter 
##  can be extracted by  W.roots[W.N]. If <W> is not irreducible then
##  an error message is returned.
##
HighestShortRoot:=function(W)
  if Length(CartanType(W))=1 then 
    return First([W.N,W.N-1..1],i->W.rootLengths[i]=1);
  else 
    Error(" group is not irreducible\n");
  fi;
end;


#############################################################################
##
#F  CoxeterElementsLength( <W>, <l> ) . . the list of elements of W of length l
##
##  Returns as a list of permutations the set of elements of W of length l,
##  sorted (according to the default order on permutations).
##  

CoxeterElementsLength:=function(W,l)local H,e,x,i;
  if l>W.N then 
    InfoChevie("#W no elements of that length\n");
    return false;
  fi;
  if not IsBound(W.elts) then W.elts:=[[()]];fi;
  if IsBound(W.elts[l+1])then return W.elts[l+1];fi;
  H:=ReflectionSubgroup(W,W.rootInclusion{[1..W.semisimpleRank-1]});
  W.elts[l+1]:=[];
  if not IsBound(W.rc) then 
    W.rc:=List([0..W.N],x->[]);
    for x in ReducedRightCosetRepresentatives(W,H) do
       Add(W.rc[1+CoxeterLength(W,x)],x);
    od;
  fi;
  for i in [0..Minimum(l,H.N)] do
    e:=CoxeterElementsLength(H,i);
    for x in W.rc[1+l-i] do Append(W.elts[l+1],e*x);od;
  od;
  W.elts[l+1]:=Set(W.elts[l+1]);
  InfoChevie("#I Number of elements of length ",l,": ",
                                   Length(W.elts[l+1]),"\n");
  if W.N-l<>l then
    W.elts[1+W.N-l]:=Set(W.elts[l+1]*LongestCoxeterElement(W));
  fi;
  return W.elts[l+1];
end;

#############################################################################
##
#F  CoxeterWords( <W> )  . . . . . . . . . . . . . . . . .  all elements of W
##
##  'CoxeterWords' returns the list words in the Coxeter group W.
##  The list is obtained by concatenating CoxeterElementsLength in
##  order of increasing length.
##

CoxeterWords := W->Concatenation(List([0..W.N],
          i->List(CoxeterElementsLength(W,i),w->CoxeterWord(W,w))));

#############################################################################
##
#F  CoxeterLength( <W> , <w> )  . . . . . length of a permutation w in W
##
##  'CoxeterLength'  returns  the number of positive roots send by w to
##  negative roots.
##  

CoxeterLength:=function ( W, w )
  local res,r;
# it's a pity the line below is 5 times less efficient than the next 5 lines
#    return Number(W.rootInclusion{[1..W.N]},r->r ^ w > W.parentN);
#
  res:=0;
  for r in W.rootInclusion{[1..W.N]} do
    if r ^ w > W.parentN then res:=res+1; fi;
  od;
  return res;
end;

#############################################################################
##
#F  LongestCoxeterElement( <W> ) . . Longest element of W as a permutation
##  
##  LongestCoxeterElement returns as a permutation the longest element of W
##  

LongestCoxeterElement:=function( W ) 
  local  i, res;
  if IsBound(W.longestPerm) then return W.longestPerm; fi;
  res := ();
  if W.semisimpleRank=0 then W.longestPerm:=(); return ();fi;
  while true do
      i := 1;
      while W.rootInclusion[i] / res > W.parentN do
	 i := i + 1;
	 if i > W.semisimpleRank then 
	   W.longestPerm:=res;
	   return W.longestPerm;
	 fi;
      od;
      res := res * W.generators[i];
  od;
end;

#############################################################################
##
#F  LongestCoxeterWord( <W> )  . . . . . . . . . .  the longest element in W
##
## 'LongestCoxeterWord' returns a reduced expression in the  standard 
##  generators of the unique longest element of the Coxeter group W.
##
LongestCoxeterWord:=function(W)local save;
  if not IsBound(W.longestCoxeterWord) then 
    save:=CHEVIE.BrieskornNormalForm;
    CHEVIE.BrieskornNormalForm:=false;
    W.longestCoxeterWord:=CoxeterWord( W, LongestCoxeterElement(W));
    CHEVIE.BrieskornNormalForm:=save;
  fi;
  return W.longestCoxeterWord;
end;

#############################################################################
##
#F  ReducedRightCosetRepresentatives( <W>, <H> )  . . . . . . . . . . . . . 
#F  . . . . . . . . . . . . . . .  distinguished right coset representatives
##
##  'ReducedRightCosetRepresentatives'  returns a list of reduced words in the
##  Coxeter group <W> that are distinguished right  coset representatives for
##  the right cosets H\W where  H is a reflection subgroup  of W.
##
ReducedRightCosetRepresentatives:=function(W,H)
  local res, ress, new, supp, w, v, s;
  res:=[];
  ress:=[];
  new:=[];
  supp:=Set([()]);
  while Length(supp)>0 do
    new:=supp;
    supp:=[];
    Append(res,new);
    UniteSet(ress,new);
    for w in new do
      for s in W.generators do
        v:=ReducedInCoxeterCoset(H,w*s);
        if not v in ress then 
          AddSet(supp,v);
        fi;
      od;
    od;
  od;
  InfoChevie("#nb. of cosets: ",Length(res),"\n");
  return res;
end;

#############################################################################
##
#F  PermCosetsSubgroup ( <W>, <H> ) . . . . . . . . . . . . . . . . .
#F  . . . .  permutation representation on the cosets of a subgroup
##
##  'PermCosetsSubgroup '  returns  the  list of permutations induced
##  by the standard generators of the Group  <W>  on the cosets  of  the
##  subgroup <H>. The cosets are in the order determined by the result of
##  the  function 'ReducedRightCosetRepresentatives( <W>, <H>)'.
##

PermCosetsSubgroup :=function(W,H)local D;
  D:=ReducedRightCosetRepresentatives(W, H);
  return List(W.generators,s->Sortex(List(D*s,x->ReducedInCoxeterCoset(H,x))))/
                                                     Sortex(ShallowCopy(D));
end;

#############################################################################
##
#F  ReducedInCoxeterCoset( <W> , <w> )  . . . . . reduced element in coset W.w
##
##  w is an automorphism of a parent group of W, given as a permutation of
##  the roots. 'ReducedInCoxeterCoset' returns the unique element in the right
##  coset W.w which sends all roots of W to positive roots.
##  
ReducedInCoxeterCoset:=function( W, w ) local  i;
  while true  do
      i := 0;
      repeat
	 if i>=W.semisimpleRank then return w;fi;
	 i := i + 1;
      until W.rootInclusion[i] ^ w > W.parentN;
      w := W.generators[i] * w;
  od;
end;

#############################################################################
##
#F  ReducedCoxeterWord( <W> , w )  . . . . . . . . a reduced word for w in W
##
##  'ReducedCoxeterWord' returns a reduced expression for the element w, given
##  as  an  arbitrary  list  of  integers  where  each  entry  i  in this list
##  represents the  i-th standard  generator of W.
##
ReducedCoxeterWord := function( W, w ) 
  return CoxeterWord(W,PermCoxeterWord (W,w));
end;

#############################################################################
##
#F  Bruhat( <W>, <y>, <w>[, <ly>, <lw> ] ) . . . . . . . Bruhat partial order
##
##  'Bruhat'  returns true, if the element  <y>  is less than or equal to the 
##  element <w> of the Coxeter group <W>, and false otherwise. Both <y> and  
##  <w>  must be given as permutations on the root vectors of <W>. The 
##  additional arguments <ly> and <ly> can contain the lengths of <y> and <w>.
##
Bruhat := function ( arg ) 
  local  s, x, y, lx, ly, W, rI, pN, gen;
  W:=arg[1];
  x:=arg[2];
  y:=arg[3];
  if Length(arg)>3 then
    lx:=arg[4];
    ly:=arg[5];
  else
    lx:=CoxeterLength(W,x);
    ly:=CoxeterLength(W,y);
  fi;
  rI:=W.rootInclusion;
  pN:=W.parentN;
  gen:=W.generators;
  
  while lx<ly and lx<>0 and ly<>0 do
    s:=1;
    while rI[s]^y <= pN do
      s:=s+1;
    od;
    if rI[s]^x > pN then
      x:=gen[s]*x;
      lx:=lx-1;
    fi;
    y:=gen[s]*y;
    ly:=ly-1;      
  od;      
  return lx=0 or (lx=ly and x=y);
end;

#############################################################################
##
#F  LeftDescentSet( W, x)  the set of generators s such that sx < x
##  
##  the generators are numbered by the corresponding reflections of the parent
##  group
##  
LeftDescentSet:=function(W,w) 
# the long-winded version below is 3 times faster than this line
# return Filtered(W.rootInclusion{[1..W.semisimpleRank]},i->i^w>W.parentN);end;
   local res,j;
   res:=[];
   for j in W.rootInclusion{[1..W.semisimpleRank]} do
     if j^w>W.parentN then Add(res,j);fi;
   od;
   return res;
end;

#############################################################################
##
#F  RightDescentSet( W, x)  the set of generators s such that xs < x
##
RightDescentSet:=function(W,w)
  return LeftDescentSet(W,w^-1);
end;

#############################################################################
##
##  general dispatcher function to get data on Coxeter groups from 
##  files weylxx.g
##
CHEVIE.Load:=function(type,rank)local res;
  if type="~A" then res:="A";
  elif type="C" then res:="B";
  elif type in ["E","F","G","H"] then res:=Concatenation(type,String(rank));
  else res:=type;
  fi;
  return ReadChvTbl(res);
end;

#############################################################################
##
##  general dispatcher function to get data on Coxeter cosets from files 
##  weylxx.g
##
CHEVIE.LoadCoset:=function(type,rank,o)local res;
  if o=1 then return CHEVIE.Load(type,rank);fi;
  if type in ["A","~A"]  then res:="2A";
  elif o=3 then res:="3D4";
  elif type="D" then res:="2D";
  elif type="E" then res:="2E6";
  elif type="I" then res:="2I";
  fi;
  return ReadChvTbl(res);
end;

##############################################################################
##
#F CoxeterCharTableIrred( <arg> ) . . . . . . . . . . . . . . . . . . . . 
##
## the function 'CoxeterCharTableIrred' accepts an argument which is the type
## of an irreducible Coxeter group (like a component of the result of 
## CartanType). Arguments: type,indices[,bond (type I)]
## The point of this function is that it constructs the chartable
## without constructing a group. 
##
CoxeterCharTableIrred:=function(arg)
  local p, r, rank, tbl, type, inds;

  type:=arg[1];
  if IsList(arg[2]) then inds:=arg[2];
  else inds:=[1..arg[2]];
  fi;
  rank:=Length(inds);
  r:=CHEVIE.Load(type,rank);
  if type in ["A","~A","B","C","D"] then tbl:=r.CharTable(rank);  
  elif type in ["E","F","G","H"]  then tbl:=r.HeckeCharTable(1);
  elif type="I" then tbl:=r.HeckeCharTable(arg[3],1);
  fi;
  tbl.classtext:=List(tbl.classtext,x->
                          OnTuples(x,MappingPermListList([1..rank],inds)));
  
  # change '.name' if tbl is specialized from table for Hecke algebra:
  if not IsBound(tbl.name) then
    tbl.name:=tbl.identifier;
  fi;
  if not IsBound(tbl.order) then
    tbl.order:=tbl.size;
  fi;
  p:=Position(tbl.name,'H');
  if p<>false and tbl.name[p+1]='(' then
    tbl.name[p]:='W';
    tbl.identifier:=tbl.name;
  fi;
  return tbl;
end;

###########################################################################
##
#F  LowestPowerFakeDegrees(W) . . . . . . The list of b for characters of W
##
##  returns for each character chi, the smallest b such that chi occurs
##  in the b-th symmetric power of the reflection representation of W.
##  (the valuation of the fakedegree of chi).
##
LowestPowerFakeDegrees:=function(W)local t, res,r,type,rank;
  if not IsBound(W.b) then
    res:=[];
    for t in CartanType(W) do
      type:=t[1];
      rank:=Length(t[2]);
      r:=CHEVIE.Load(type,rank);
      if type in ["E","F","G","H"] then Add(res,List(r.CharParams(),x->x[2]));
      elif type="I" then Add(res,List(r.CharParams(t[3]),x->x[2]));
      else Add(res,List(r.CharParams(rank),x->r.Charb(x)));
      fi;
    od;
    W.b:=List(Cartesian(res),Sum);
  fi;
  return W.b;
end;

###########################################################################
##
#F  HighestPowerFakeDegrees(W) . . . . . . The list of B for characters of W
##
##  returns for each character chi, the degree of the fake degree of chi
##
HighestPowerFakeDegrees:=function(W)local t, res,r,rank,type;
  if not IsBound(W.B) then
    res:=[];
    for t in CartanType(W) do
      type:=t[1];
      rank:=Length(t[2]);
      r:=CHEVIE.Load(type,rank);
      if type in ["E","F","G","H"] then
	Add(res,List(r.vpolfakedegrees,x->Length(x[1])+x[2]-1));
      elif type in ["A","~A","B","C","D"] then 
        Add(res,List(r.CharParams(rank),x->r.CharB(x)));
      elif type="I" then
        Add(res,List(r.CharParams(t[3]),x->r.CharB(t[3],x)));
      fi;
    od;
    W.B:=List(Cartesian(res),Sum);
  fi;
  return W.B;
end;

############################################################################
##
#F  jInductionTable( <subgroup>, <group> ) . . j-induction of characters
##  
##  This    function  works   like  'InductionTable'   but   computes  the
##  j-induction, as defined in [Lusztig-book, p.78].
##  
jInductionTable:=function(u,g)
  local it,i,j,bu,bg;
  if not (IsBound(u.isCoxeterGroup) and IsBound(g.isCoxeterGroup)) then
    Error("groups must be CoxeterGroup groups.\n");
  fi;
  it:=InductionTable(u,g);
  it.headString:=String(Concatenation("j-",it.headString));
  bu:=LowestPowerFakeDegrees(u);bg:=LowestPowerFakeDegrees(g);
  for i in [1..Length(bu)] do
    for j in [1..Length(bg)] do
      if bg[j]<>bu[i] then it.scalar[j][i]:=0;fi;
    od;
  od;
  it.operations := ShallowCopy(it.operations);
  it.operations.Print:=function(t)
       Print("jInductionTable( ",t.u,", ",t.g,")");
     end;
  return it;
end;

#############################################################################
##
#F  Reflections( <W> )  . . . . . . . . . . . .  the reflections in W
##
## 'Reflections'  returns  the set of all reflections in the Coxeter group
## <W>, as a list of permutations. The i-th entry in this list
##  is the reflection along the i-th root in <W>.roots.
##  
Reflections:=function(W)
  if not IsBound(W.reflectionsPerm) then
    W.reflectionsPerm:=List([1..W.N],i->W.generators
	  [W.orbitRepresentative[i]]^W.orbitRepresentativeElements[i]);
    W.reflectionsPerm{[W.N+1..2*W.N]}:=W.reflectionsPerm{[1..W.N]};
  fi;
  return W.reflectionsPerm;
end;

#############################################################################
##
#F  CoxeterWordReflections( <W> )  . . . . . . . . . .  the reflections in W
##
## 'CoxeterWordReflections'  returns  the set of all reflections in the 
##  Coxeter group <W>, as a list of words. The i-th entry in this list
##  is the reflection along the i-th root in <W>.roots.
##  
## JM 30.5.96: restored since it is was there in the old version and is 
## documented in the GAP manual. Maybe trivial enough to be suppressed?
##  
CoxeterWordReflections:=W->List(Reflections(W),x->CoxeterWord(W,x));

#############################################################################
##
#F  SimpleRootsSubsystem( <W>, <l> ) . . . . . . . simple roots for subsystem
#F  of reflection subgroup
##  
##  <l> must  be a  subset of [1..2*Parent(<W>).N]. 'SimpleRootsSubsystem'
##  returns the  set of simple roots of  the root system of the reflection
##  subgroup              of       Parent(<W>),      generated          by
##  Reflections(Parent(<W>)){<l>}.
##  
SimpleRootsSubsystem:=function(W,l)
  local orb, refperm, gen, x, tmp, i, j, k, n, s, rk, ll, bl, erg;
  
  ll:=Set(l);
  if IsBound(W.parent) then
    W:=Parent(W);
  fi;
  
  # trivial case (fast handling of parabolic subgroups):
  if IsSubset([1..W.semisimpleRank],ll) then
    return ll;
  fi;
  
  refperm:=Reflections(W);
  gen:=refperm{ll};
  
  # compute orbit:
  tmp:=Set(Concatenation(
               PermGroupOps.Orbits(rec(generators:=gen),ll,OnPoints)));
  orb:=tmp{[1..Length(tmp)/2]};
  
  erg:=Set([]);
  
  # first check input (get the result faster, if input is set of simple roots):
  for i in ll do
    if i<=W.N then
      n:=0;
      for j in orb do
        if n<2 and j^refperm[i]>W.N then
          n:=n+1;
        fi;
      od;
      if n=1 then
        AddSet(erg,i);
      fi;
    fi;  
  od;
  if erg=ll then return erg; fi;
  
  for i in orb do
    if not i in ll then
      n:=0;
      for j in orb do
        if n<2 and j^refperm[i]>W.N then
          n:=n+1;
        fi;
      od;
      if n=1 then
        AddSet(erg,i);
      fi;
    fi;  
  od;
  
  return erg;
end;

#############################################################################
##
#F  CoxeterConjugacyClasses( <W> ) . . . . representatives of conjugacy classes
#F  of <W> as CoxeterWords
##  
##  classification is used, see 'CoxeterClassInfoIrred'.
##  
CoxeterConjugacyClasses:=W->ChevieClassInfo(W).classtext;

#############################################################################
##
#F  CoxeterClassInfoIrred( <type>, <l>[, <o> ]) . . . . . . . . information
#F  about conjugacy classes of simple Coxeter groups
##  
##  <l> must be a list of positive integers.  <type>, Length(<l>) (and <o>
##  if  <type>="I") determine a simple Coxeter  group W. 
##  'CoxeterClassInfoIrred' returns a record
##  containing  information about the conjugacy  classes of W, which has
##  three lists as components:
##  
##    .classtext   representatives of classes as words in elements of <l>,
##                 taken as standard generators of W
##    .classnames  corresponding names for the classes
##    .classparams  corresponding parameters for the classes
##  
##  The function  calls functions  CHEVIE.X.ClassInfo,  where X  describes the
##  type. See  these  functions for  more information about the single cases.
##  
CoxeterClassInfoIrred:=function(arg)
  local rank, cl, r, type;
  type:=arg[1];
  rank:=Length(arg[2]);
  r:=CHEVIE.Load(type,rank);
  if   type in ["A","~A","B","C","D"] then cl:=r.ClassInfo(rank);
  elif type in ["E","F","G"] then cl:=rec(classtext:=r.classtext,
         classnames:=r.classnames, classparams:=r.classnames);
  elif type="H" then cl:=rec(classtext:=r.classtext,
		 classnames:=List(r.classtext,IntListToString),
		 classparams:=List(r.classtext,IntListToString));
  elif type="I" then cl:=r.ClassInfo(arg[3]);
  fi;

  r:=MappingPermListList([1..rank],arg[2]);
  cl.classtext:=List(cl.classtext,x->OnTuples(x,r));
  return cl;
end;

#############################################################################
##
#V  CoxeterGroupOps  . . . . . .  operation record for Coxeter group category
##
##  'CoxeterGroupOps'  is  the  operation  record for   Coxeter  groups.  
##  It contains  the  functions  for   domain  operation,    e.g.,    'Size'
##  as well  as  the functions  for group    operations, e.g., 'Centralizer'.
##
##  'CoxeterGroupOps' is initially a copy of 'PermGroupOps', thus 
##  Coxeter groups inherit the default group functions, e.g., 
##  'DerivedSubgroup' and 'Index'. However  'CoxeterGroupOps'   overlays  
##  some of  those    functions with more  efficient ones, e.g., 'Size'.
##
CoxeterGroupOps := OperationsRecord("CoxeterGroupOps",PermGroupOps);

CoxeterGroupOps.Size:= W->Product(ReflectionDegrees(W));

CoxeterGroupOps.Elements:=function(W)
  # we compute at the same time all elements with given length:
  return Set(Concatenation(List([0..W.N],x->CoxeterElementsLength(W,x))));
end;

#############################################################################
##
#F  CoxeterGroupOps.CartanType( <W> ) . . . . . type for CoxeterGroup record
##  
##  This  is essentially the same   as CartanType(<W>.cartan), but with  one
##  exception: A subsystem of type "A" is said to be of type "~A", iff the
##  following conditions are fulfilled:
##  
##   - <W> is not a parent group
##   - the subsystem is contained in an irreducible subsystem of 
##     Parent(<W>), which is of type "B", "C", "G" or "F"
##   - the subsystem consists of short roots
##  
CoxeterGroupOps.CartanType:= function(W)
  local p, a, b, res;
  res:=CartanType(W.cartan);
  
  # parent or all roots of parent have same length:
  if not IsBound(W.parent) or Maximum(W.parent.rootLengths)=1 then
    return res;
  fi;
  
  for a in res do
    p:=W.rootInclusion[a[2][1]];
    if a[1]="A" and Parent(W).rootLengths[p]=1 then
      p:=Parent(W).orbitRepresentative[p];
      for b in CartanType(Parent(W)) do
        if p in b[2] and b[1] in ["B","C","F","G"] then
          a[1]:="~A";
        fi;
      od;
    fi;
  od;
  
  return res;
end;
        
#############################################################################
##
#F  CoxeterGroupOps.PrintDynkinDiagram( <W> )  . see corresponding dispatcher
#F  function
##  
CoxeterGroupOps.PrintDynkinDiagram:= function(W)
  local tmp, i;
  tmp:=Copy(CartanType(W));
  for i in [1..Length(tmp)] do
    tmp[i][2]:=W.rootInclusion{tmp[i][2]};
  od;
  PrintDynkinDiagram(tmp);
end;

#############################################################################
##
#F  CoxeterGroupOps.CartanName( <W> ) . see corresponding dispatcher function
##  
CoxeterGroupOps.CartanName:= W->CartanName(CartanType(W));

#############################################################################
##
#F  CoxeterGroupOps.ConjugacyClasses( <W> ) . . . . . . . 'ConjugacyClasses',
#F  using the classification
##  
##  see 'CoxeterConjugacyClasses'
##  
CoxeterGroupOps.ConjugacyClasses:=W->List(CoxeterConjugacyClasses(W),
     x->ConjugacyClass(W,PermCoxeterWord (W,x)));

#############################################################################
##
#F  CoxeterGroupOps.ChevieClassInfo( <W> ) . . . see corresponding dispatcher 
#F  function
##
CoxeterGroupOps.ChevieClassInfo:=function(W) local tmp, i;
  if not IsBound(W.classInfo) then
    
    tmp:=Copy(CartanType(W));
    for i in [1..Length(tmp)] do
      tmp[i][2]:=W.rootInclusion{tmp[i][2]};
    od;
    tmp:=List(tmp,x->ApplyFunc(CoxeterClassInfoIrred,x));

    W.classInfo:=rec(
      classtext:=List(Cartesian(List(tmp,x->x.classtext)),Concatenation),
      classparams:=Cartesian(List(tmp,x->x.classparams)),
      classnames:=List(Cartesian(List(tmp,x->x.classnames)),
	      function(l)local s;
		s:=Concatenation(List(l,x->Concatenation(x,",")));
		return String(s{[1..Length(s)-1]});
	      end));
  fi;
  return W.classInfo;
end;

###########################################################################
##
#F  CharName(W,p) . . . . . . . . Name of character with .charparam p
##
CoxeterGroupOps.CharName:=function(W,p)local t,i,s;
  s:=[]; t:=CartanType(W);
  for i in [1..Length(t)] do
    Append(s,CHEVIE.Load(t[i][1],Length(t[i][2])).CharName(p[i]));
    if i<>Length(t) then Append(s,",");fi;
  od;
  return String(s);
end;

###########################################################################
##
#F  GetChevieInfo(W) . . . . . . . . . . get information
##
CoxeterGroupOps.GetChevieInfo:=function(W,fname)local t,type,rank,res,p;
  res:=[];
  for t in CartanType(W) do
    rank:=Length(t[2]);
    type:=t[1];
    p:=CHEVIE.Load(type,rank);
    if   type in ["A","~A","B","C","D"] then p:=p.(fname)(rank);
    elif type in ["E","H","F", "G"] then p:=p.(fname)();
    elif type="I" then p:=p.(fname)(t[3]);
    fi;
    Add(res,p);
  od;
  return res;
end;

###########################################################################
##
#F  CharParams(W) . . . . . . . . . . params of characters of W
##
CoxeterGroupOps.CharParams:=function(W)local t,type,rank,res,p;
  if not IsBound(W.charparam) then
    W.charparam:=Cartesian(GetChevieInfo(W,"CharParams"));
  fi;
  return W.charparam;
end;

###########################################################################
##
#F  PositionId(W) . . . . . . . . . . position of identity character
##
CoxeterGroupOps.PositionId:=function(W)local pos,prod,res,p,n,i;
  if not IsBound(W.positionId) then
    p:=GetChevieInfo(W,"PositionId");
    n:=List(GetChevieInfo(W,"CharParams"),Length);
    pos:=1;prod:=1;
    for i in [Length(n),Length(n)-1..1] do
      pos:=pos+(p[i]-1)*prod;
      prod:=prod*n[i];
    od;
    W.positionId:=pos;
  fi;
  return W.positionId;
end;

###########################################################################
##
#F  PositionSgn(W) . . . . . . . . . . position of sign character
##
CoxeterGroupOps.PositionSgn:=function(W)local pos,prod,res,p,n,i;
  if not IsBound(W.positionSgn) then
    p:=GetChevieInfo(W,"PositionSgn");
    n:=List(GetChevieInfo(W,"CharParams"),Length);
    pos:=1;prod:=1;
    for i in [Length(n),Length(n)-1..1] do
      pos:=pos+(p[i]-1)*prod;
      prod:=prod*n[i];
    od;
    W.positionSgn:=pos;
  fi;
  return W.positionSgn;
end;

#############################################################################
##
#F  CoxeterGroupOps.ChevieCharInfo( <W> ). . . . see corresponding dispatcher 
#F  function
##
CoxeterGroupOps.ChevieCharInfo:=function(W) local t,c,type,rank,p;
  if not IsBound(W.charInfo) then
    t:=CharParams(W);
    W.charInfo:=rec(charparams:=t, charnames:=List(t,i->CharName(W,i)),
     a:=LowestPowerGenericDegrees(W), A:=HighestPowerGenericDegrees(W),
     b:=LowestPowerFakeDegrees(W), B:=HighestPowerFakeDegrees(W),
     positionId:=CoxeterGroupOps.PositionId(W),
     positionSgn:=CoxeterGroupOps.PositionSgn(W));
  fi;
  c:=CartanType(W);
  if Length(c)=1 then 
    W.charInfo.positionRefl:=GetChevieInfo(W,"PositionRefl")[1];
    rank:=Length(c[1][2]);
    type:=c[1][1];
    p:=CHEVIE.Load(type,rank);
    if type="F" then W.charInfo.kondo:=p.kondo;
    elif type="E" then W.charInfo.frame:=p.frame;
    fi;
  fi;
  return W.charInfo;
end;

#############################################################################
##
#F  CoxeterGroupOps.CharTable( <W> ) . . . . . . . . . . . 'CharTable', using
#F  the classification
##
##  see 'CoxeterCharTableIrred'.
##
CoxeterGroupOps.CharTable:=function(W)
  local t, l, tmp, i, res, tbl, cl, f;

  tmp:=Copy(CartanType(W));
  for i in [1..Length(tmp)] do
    tmp[i][2]:=W.rootInclusion{tmp[i][2]};
  od;

  l:=List(tmp,t->ApplyFunc(CoxeterCharTableIrred,t));
  if Length(l)>0 then 
    res:=l[1];
  else 
    res:=Group(());res.name:="";
    res:=GroupOps.CharTable(res);
    Unbind(res.group);
  fi;

  for t in l{[2..Length(l)]} do
    res:=CharTableDirectProduct(res,t); 
    Unbind(res.fusionsource);
    Unbind(res.fusions);
  od;
  
  res.cartan:=W.cartan;
  res.irredinfo:=List(CharParams(W),x->rec(charparam:=x,
                                           charname:=CharName(W,x)));
  cl:=ChevieClassInfo(W);
  for f in RecFields(cl) do res.(f):=cl.(f);od;
  if IsBound(W.name) then
    res.name:=String(Concatenation("W( ",W.name," )"));
  else
    # default:
    res.name:=String(Concatenation("W( ",CartanName(W)," )"));
  fi;
  
  res.identifier:=res.name;
  return res;
end;

#############################################################################
##
#F  CoxeterGroupOps.ClassInvariants( <W> ) . . . . . 'ClassInvariants' for 
##  Coxeter groups
#F  CoxeterGroupOps.PositionClass( <W>, <w> ) . . . . . . 'PositionClass' for 
##  Coxeter groups
##  
CoxeterGroupOps.ClassInvariants:=CoxeterGroupOpsClassInvariants;
CoxeterGroupOps.PositionClass:=PermGroupOps.PositionClass;

#############################################################################
##
#F  CoxeterGroupOps.FusionConjugacyClasses( <W1>, <W2> )  . . . . . overwrite
#F  'FusionConjugacyClasses' for Coxeter groups
##  
##  'FusionConjugacyClasses'   is overwritten   for permutation groups  by
##  reading classinv.g.
##  
CoxeterGroupOps.FusionConjugacyClasses:=PermGroupOps.FusionConjugacyClasses;

#############################################################################
##
#F  CoxeterGroupOps.\=( <W1>, <W2> )  . . . equality test for Coxeter groups
##  
##  Here  two CoxeterGroup records  are defined to  be equal if  they are 
##  equal as permutation  groups, and if the simple  roots,  the simple 
##  coroots and the groups .omega are equal.
##  
CoxeterGroupOps.\= := function(W1,W2)
  if not( IsRec(W1) and IsRec(W2) and IsBound(W1.isCoxeterGroup) and
          IsBound(W2.isCoxeterGroup)) then
    return false;
  elif W1.simpleRoots<>W2.simpleRoots 
                     or W1.simpleCoroots<>W2.simpleCoroots then
    return false;
  elif PermGroupOps.\=(W1,W2)=false then
    return false;
  elif IsBound(W1.omega) then
    if IsBound(W2.omega) then
      return W1.omega=W2.omega;
    else
      return Size(W1.omega)=1;
    fi;
  else
    if IsBound(W2.omega) then
      return Size(W2.omega)=1;
    else
      return true;
    fi;
  fi;
end;

#############################################################################
##
#F  CoxeterGroupOps.Print( <W> ) . . . . . . . . . . . 'Print' function for 
#F  CoxeterGroup records
##  
##  If <W>.name is bound then this component is printed.
##  
CoxeterGroupOps.Print:=function(W)
  local a;
  if IsBound(W.name) then 
    Print(W.name);
  elif not IsBound(W.parent) then
     # this would be similar to old version of the "weyl" package:   
##if not IsBound(W.omega) or Size(W.omega)=1 then
##  Print("CoxeterGroup(",W.simpleRoots,", ",W.simpleCoroots,")");
##else
##  Print("CoxeterGroup(",W.simpleRoots,", ",W.simpleCoroots,", ",W.omega,")");
##fi;
    Print("CoxeterGroup(");
    for a in [1..Length(W.createArgs)] do
      if IsString(W.createArgs[a]) then
        Print("\"",W.createArgs[a],"\"");
      else
        Print(W.createArgs[a]);
      fi;
      if a<>Length(W.createArgs) then
	Print(", ");
      fi;
    od;
    Print(")");
  else
    if not IsBound(W.omega) or Size(W.omega)=1 then
      Print("ReflectionSubgroup(",W.parent,", ",
            W.rootInclusion{[1..W.semisimpleRank]},")");
    else
      Print("ReflectionSubgroup(",W.parent,", ",
            W.rootInclusion{[1..W.semisimpleRank]},
            ", ",W.omega,")");
    fi;
  fi;
end;

#############################################################################
##
#F  ReflectionDegrees( <W>)  . . . . . . . degrees as a reflection group
##
##  Returns the degrees of the  reflection group W,
##  as an increasing list of integers.
##  
CoxeterGroupOps.ReflectionDegrees:=function(W)local p,type,res;
  if ForAll( Concatenation( W.cartan ), IsInt )  then
# Crystallographic group: general algorithm
    p:=List(Collected(List(W.roots{[1..W.N]},Sum)),x->x[2]);
    if p=[] then res:= [];
    else res:= Reversed(1+List([1..Maximum(p)],x->Number(p,y->y>=x)));
    fi;
  else 
# else use the classification
    type:=CartanType(W);
    if Length(type)=1 and IsList(type[1]) then
      if type[1][1]="H" then 
	if Length(type[1][2])=4 then res:= [2,12,20,30];
	  elif Length(type[1][2])=3 then res:= [2,6,10];
	  else res:= [2,5];
	fi;
      else res:= [2,type[1][3]]; # type I2
      fi;
    else
      res:=Concatenation(List(type,p->
	     ReflectionDegrees(ReflectionSubgroup(W,W.rootInclusion{p[2]}))));
      Sort(res);
    fi;
  fi;
  return Concatenation(List([W.semisimpleRank+1..W.rank],x->1),res);
end;
    
CoxeterGroupOps.Hecke:=function(arg)
  ReadChv("prg/hecke");return ApplyFunc(CoxeterGroupOps.Hecke,arg);
end;

#############################################################################
##
#F  ReflectionCharValue( <W>, <w> ) ... The reflection character on w
##
##  'ReflectionCharValue' returns  the value of the reflection character of W 
##  at the element w.
##  
CoxeterGroupOps.ReflectionCharValue:=function(W,w)
  return W.rank-W.semisimpleRank+Sum([1..W.semisimpleRank],
     i->W.roots[W.rootRestriction[W.rootInclusion[i]^w]][i]);
end;

############################################################################
##
#F  FakeDegrees( <W>, <q> ) . . . . . .  Fake Degrees of Coxeter group <W>
##
##  This returns the list of polynomials describing the multiplicity of
##  each character in the graded version of the regular representation given
##  by the quotient S/I where S is the symmetric algebra of the reflection
##  representation and I is the ideal generated by the homogenous invariants
##  of positive degree in S.
##  The ordering of the result corresponds to the ordering of the characters 
##  in the  CharTable.
##
#
CoxeterGroupOps.FakeDegrees:=function(W,q)
  return List(CharParams(W),p->FakeDegree(W,p,q));
end;

#############################################################################
##
#F  PermCoxeterWord ( <W> , <w> ) . .  convert a word to a permutation
##
##  'PermCoxeterWord ' returns the permutation on the root vectors determined
##  by  the  element  w  of  W,  given  as  a  list  of integers
##  representing reflexions of the parent.
##  

PermCoxeterWord  := function ( W, w )
    if Length(w)=0 then return ();
    else return Product(W.generators{W.rootRestriction{w}});fi;
end;

#############################################################################
##
#F  CoxeterWord( <W> , <w> )  . . .  convert a permutation to a reduced word
##  
##  'CoxeterWord' returns a reduced word in  the standard generators of W
##  determined by the    permutation w in  W   on the  root   vectors. The
##  generators       are   denoted      by       their     position     in
##  Reflections(Parent(<W>)).
##  
CoxeterWord := function( W, w )local  i,l,I,WI;
    l := [  ];
    if CHEVIE.BrieskornNormalForm then
      while true do
	I:=LeftDescentSet(W,w);
	if Length(I)=0 then return l;fi;
	if Length(I)=1 then # speedup by special-coding |I|=1
	  Add(l,I[1]);
	  w:=W.generators[W.rootRestriction[I[1]]]*w;
	else
	  WI:=ReflectionSubgroup(W,I);
	  Append(l,LongestCoxeterWord(WI));
	  w:=LongestCoxeterElement(WI)*w;
	fi;
      od;
    else
      while true  do
	  i := 0;
	  repeat
	      if i>=W.semisimpleRank then return l;fi;
	      i := i + 1;
	  until W.rootInclusion[i] ^ w > W.parentN;
	  Add( l, W.rootInclusion[i] );
	  w := W.generators[i] * w;
      od;
    fi;
end;

#############################################################################
##
#F  MatXPerm( <W>, <w> )  . . . . . . . . . . . . convert a permutation to
#F  corresponding matrix operating on X
#F  MatYPerm( <W>, <w> )  . . . . . . . . . . . . convert a permutation to
#F  corresponding matrix operating on Y
##  
##  Let <w>  be a  permutation with the  following property: The images of
##  the        simple        roots         of       <W>      (       i.e.,
##  <W>.rootInclusion{[1..<W>.semisimpleRank]}  )  must be roots of  <W> (
##  i.e., in  the set <W>.rootInclusion ).   
##  
##  Let X_1 be the  sublattice of X  consisting of the elements orthogonal
##  to  all   coroots of <W>.    'MatXPerm'  returns the unique invertible
##  matrix which, as matrix acting  on X, maps the  simple roots of <W> as
##  indicated by <w> and which induces the identity map on X_1.
##  
##  If in particular <w> is  an element of  <W> then 'MatXPerm( <W>, <w> )
##  is  the matrix   representing <w> as  linear map   on X. This  follows
##  immediately from the formula for the generating reflections of <W>.
##  
##  'MatYPerm(  <W>, <w> )'  returns   the transposed of 'MatXPerm(   <W>,
##  <w>^-1 )'.  If in particular <w> is an element  of <W> then 'MatYPerm(
##  <W>, <w> )' is the matrix representing <w> as linear map on Y.
##  
MatXPerm:=function(W, w)
  local tmp, i, M;
  
  if W.semisimpleRank=0 then return IdentityMat(W.rank);fi;

  if not IsBound(W.forMatX) then
    # preparing for fast computation of matrix (operation is trivial 
    # on space orthogonal to all simple coroots)
    tmp:=W.simpleRoots{[1..W.semisimpleRank]};
    if W.semisimpleRank<W.rank then
      Append(tmp,NullspaceMat(TransposedMat(W.simpleCoroots)));
    fi;
    W.forMatX:=[0*[W.semisimpleRank+1..W.rank],
                IdentityMat(W.rank){[W.semisimpleRank+1..W.rank]},
                tmp,tmp^-1];
  fi;

  tmp:=W.forMatX;
  M:=[];
  for i in W.rootInclusion{[1..W.semisimpleRank]} do
    Add(M,Concatenation(W.roots[W.rootRestriction[i^w]],tmp[1]));
  od;
  return tmp[4]*Concatenation(M,tmp[2])*tmp[3];
end;

MatYPerm:=function(W,w)
  return TransposedMat(MatXPerm(W,w^-1));
end;

#############################################################################
##
#F  PermMatX( <W> , <M> )  . . . . . . . . .  convert a matrix in X which
#F  preserves the roots to permutation of the roots of Parent(<W>)
#F  PermMatXCoxeterElement( <W> , <M> )  . . . . convert a matrix in X which
#F  represents an element of Parent(<W>) to correspnding permutation
#F  PermMatY( <W> , <M> )  . . . . . . . . .  convert a matrix in Y which
#F  preserves the coroots to permutation of the roots of Parent(<W>)
##  

# this is faster than 'PermMatX', but works for elements of the Coxeter 
# group only:
PermMatXCoxeterElement:=function(W, M)
  local tmp;
  
  # we always have to look at the parent group:
  if IsBound(W.parent) then
    W:=W.parent;
  fi;

  # the images of the points Base(W) determine the element uniquely:
  tmp:=W.roots*W.simpleRoots;
  tmp:=List(Base(W),p->Position(tmp,tmp[p]*M));
  return RepresentativeOperation(W,Base(W),tmp,OnTuples);
end;

PermMatX:=function(W, M)
  local tmp;

  # we always have to look at the parent group:
  if IsBound(W.parent) then
    W:=W.parent;
  fi;

  tmp:=W.roots*W.simpleRoots;
  return PermListList(tmp,tmp*M);
end;

PermMatY:=function(W, M)
  return PermMatX(W, TransposedMat(M))^-1;
end;

#############################################################################
##
#F  AddComponentsCoxeterGroup ( ... )  only used internally by 'CoxeterGroup'
#F  and 'ReflectionSubgroup'
##  
# arguments: res [,parent]
AddComponentsCoxeterGroup :=function(arg)
  local res, parent, tmp, i, p;

  res:=arg[1];
  if Length(arg)>1 then
    parent:=arg[2];
  else
    parent:=res;
  fi;

  if IsBound(parent.rank) then
    res.rank:=parent.rank;
  elif res.semisimpleRank>0 then
    res.rank:=Length(res.simpleRoots[1]);
  else
    res.rank:=0;
  fi;

  # root lengths from parent group, if already known:
  if IsBound(parent.rootLengths) then
    res.rootLengths:=parent.rootLengths{res.rootInclusion};
  fi;

  # for each root in a parent group we determine a simple root in 
  # the same W-orbit:
  tmp:=PointsAndRepresentativesOrbits(res,2*res.parentN);
  res.orbitRepresentative:=[];
  res.orbitRepresentativeElements:=[];
  for i in [1..Length(tmp[1])] do 
    if tmp[1][i][1] in res.rootInclusion then
      res.orbitRepresentative{res.rootRestriction{tmp[1][i]}}
                                                :=0*tmp[1][i]+tmp[1][i][1];
      res.orbitRepresentativeElements{res.rootRestriction{tmp[1][i]}}
                                                               :=tmp[2][i];
    fi;
  od;

  # root lengths (for parent group):
  if not IsBound(res.rootLengths) then
    res.rootLengths:=[];
    # first for the simple roots using the classification:
    for tmp in CartanType(res) do
      i:=tmp[2];
      if tmp[1]="B" then
        res.rootLengths{i}:=0*i+2;
        res.rootLengths[i[1]]:=1;
      elif tmp[1]="C" then
        res.rootLengths{i}:=0*i+1;
        res.rootLengths[i[1]]:=2;
      elif tmp[1]="F" then
        res.rootLengths{i}:=[2,2,1,1];
      elif tmp[1]="G" then
        res.rootLengths{i}:=[3,1];
      elif tmp[1]="I" and tmp[3] mod 2 = 0 then
        res.rootLengths{i}:=[tmp[3]/2,1];
      else
        res.rootLengths{i}:=0*i+1;
      fi;
    od;
    for i in [res.semisimpleRank+1..2*res.N] do
      res.rootLengths[i]:=res.rootLengths[res.orbitRepresentative[i]];
    od;
  fi;
  
  # the set of coroots: i-th entry is coroot to roots[i]
  if res.cartan=TransposedMat(res.cartan) then
    res.coroots:=res.roots;
  elif not IsBound(parent.coroots) then
    
    # the dual group as permutation group on the coroots:
    tmp:=RootsAndMatgens(TransposedMat(res.cartan));
    tmp.rootRestriction:=[1..2*tmp.N];
    tmp.generators:=List(tmp.roots*tmp.matgens,SortingPerm)/
                                                   SortingPerm(tmp.roots);
    res.coroots:=[];
    for i in [1..res.N] do
      p:=res.orbitRepresentative[i];
      p:=p^PermCoxeterWord (tmp,CoxeterWord(res,
					res.orbitRepresentativeElements[i]));
      Add(res.coroots,tmp.roots[p]);
    od;
    Append(res.coroots,-res.coroots);
  else
    tmp:=[parent.coroots{res.rootInclusion},
          RootsAndMatgens(TransposedMat(res.cartan)).roots{[1..res.N]},
          parent.coroots{res.rootInclusion{[1..res.semisimpleRank]}}];
    res.coroots:=[];
    for p in tmp[2] do
      res.coroots[Position(tmp[1],p*tmp[3])]:=p;
    od;
    Append(res.coroots,-res.coroots);
  fi;

  # Order of the Coxeterp group 
  # (This doesn't use a stabilizer chain but allows a much faster 
  # construction of a stabilizer chain)
  # here also the 'CartanType' is computed.
  Size(res);

end;

#############################################################################
##  
#F  CoxeterGroup( <cartan>[, <omega>])
#F  CoxeterGroup( <roots>, <coroots>[, <omega>])
#F  CoxeterGroup( <type>, <rank>[, <type>, <rank> ...][, "sc"][, <omega>] )
#F  CoxeterGroup( <rec> )
##  
CoxeterGroup:=function(arg)
  local l, sc, om, W, typ, a, tmp, t;

  l:=Length(arg);
  
  # get back the .coxeter entry of a record:
  if l=1 and IsRec(arg[1]) and IsBound(arg[1].coxeter) then
    return arg[1].coxeter;
  fi;
  
  if l>0 and IsGroup(arg[l]) then
    om:=arg[l];
    l:=l-1;
  fi;
  if l>0 and arg[l]="sc" then
    sc:=true;
    l:=l-1;
  else
    sc:=false;
  fi;

  W:=rec();
  if l=2 and IsMat(arg[1]) then
    if sc then 
      Error("CoxeterGroup: argument \"sc\" not allowed ",
            "together with 2 matrices\n");
    fi;
    W.simpleRoots:=arg[1];
    W.simpleCoroots:=arg[2];
    W.cartan:=W.simpleCoroots*TransposedMat(W.simpleRoots);
  else
    if l=0 then
      W.cartan:=[];
    elif IsString(arg[1]) then
      W.cartan:=ApplyFunc(CartanMat,arg{[1..l]});
    else
      W.cartan:=arg[1];
    fi;
    if sc then
      W.simpleRoots:=TransposedMat(W.cartan);
      W.simpleCoroots:=IdentityMat(Length(W.cartan));
    else
      W.simpleRoots:=IdentityMat(Length(W.cartan));
      W.simpleCoroots:=W.cartan;
    fi;
  fi;

  tmp:=RootsAndMatgens(W.cartan);
  for t in RecFields(tmp) do W.(t):=tmp.(t);od;
  if W.roots=[] then
    tmp:=Group([],()); # because []*[] not allowed in GAP!
  else
##     tmp:=Group(List(W.roots*W.matgens,Sortex)* Sortex(ShallowCopy(
##  	  W.roots))^-1,()); # ShallowCopy since Sortex changes its argument!
    tmp:=Group(List(W.roots*W.matgens,SortingPerm)/SortingPerm(W.roots),());
  fi;
  for t in RecFields(tmp) do W.(t):=tmp.(t);od;

  W.rootInclusion := [1..2*W.N];
  W.rootRestriction:=[1..2*W.N];
  W.parentN:=W.N;
  W.isCoxeterGroup := true;
  W.operations:=CoxeterGroupOps;

  AddComponentsCoxeterGroup (W);

  if IsBound(om) then
    if IsPermGroup(om) then
      if W.semisimpleRank<> W.rank then
        Error("#I Omega can be permutation ",
              "group only for semisimple datum.\n");
      fi;
      om:=Group(List(om.generators,x->MatXPerm(W,x)),
                IdentityMat(W.rank));
    fi;
    if not ForAll(om.generators,M->IsNormalizing(W.roots*W.simpleRoots,M) and
               IsNormalizing(W.coroots*W.simpleCoroots,TransposedMat(M))) then
      Error("#I Generators of Omega should normalize roots and coroots.\n");
    fi;
    W.omega:=om;
    tmp:=W.roots*W.simpleRoots;
    W.omega.elementsPerm:=List(Elements(W.omega),M->PermListList(tmp,tmp*M));
  fi;
  
  # for printing:
  # simplify arguments: first check if one matrix in argument is 
  # superfluous:
  if l=2 and W.rank=W.semisimpleRank and IsMat(arg[1]) then
    tmp:=IdentityMat(W.rank);
    if arg[1]=tmp then
      arg:=arg{[2..Length(arg)]};
      l:=1;
    elif arg[2]=tmp then
      arg[1]:=TransposedMat(arg[1]);
      arg[2]:="sc";
      l:=1;
    fi;
  fi;

  # check in case of one matrix if it can be created by 'CartanMat':
  if l=1 then
    tmp:=Copy(CartanType(W));
    Sort(tmp,function(a,b) return a[2]<b[2]; end);
    typ:=[];
    for a in tmp do
      if typ<>false then
        if a[1] in ["B","C"] and Length(a[2])=2 then
          if a[2][1]+1=a[2][2] then
            Append(typ,[a[1],Length(a[2])]);
          elif a[2][2]+1=a[2][1] then
            Append(typ,Difference(["B","C"],[a[1]]));
            Add(typ,2);
          else
            typ:=false;
          fi;
        else
          if a[2]=[a[2][1]..a[2][1]+Length(a[2])-1] then
            if a[1]<>"I" then
              Append(typ,[a[1],Length(a[2])]);
            else
              Append(typ,[a[1],Length(a[2]),a[3]]);
            fi;
          else
            typ:=false;
          fi;
        fi;
      fi;
    od;
    if typ<>false then
      arg:=Concatenation(typ,arg{[2..Length(arg)]});
    fi;
  fi;
  W.createArgs:=ShallowCopy(arg);

  return W;
end;

#############################################################################
##
#F  OmegaSubgroup( <W>, <ws>[, <oms> ]) . . . . . . . . . . . . . subgroup of 
#F  < <W>, <W>.omega > as matrix group
##  
##  <ws> must be list of  elements of <W>,  <oms> must be list of elements
##  of  <W>.omega.  'OmegaSubgroup'  returns  the   matrix group (matrices
##  acting on X) generated by the elements in <ws> and <oms>.
##  
OmegaSubgroup:=function(arg)
  local W, wgen, ogen, gen, tmp, erg;
  
  if IsBound(arg[1].parent) then
    W:=arg[1].parent;
  else
    W:=arg[1];
  fi;
  
  wgen:=arg[2];
  if Length(arg)>2 then
    ogen:=arg[3];
  else
    ogen:=[];
  fi;
  
  if not ForAll(wgen,x->x in W) and (
          (IsBound(W.omega) and ForAll(ogen,x->x in W.omega)) or ogen=[]) then
    Error("#I second argument must be list of elements in parent of",
          " first argument W,\n",
          "#I third argument must be list of elements of W.omega.\n");
  fi;
  erg:=Group(Concatenation(List(wgen,x->MatXPerm(W,x)),ogen),
             IdentityMat(W.rank));
  tmp:=W.roots*W.simpleRoots;
  erg.elementsPerm:=List(Elements(erg),M->PermListList(tmp,tmp*M));
  
  erg.Wgenerators:=wgen;
  erg.Omegagenerators:=ogen;
  return erg;
end;


#############################################################################
##
#F  ReflectionSubgroup( <W>, <gens>[, <Omega> ] ) . . . . . 
##  

ReflectionSubgroup:=function(arg)
  local W, simpleroots, tmp, Om, res, i, n, f, r, permrefs, gens, pos, callarg;
  
  if IsBound(arg[1].parent) then
    W:=arg[1].parent;
  else
    W:=arg[1];
  fi;
  
##    gens:=arg[1].rootInclusion{arg[2]};
  # this is now changed, generators must be given as reflections of the
  # parent group:
  gens:=arg[2];
  
  if Length(arg)>2 then
    Om:=arg[3];
  fi;
  
  # components for caching subgroups:
  if not IsBound(W.reflectionSubgroups) then
    W.reflectionSubgroups:=[];
    W.reflectionSubgroupArgs:=[];
  fi;
  
  # checking if subgroup in cache:
  callarg:=[gens];
  if IsBound(Om) then 
    Add(callarg,Om.generators);
  fi;
  pos:=PositionSorted(W.reflectionSubgroupArgs,callarg);
  if IsBound(W.reflectionSubgroupArgs[pos]) and 
                                  W.reflectionSubgroupArgs[pos]=callarg then
    return W.reflectionSubgroups[pos];
  fi;
  
  # determine a set of simple roots:
  simpleroots:=SimpleRootsSubsystem(W,gens);
  if Set(gens)=simpleroots and Length(gens)=Length(Set(gens)) then
    simpleroots:=gens;
  fi;
  
  gens:=Reflections(W){simpleroots};

  # The command 'Subgroup' returns W, if Set(gens)=Set(W.generators).
  # To avoid this we substitute it by some more lines.
#  res:=Subgroup(W,gens);
  res:=rec(
    isDomain    := true,
    isGroup     := true,
    parent      := W,
    identity    := (),
    generators  := Copy(gens),
    isPermGroup := true);
  for i in [1..Length(gens)] do
    res.(i):=gens[i];
  od;
 
  res.parent:=W;
  res.operations:=CoxeterGroupOps;
  res.isCoxeterGroup:=true;
  res.parentN:=W.N;

  # case of empty simpleroots:
  if simpleroots=[] then
    res.simpleRoots:=[];
    res.simpleCoroots:=[];
    res.cartan:=[];
    res.matgens:=[];
    res.roots:=[];
    res.N:=0;
    res.rootInclusion:=[];
    res.rootRestriction:=[];
    res.semisimpleRank:=0;
  else
    res.simpleRoots:=W.roots{simpleroots}*W.simpleRoots;
    res.simpleCoroots:=W.coroots{simpleroots}*W.simpleCoroots;
    res.cartan:=W.coroots{simpleroots}*
                        W.cartan*TransposedMat(W.roots{simpleroots});
    tmp:=RootsAndMatgens(res.cartan);
    for f in RecFields(tmp) do
      res.(f):=tmp.(f);
    od;

    if res.roots<>[] then r:=res.roots*W.roots{simpleroots};
                   else r:=[];
    fi;
    res.rootInclusion:=ListBlist([1..2*W.N],BlistList(W.roots,r));
    res.rootInclusion:=Permuted(res.rootInclusion,
                            PermListList(r,W.roots{res.rootInclusion}));

    res.rootRestriction:=[];
    res.rootRestriction{res.rootInclusion}:=[1..2*res.N];

    res.reflectionsPerm:=W.reflectionsPerm{res.rootInclusion};
  fi;
  
  if IsBound(Om) then
    tmp:=Set(res.rootInclusion);
    if not ForAll(Om.elementsPerm,x->
                             Set(OnTuples(res.rootInclusion,x))=tmp) then
      Error("#I Omega as permutation group must normalize the ",
            "CoxeterGroup subgroup.\n");
    fi;
    res.omega:=Om;
  fi;
  
  AddComponentsCoxeterGroup (res,W);

  # caching the result (in position 'pos'):
  W.reflectionSubgroups{[pos+1..Length(W.reflectionSubgroups)+1]}:=
          W.reflectionSubgroups{[pos..Length(W.reflectionSubgroups)]};
  W.reflectionSubgroupArgs{[pos+1..Length(W.reflectionSubgroupArgs)+1]}:=
          W.reflectionSubgroupArgs{[pos..Length(W.reflectionSubgroupArgs)]};
  W.reflectionSubgroups[pos]:=res;
  W.reflectionSubgroupArgs[pos]:=callarg;
  
  return res;
end;

############################################################################
##
#F  FakeDegree( <W>, <p>, <q> ) . . .Fake Degree of char with charparam <p>
##  
##  This   returns the   polynomial  describing the   multiplicity of  the
##  character chi with .charparam  p in the graded  version of the regular
##  representation  given  by the quotient  S/I  where S  is the symmetric
##  algebra of the reflection representation and  I is the ideal generated
##  by the homogenous invariants of positive degree in S.
##  
FakeDegree:=function(W,p,q)local rank,res,t,type,r,i,v;
  res:=[];t:=CartanType(W);
  for i in [1..Length(t)] do
    type:=t[i][1];
    rank:=Length(t[i][2]);
    r:=CHEVIE.Load(type,rank);
    if type in ["A","~A"] then Add(res,r.FakeDegree(p[i],q));
    elif type in ["B","C","D"] then Add(res,Value(r.vcycFakeDegree(p[i]),q));
    elif type in ["E","F","G","H"] then 
        Add(res,FastValue(r.vpolfakedegrees[Position(r.CharParams(),p[i])],q));
    elif type="I" then Add(res,r.FakeDegree(t[i][3],p[i],q));
    fi;
  od;
  return Product(res);
end;
