#############################################################################
##
#A  wclassinv.g               CHEVIE library                     Frank Luebeck
##
#A  $Id: wclsinv.g,v 1.1 1997/01/21 13:46:37 gap Exp $
##  
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##  
##  This  file contains   special  functions for  computing invariants  of
##  conjugacy classes in Coxeter groups and Coxeter cosets.
##  
##  The main difference   to the general functions   'ClassInvariants' and
##  'PositionClass' for permutation groups is the use of the 'ClassParamX'
##  functions defined  for the  different  types of simple  groups. If the
##  class of  an element cannot be identified  cheaply then the element is
##  transformed into a   word and the CHEVIE  label  for its corresponding
##  conjugacy class is computed by quite efficient functions.
##  

#############################################################################
##
#F  CoxeterGroupOps.ClassInvariants(  ... )  . . . . . 'ClassInvariants' for
#F  Coxeter groups
#F  CoxeterGroupOps.PositionClass( ... ) . . . . . . . . . 'PositionClass' for
#F  Coxeter groups
##
##  Here we use the   CoxeterClassParamIrred function to  distinguish classes
##  which are not distinguished by cycle  type (and cycle type of elements
##  multiplied by center elements and not lying in small classes).
##  
# the command 
#     CoxeterGroupOps.ClassInvariants:=CoxeterGroupOpsClassInvariants;
# is in coxeter.g
CoxeterGroupOpsClassInvariants:=function(arg)
  if IsList(arg[1]) then
    arg:=arg[1];
  fi;
  if Length(arg)=1 then
    return PermGroupOps.ClassInvariants(arg[1],
                   CentreMultFunction,
                   ShortClassListFunction,
                   CoxeterClassParamCheckFunction);
  else
    return PermGroupOps.ClassInvariants(arg);
  fi;
end;

# same as for permutation groups ( assignment in coxeter.g)
# CoxeterGroupOps.PositionClass:=PermGroupOps.PositionClass;

#############################################################################
##
#F  CoxeterCosetOps.ClassInvariants(  ... )  . . . . . 'ClassInvariants' for
#F  Coxeter groups
#F  CoxeterCosetOps.PositionClass( ... ) . . . . . . . 'PositionClass' for
#F  Coxeter groups
##
##  Here we use the   CoxeterClassParamIrred function to  distinguish classes
##  which are not distinguished by cycle  type (and cycle type of elements
##  multiplied by center elements and not lying in small classes).
##  
# the command 
#     CoxeterCosetOps.ClassInvariants:=CoxeterCosetOpsClassInvariants;
# is in coset.g

CoxeterCosetOpsClassInvariants:=function(arg)
  local tmp, res;
  if IsList(arg[1]) then
    arg:=arg[1];
  fi;
  
  # in non twisted case use the function for Coxeter groups:
  if arg[1].phi=() then
    return ClassInvariants(arg[1].coxeter);
  fi;
  
  tmp:=arg[1].coxeter.generators;
  if arg[1].phi<>() then
    Add(tmp,arg[1].phi);
  fi;
  tmp:=Group(tmp,());
  tmp.conjugacyClasses:=ConjugacyClasses(arg[1]);
  tmp.size:=Size(arg[1]);
  tmp.coset:=arg[1];
  
  if Length(arg)=1 then
    res:=PermGroupOps.ClassInvariants(tmp,
                   ShortClassListFunction,
                   CoxeterCosetClassParamCheckFunction);
  else
    Error("No function arguments allowed in this case.\n");
  fi;
  return res;
end;

# same as for permutation groups ( assignment in coset.g):
# CoxeterCosetOps.PositionClass:=PermGroupOps.PositionClass;

# read dispatcher functions:
PositionClass;

#############################################################################
##
#F  ComponentWordsPerm( <W>, <w> ) . . . . . . . 
#F  ComponentWordsPermCoset( <WF>, <w> ). . . . . similar to 'CoxeterWord'
#F  but giving lists of  words in standard generators of simple components
##    
ComponentWordsPerm:=function( W, w )
  local  i, l, cw, typ;
  
  if IsBound(W.forComponentWords) then
    cw:=W.forComponentWords;
  else
    cw:=[];
    typ:=CartanType(W);
    for i in [1..Length(typ)] do
      for l in [1..Length(typ[i][2])] do
        cw[typ[i][2][l]]:=[i,l];
      od;
    od;
    W.forComponentWords:=cw;
  fi;
  
  l := List(W.type,i->[]);
  while true  do
    i := 0;
    repeat
        if i>=W.semisimpleRank then return l;fi;
        i := i + 1;
    until W.rootInclusion[i] ^ w > W.parentN;
    Add( l[cw[i][1]], cw[i][2]);
    w := W.generators[i] * w;
  od;
end;

ComponentWordsPermCoset:=function( WF, w )
  local  W, i, j, l, cw, typ;

  W:=WF.coxeter;
  w:=w/WF.phi;

  if IsBound(WF.forComponentWordsCoset) then
    cw:=WF.forComponentWordsCoset;
  else
    cw:=[];
    typ:=CartanType(WF);
    for i in [1..Length(typ)] do
      for j in [1..Length(typ[i].orbit)] do
        for l in [1..Length(typ[i].orbit[j][2])] do
          cw[typ[i].orbit[j][2][l]]:=[i,j,l];
        od;
      od;
    od;
    WF.forComponentWordsCoset:=cw;
  fi;

  l := List(WF.type,i->List(i.orbit,j->[]));
  while true  do
    i := 0;
    repeat
        if i>=W.semisimpleRank then return l;fi;
        i := i + 1;
    until W.rootInclusion[i] ^ w > W.parentN;
    Add( l[cw[i][1]][cw[i][2]], cw[i][3]);
    w := W.generators[i] * w;
  od;
end;


#############################################################################
##
#F  CoxeterClassParamCheckFunction( <r>, <l> ) . . . . . . . 
#F  CoxeterCosetClassParamCheckFunction( <r>, <l> ) . . . . returns function
#F  which computes a label for the (F-)conjugacy class of a given element 
##  
##  This function  can be used as argument  for 'ClassInvariants' for Coxeter
##  groups or Coxeter cosets, respectively.   They return a function which is
##  always  able to distinguish all   classes (and this quite  effectively
##  compared to backtrack conjugacy tests).
##  
##  It essentially used 'CoxeterClassParamIrred', hence the classification.
##  
CoxeterClassParamCheckFunction:=function(r,l)
  return function(x) local cw, typ; 
            cw:=ComponentWordsPerm(r.g,x);
            typ:=r.g.type;
            return List([1..Length(typ)],i->CoxeterClassParamIrred(typ[i][1],
                           Length(typ[i][2]),cw[i]));
          end;
end;
       
CoxeterCosetClassParamCheckFunction:=function(r,l)
  return function(x) local i, j, h, cw, typ; 
            cw:=ComponentWordsPermCoset(r.g.coset,x);
            
            # change to other representative in same F-conjugacy class,
            # which is of form (x,1,...,1) on F orbits of simple
            # components:
            for i in [1..Length(cw)] do
              h:=[];
              for j in [2..Length(cw[i])] do
                h:=Concatenation(h,Reversed(cw[i][j]));
              od;
              cw[i]:=Concatenation(cw[i][1],Reversed(h));
            od;

            typ:=r.g.coset.type;
            return List([1..Length(typ)],i->
                        CoxeterClassParamIrred(typ[i].orbit[1][1],
                           Length(typ[i].orbit[1][2]),cw[i]));
         end;
end;
       

#############################################################################
##
#F  CoxeterClassParamIrred( <typ>, <rank>[, <o>], <w> ) . . . calling function
#F  for the 'ClassParamX' functions
##  
CoxeterClassParamIrred:=function(arg)
  local r,t, l, o, w;
  
  t:=arg[1];
  l:=arg[2];
  if IsInt(arg[3]) then
    o:=arg[3];
    w:=arg[4];
  else
    o:=1;
    w:=arg[3];
  fi;
  r:=CHEVIE.LoadCoset(t,l,o);
  
  if t in ["A","~A","B","C","D"] and o<>3 then return r.ClassParam(l,w);
  elif t in ["D","E","F","G"] then return r.ClassParam(w);
  # nothing special for types H and I, so long:
  elif t="H" or t="I" then
    return 0;
  else
    Error("no valid arguments: type, rank[, order], word\n");
  fi;
end;

