#############################################################################
##
#A  weylb.g                CHEVIE library                    G\"otz Pfeiffer.
##
#A  $Id: weylb.g,v 1.1 1997/01/21 13:46:48 gap Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##
##  This file contains special functions for Coxeter groups and Iwahori-Hecke 
##  algebras  of type  B.
##

CHEVIE.B:=rec();

CHEVIE.B.PositionId:=function(n) return Length(PartitionTuples(n,2))-1; end;

CHEVIE.B.PositionSgn:=function(n) return n+1; end;

CHEVIE.B.PositionRefl:=function(n) return Length(PartitionTuples(n,2))-3; end;

#############################################################################
##
#F  CHEVIE.B.ClassInfo( <n> ) . . . . . . . . conjugacy classes for type B
##
##  'CHEVIE.B.ClassInfo' returns a record with three components:
##    classtext:   representatives of minimal length in  the  conjugacy  
##                 classes, as words in generators in standard order
##    classparams:  double partitions, parameterizing the classes
##    classnames:  strings for double partitions
##  
##  The  ordering  corresponds  to the  order of  the columns of the ordinary
##  character table of the Coxeter group of type $B_n$, as returned by the GAP
##  function 'CharTable("WeylB", <n>)'.
##
CHEVIE.B.ClassInfo:= function(n)

   local w, i, l, pi, res;

   res:= rec(classtext:=[], classnames:=[], classparams:=[]);

   #  loop over the labels.
   for pi in PartitionTuples(n, 2) do
      w:= [];  i:= 1;

      #  handle signed cycles in reversed order.
      for l in Reversed(pi[2]) do
         Append(w, [i, i-1 .. 2]);
         Append(w, [1 .. i+l-1]);
         i:= i+l;
      od;

      #  the unsigned cycles.
      for l in pi[1] do
         Append(w, [i+1 .. i+l-1]);
         i:= i+l;
      od;

      Add(res.classtext, w);
      Add(res.classparams, pi);
      Add(res.classnames, DoublePartitionToString(pi));
   od;

   #  return the list.
   return res;

end;

#how to make charname from charparam
CHEVIE.B.CharName:=DoublePartitionToString;

#############################################################################
##
#F  CHEVIE.B.CharParams( <n> ) . . . . . . . . . . . . . . . characters for type B
##  
CHEVIE.B.CharParams:=n->PartitionTuples(n,2);

#############################################################################
##
#V  ClassParamB( <W>, <w> )  . . . . . . . .  class parameter of w in type B.
##
##  given an element w of a Coxeter group W of type B, returns the classparam 
##  of its conjugacy class (a double partition).
##
ClassParamBold:= function(W, w)

   local gens, n, i, d, v, left, right, c;

   #  construct signed cycle representation of W.
   n:= W.semisimpleRank;
   gens:= [(n,n+1)];
   for i in [n, n-1 .. 2] do 
      Add(gens, (i,i-1)(2*n+2-i,2*n+1-i));
   od;

   #  compute orbits of w.
   v:= ();
   for i in w do 
      v:= v*gens[i];
   od;

   d:= List([1..2*n], x-> [x, 2*n+1-x]);
   v:= Cycles(v, d, OnPairs);

   left:= [];  right:= [];
   for c in v do
      if Length(Set(List(c, Set))) = Length(c) then 
         Add(left, Length(c));
      else
         Add(right, Length(c)/2);
      fi;
   od;

   v:= Collected(left);
   left:= [];
   for c in v do
      for i in [1..c[2]/2] do 
         Add(left, c[1]);
      od;
   od;
   
   Sort(left);Sort(right);
   return [Reversed(left), Reversed(right)];

end;  

# faster version of 'ClassParamB':
# parameter changed to n = semisimple rank instead of W
CHEVIE.B.ClassParam:=function(n,w)
  local x, i, res, mark, cyc, j;
  
  x:=();
  for i in w do
    if i=1 then x:=x*(1,n+1); else x:=x*(i-1,i)(i-1+n,i+n);fi;
  od;
  
  res:=[[],[]];
  mark:=[1..n];
  for i in [1..n] do
    if mark[i]<>0 then
      cyc:=CyclePermInt(x,i);
      if i+n in cyc then
        Add(res[2],Length(cyc)/2); 
      else
        Add(res[1],Length(cyc));
      fi;  
      for j in cyc do
        if j>n then
          mark[j-n]:=0;
        else
          mark[j]:=0;
        fi;
      od;
    fi;
  od;
  
  Sort(res[1]);
  Sort(res[2]);
  return [Reversed(res[1]),Reversed(res[2])];
end;


CHEVIE.B.CharTable:=function(rank)local tbl,cl,f;
  tbl:=CharTable("WeylB",rank);
  if not IsBound(tbl.size) then
     tbl.size:=2^rank*Factorial(rank);
  fi;    
  tbl.identifier:=String(Concatenation("W(B",String(rank),")"));
  tbl.cartan:=CartanMat("B",rank);
  cl:=CHEVIE.B.ClassInfo(rank);
  for f in RecFields(cl) do tbl.(f):=cl.(f);od;
  tbl.irredinfo:=List(CHEVIE.B.CharParams(rank),x->rec(charparam:=x,
      charname:=CHEVIE.B.CharName(x)));
  return tbl;
end;

#############################################################################
##
#V  CHEVIE.B.HeckeCharTable( <n>, <para> ). . .  character table of $H(B_n)$.
##
##  'CHEVIE.B.HeckeCharTable'   returns the character  table of the Hecke  
##  algebra  $H(B_n)$ with parameters <para>.
##
CHEVIE.B.Hk:= Copy(CharTableWeylB);

CHEVIE.B.Hk.identifier:= "HeckeB";  

CHEVIE.B.Hk.specializedname:= 
   (nq -> Concatenation("H(B", String(nq[1]), ")"));

CHEVIE.B.Hk.order:= (nq-> 2^nq[1]*Factorial(nq[1]));  
CHEVIE.B.Hk.size:= CHEVIE.B.Hk.order;

CHEVIE.B.Hk.domain:= function(nq)
   return IsList(nq) and Length(nq) = 3 and IsInt(nq[1]) and nq[1] > 0;
end;

CHEVIE.B.Hk.text:= "generic character table of Hecke algebras of type B";

CHEVIE.B.Hk.classparam:= [nq -> PartitionTuples(nq[1], 2)];
CHEVIE.B.Hk.charparam:= CHEVIE.B.Hk.classparam;

CHEVIE.B.Hk.irreducibles[1][1]:= function(nq, gamma, pi)

   local n, q, Q, k, val, t, nn, alpha, dif;

   #  for the sake of clearness.
   n:= nq[1];  q:= nq[3];  Q:= nq[2];

   #  termination condition.
   if n = 0 then
      return q^0;
   fi;

   #  initialize character value.
   val:= 0*q;

   #  positive cycles first.
   if pi[1] <> [] then 

      #  get length of the longest cycle.
      k:= pi[1][1];

      #  loop over double paritions of n-k.
      for alpha in PartitionTuples(n-k, 2) do
         dif:= [];
         dif[1]:= DifferencePartitions(gamma[1], alpha[1]);
         dif[2]:= DifferencePartitions(gamma[2], alpha[2]);
         if dif[1] <> false and dif[2] <> false then
            dif:= rec(cc:= dif[1].cc + dif[2].cc, ll:= dif[1].ll + dif[2].ll);
            val:= val + (q-1)^(dif.cc-1) * (-1)^dif.ll * q^(k-dif.ll-dif.cc)
                * CHEVIE.B.Hk.irreducibles[1][1]([n-k, Q, q], alpha, 
                                    [pi[1]{[2..Length(pi[1])]}, pi[2]]);
         fi;
      od;

   else # pi[2] = []

      #  get length of the longest cycle.
      k:= pi[2][1];

      #  loop over k-hooks in gamma[1].
      nn:= Sum(gamma[1]);
      if nn >= k then
         for alpha in Partitions(nn - k) do
            dif:= DifferencePartitions(gamma[1], alpha);
            if dif <> false and dif.cc = 1 then
               val:= val + Q * (-1)^dif.ll * q^(n+dif.d)
                 * CHEVIE.B.Hk.irreducibles[1][1]([n-k, Q, q], [alpha, gamma[2]],
                                    [pi[1], pi[2]{[2..Length(pi[2])]}]);
            fi;
         od;
      fi;

      #  loop over hooks in gamma[2].
      nn:= Sum(gamma[2]);
      if nn >= k then
         for alpha in Partitions(nn - k) do
            dif:= DifferencePartitions(gamma[2], alpha);
            if dif <> false and dif.cc = 1 then
               val:= val + (-1)^(dif.ll+1) * q^(n+dif.d)
                 * CHEVIE.B.Hk.irreducibles[1][1]([n-k, Q, q], [gamma[1], alpha], 
                                    [pi[1], pi[2]{[2..Length(pi[2])]}]);
            fi;
         od;
      fi;

   fi;

   #  return the result.
   return val;

end;

CHEVIE.B.Hk.matrix:= function(nq)

   local scheme, beta, pm, i, m, k, t, n, x, y, np, col, res, charCol, hooks;

   n:= nq[1]; x:= nq[3]; y:= nq[2];

   pm:= [];
   scheme:= [];

   #  how to encode all hooks.
   hooks:= function(beta, m)
      local i, j, k, hk, pr, cbs, prs, leg, hks, lb, ll, lg, lh, gamma, new;

      hks:= List([1..m], x->[]);
      prs:= [];
      lb:= [Length(beta[1]), Length(beta[2])];

      #  find all hooks.
      for i in [1, 2] do
         prs[i]:= [];
         for j in beta[i] do
            leg:= 0;
            for k in Reversed([0..j-1]) do
               if  k in beta[i] then
                   leg:= leg + 1;
               else
                   Add(prs[i], 
                        rec(from:= j, to:= k, leg:= leg, pow:= m+k-lb[i]));
               fi;
            od;
         od;
      od;

      #  construct combinations.
      cbs:= List(prs[1], x-> [[x], []]);
      Append(cbs, List(prs[2], x-> [[], [x]]));
      for hk in cbs do

         #  extend.
         for pr in prs[1] do
            if hk[2] = [] and pr.to > hk[1][Length(hk[1])].from then
               new:= Copy(hk);
               Add(new[1], pr);
               Add(cbs, new);
            fi;
         od;
         for pr in prs[2] do
            if hk[2] = [] or pr.to > hk[2][Length(hk[2])].from then
               new:= Copy(hk);
               Add(new[2], pr);
               Add(cbs, new);
            fi;
         od;


         #  encode.
         ll:= Sum(hk[1], x-> x.from - x.to) + Sum(hk[2], x-> x.from - x.to);
         lg:= Sum(hk[1], x-> x.leg) + Sum(hk[2], x-> x.leg);
         lh:= Length(hk[1]) + Length(hk[2]);
         new:= rec(wgt:= [(-1)^lg * x^(ll-lg-lh) * (x-1)^(lh-1), 0],
                   adr:= 1);
         if lh = 1 then
            if IsBound(hk[1][1]) then
               new.wgt[2]:= (-1)^lg * y * x^hk[1][1].pow;
            else
               new.wgt[2]:= (-1)^(lg+1) * x^hk[2][1].pow;
            fi;
         fi;

         #  recalculate address.
         if ll < m then 
            gamma:= [];
            for i in [1, 2] do
                gamma[i]:= Difference(beta[i], List(hk[i], x-> x.from));
                UniteSet(gamma[i], List(hk[i], x-> x.to));
                if 0 in gamma[i] then
                    j:= 0;
                    while j < Length(gamma[i]) and gamma[i][j+1] = j do
                       j:= j+1;
                    od;
                    gamma[i]:= gamma[i]{[j+1..Length(gamma[i])]}-j;
                fi;
            od;
            new.adr:= Position(pm[m-ll], gamma);
         fi;

         #  insert.
         Add(hks[ll], new);
      od;
      return hks;
   end;

   #  collect hook encodings.
   InfoCharTable2("#I  Scheme: \c");
   for i in [1..n] do
      InfoCharTable2(i, " \c");
      pm[i]:= List(PartitionTuples(i, 2), p-> List(p, BetaSet));
      scheme[i]:= [];
      for beta in pm[i] do
         Add(scheme[i], hooks(beta, i));
      od;
   od;
   InfoCharTable2("done.\n");

   #  how to construct a new column.
   charCol:= function(n, t, k, p)
      local col, pi, hk, val;
      col:= [];
      for pi in scheme[n] do
         val:= 0*y;
         for hk in pi[k] do
            val:= val + hk.wgt[p] * t[hk.adr];
         od;
         Add(col, val);
      od;
      return col;
   end;

   #  construct the columns.
   InfoCharTable2("#I  Cycles: \c");
   pm:= List([1..n], x->[]);

   #  second position.
   for m in [1..n] do 

      #  add the m-cycle.
      InfoCharTable2(m, " \c");
      Add(pm[m], charCol(m, [1], m, 2));
      for k in [m+1..n] do
         for t in pm[k-m] do
            Add(pm[k], charCol(k, t, m, 2));
         od;
      od;
   od;

   #  first position.
   for m in [1..QuoInt(n,2)] do 

      #  add the m-cycle.
      InfoCharTable2(m, " \c");
      Add(pm[m], charCol(m, [1], m, 1));
      for k in [m+1..n-m] do
         for t in pm[k-m] do
            Add(pm[k], charCol(k, t, m, 1));
         od;
      od;
   od;
   InfoCharTable2("done.\n");

   #  collect.
   InfoCharTable2("#I  Tables: \c");
   res:= [];
   for k in [1..n-1] do
      InfoCharTable2(k, " \c");
      for t in pm[n-k] do
         Add(res, charCol(n, t, k, 1));
      od;
   od;
   Add(res, charCol(n, [1], n, 1));
   Append(res, pm[n]);
   InfoCharTable2("done.\n");

   res:= Permuted(res, 
            Sortex(DoublePartitions(n))/Sortex(PartitionTuples(n, 2)));

   return TransposedMat(res);

end;

CHEVIE.B.HeckeCharTable:= function(n, q)local tbl;

   if not IsList(q) then
      q:= [q, q];
   fi;
   tbl:= CharTableSpecialized(CHEVIE.B.Hk, [n, q[1], q[2]]);
   tbl.cartan:= CartanMat("B", n);
   tbl.parameter:= List([1..n], x-> q[2]);
   tbl.parameter[1]:= q[1];
   tbl.classtext:= CHEVIE.B.ClassInfo(n).classtext;
   tbl.classparam:= List(tbl.classparam, x-> x[2]);
   tbl.classnames:= List(tbl.classparam, DoublePartitionToString);
   tbl.irredinfo:=List(CHEVIE.B.CharParams(n),x->
                       rec(charparam:=x,charname:=CHEVIE.B.CharName(x)));
   return tbl;
end;

#############################################################################
##
#F  CHEVIE.B.PoincarePolynomial( <n>, <para> )  Poincare polynomial for type B.
##
##  'CHEVIE.B.PoincarePolynomial' returns the Poincare polynomial of the 
##  Coxeter group $W$ of type $B_n$, ie.  the sum of $q^l(w)$ over all 
##  elements $w$ of $W$.
##
CHEVIE.B.PoincarePolynomial:= function(n, q)

   if not IsList(q) then
      q:= [q, q];
   fi;

   return Product([0..n-1], i-> (q[2]^i*q[1] + 1) * Sum([0..i], k-> q[2]^k));

end;


#############################################################################
##
#F  CHEVIE.B.SchurElement( <n> , <para> )  . . . . Schur elements for type B.
##
##  'CHEVIE.B.SchurElement' returns the list of Schur  elements for the character
##  table of  the Hecke algebra  of type $B_n$  with parameters <para>.  Here
##  <para can be a single parameter (in which case all parameters are assumed
##  to be equal  to this one),  or a pair of   parameters (in which  case the
##  first parameter is associated  to the first generator,  and the second to
##  all the remaining generators).  It is also possible  to pass the complete
##  list of parameters as an argument.
##
##  [Reference: Carter II, 13.5, p. 447.]
##
CHEVIE.B.SchurElement:= function(pi, q)

   local uuu, la, mu, res, m, i, j, k, u, v;

   # name indeterminates.
   if not IsList(q) then
      q:= [q, q];
   fi;
   u:= q[2];  v:= q[1];

   # quantum integers.
   uuu:= k-> Sum([0..k-1], e-> u^e);

   # transform charparam pi (double partition) into a symbol [la, mu].
   la:=SymbolDpart(pi,1);mu:=la[2];la:=la[1];
   m:= Length(mu);

   # initialize Schur element.
   res:= u^((2*m+1)*m*(m-2)/3) * v^(m*(m-1)/2) * (u+v)^m;

   # loop over lambda.
   for i in la do
      for j in [0..i-1] do
         if j in la then
            if j in mu then
               res:= res * u^(-2*j);
            else
               res:= res * (u^(i-2*j-1)*v + u^-j);
            fi;
         else
            if j in mu then
               res:= res * uuu(i-j) * u^-j;
            else
               res:= res * uuu(i-j) * (u^(i-j-1)*v + 1);
            fi;
         fi;
      od;
   od;

   # loop over mu.
   for i in mu do
      for j in [0..i-1] do
         if j in mu then
            if j in la then
               res:= res * u^(-2*j+1) * v^-1;
            else
               res:= res * (u^(i-2*j+1) * v^-1 + u^-j);
            fi;
         else
            if j in la then
               res:= res * uuu(i-j) * u^(1-j) * v^-1;
            else
               res:= res * uuu(i-j) * (u^(i-j+1) * v^-1 + 1);
            fi;
         fi;
      od;
      if i in la then
         res:= res/(u^(i-1)*v + u^i);
      fi;
   od;

   #  return the result.
   return res;

end;

#############################################################################
##
#F  CHEVIE.B.vcycFakeDegree( <c> ) Fake Degree of char. with charparam <c>
##
##  This returns the polynomial describing the multiplicity of the character
##  with charaparam <c> in the graded version of the regular representation 
##  given by the quotient S/I where S is the symmetric algebra of the
##  reflection representation and I is the ideal generated by the homogenous
##  invariants of positive degree in S.
##
CHEVIE.B.vcycFakeDegree:=c->vcycFakeDegreeSymbol(SymbolDpart(c,1));

CHEVIE.B.Chara:=function(p)
  local pp, m, i, j, res;
  
  pp:=SymbolDpart(p,1);
  m:=List(pp,Length);
  res:=pp[1]*[m[1]-1,m[1]-2..0];
  if pp[2]<>[] then
    res:=res+pp[2]*[m[2]-1,m[2]-2..0];
  fi;
  for i in pp[1] do
    for j in pp[2] do
      res:=res+Minimum(i,j);
    od;
  od;
  m:=Sum(m)-2;
  while m>1 do
    res:=res-m*(m-1)/2;
    m:=m-2;
  od;
  return res;
end;

CHEVIE.B.CharA:=function(p)
  local pp, res, l, m, i, j;
  pp:=SymbolDpart(p,1);
  m:=List(pp,Length);
  l:=Sum(pp[1])+Sum(pp[2])-QuoInt((Sum(m)-1)^2,4);
  pp:=List(pp,Reversed);
  res:=l*(l+1)+pp[1]*[m[1]-1,m[1]-2..0];
  if pp[2]<>[] then
    res:=res+pp[2]*[m[2]-1,m[2]-2..0];
  fi;
  for i in pp[1] do
    for j in pp[2] do
      res:=res+Maximum(i,j);
    od;
  od;
  m:=Sum(m)-2;
  while m>1 do
    res:=res-m*(m-1)/2;
    m:=m-2;
  od;
  for i in Concatenation(pp) do
    res:=res-i*(i+1);
  od;
  return res;
end;
  
CHEVIE.B.Charb:=function(p) local pp, m, i, j, res;
  
  pp:=SymbolDpart(p,1);
  m:=Length(pp[2]);
  if Length(pp[1])<>m+1 then
    return -1;
  fi;
  res:=pp[1]*[m,m-1..0];
  if pp[2]<>[] then
    res:=res+pp[2]*[m-1,m-2..0];
  fi;
  res:=2*res+Sum(pp[2]);
  return res-m*(m-1)*(4*m+1)/6;
end;

CHEVIE.B.CharB:=p->CHEVIE.B.Chara(p)+CHEVIE.B.CharA(p)-CHEVIE.B.Charb(p);

#############################################################################
##
#E  Emacs . . . . . . . . . . . . . . . . . . . . . .  local emacs variables.
##
##  Local Variables:
##  fill-column: 77
##  fill-prefix: "##  "
##  End:
##
