###########################################################################
##
#A  orbit.g                  autag package                 Michael J Smith
##
##  November 1996
##
##  This file forms part of a package for computing automorphism groups of
##  finite soluble groups which are given in terms of special soluble group
##  presentations.
##
###########################################################################


# 12:30 Sun 28 Jul 1996 - cleaned up the default function and improved 
# comments in the code.


OrbitOps.OrbitStabiliser := function (gens, identity, point0, fnRec)
   #
   # This orbit-stabiliser routine will compute Schreier generators for the
   # stabiliser of <point0> in the group <G> generated by <gens>. The
   # generating set <gens> must be a weighted generating set; each <g> in
   # <gens> has a weight giving its position in some normal series for <G>.
   # Weight 1 should correspond to the whole group <G>, and increasing
   # weight to subgroups of <G>.  The generators for the stabiliser that
   # are returned will respect the normal series: that is, each generator
   # in the stabiliser generating set will have a weight assigned to it,
   # and the generators with weight >= some particular <w> will generate
   # the stabiliser in the weight <w> subgroup of <G>.
   
   # Here are the arguments of the function:
   #
   #     <gens> = a set of generators for a group acting on some set.
   #
   #     <identity> = the identity element of the group
   #
   #     <point0> = the point of the set that we want the stabiliser of
   #
   #     <fnRec> = a record overlaying (if necessary) the default functions
   #     needed to do operations on group elements and points etc.

   # There are defaults for each component of fnRec, except for
   # <equivalent> and <weight>, which are most likely to be context
   # dependant.
   #
   # The entries of fnRec are:
   #
   #     weight(g) = the weight of group element <g>, giving it position in
   #     some normal series for the acting group. (No default function)
   #
   #     equivalent(stabiliser, g1, g2, x1, x2, weight) = check whether the
   #     group elements <g1> and <g2> lie in the same coset of the stabiliser,
   #     or equivalently whether the two points <x1> and <x2> are the same.
   #     Must return false if not equivalent. If equivalent, this function
   #     must append the Schreier element (if needed) to the <stabiliser>
   #     set, and ensure that the new element is assigned the given
   #     <weight> --- why? Because in some contexts the information needed
   #     to establish equivalence might be useful, and we allow the user
   #     some flexibility this way.  (No default function)
   #
   #     product(g1, g2) = product of group elements <g1> and <g2>.
   #     (Default=g1*g2)
   #
   #     inverse(g) = inverse of group element <g>. (Default=g^-1)
   #
   #     action(x, g) = point obtained by acting on point <x> with group
   #     element <g>.  (Default=x^g)
   #
   #     appendTran(transversal, newelem, orbitreps, newpoint, weight) =
   #     append <newelem> to the <transversal>, and <newpoint> to the list of
   #     <orbitreps>.  The current <weight> is available. (Default simply
   #     appends without using weight).

   local ops, funcname, weights, weight, w, transversal, orbitreps, 
         stabiliser, tweight, settled, wgens, oldlen, i, elem, point, j, g, 
         newelem, newpoint, x, equiv;

   # set the functions: start with the defaults
   ops := ShallowCopy(OrbitOps.default); 
   
   # now override default functions with those given by user
   for funcname in RecFields(fnRec) do
      ops.(funcname) := fnRec.(funcname); 
   od;


   # now collect weight information from generators
   weights := List(gens, g -> ops.weight(g));

   # start at maximum weight for the loop
   weight := Maximum(weights);

   # for each weight, find a list of generator indices for that weight
   w := List([1..weight], i -> Positions(weights, j -> j=i));

   transversal := [ identity ];
   orbitreps := [ point0 ];
   stabiliser := [];

   # accumulate weights at which transversal elements were added.
   # this will be useful for computing the change in the sizes of
   # the factors of the normal series.
   tweight := [ops.weight(identity)]; 

   # number of elements in the transversal that do not need to be acted
   # upon by current weight generators
   settled := 0;

   InfoOrbit("#O   starting weight ", weight,"\n");

   # get the generators for the current weight
   wgens := gens{w[weight]};

   repeat
      #
      # we will repeat until closed under current weight gens
      
      oldlen := Length(transversal);

      for i in [settled+1..Length(transversal)] do

         InfoOrbit("#O   looking at transversal element ",i,"\n");

         elem := transversal[i];
         point := orbitreps[i];

         for j in [1..Length(wgens)] do

            g := wgens[j];
            
            InfoOrbit2("#O   Examining transversal[",i,"] * ");
            InfoOrbit2("wgen[",j,"]\n");

            newelem := ops.product(elem, g);
            newpoint := ops.action(point, g);

            x := 0;

            repeat

               x := x + 1;

               # check whether newelem is in same coset as transversal[x];
               InfoOrbit2("#O   Checking to see if newelem^-1*transversal[");
               InfoOrbit2(x,"] is in subgroup\n");

               equiv := ops.equivalent(stabiliser, newelem, transversal[x],
                                newpoint, orbitreps[x], weight);

               if equiv <> false then
                  # NB: the function equivalent has the duty of appending
                  # the generator to the stabiliser. Therefore nothing
                  # needs to be done here.
                  InfoOrbit2("#O   Added to generators of subgroup");
                  InfoOrbit2(" (element ",Length(stabiliser),")\n");
               fi;

            until x = Length(transversal) or equiv <> false;

            if equiv = false then
               # newelem was not in previously obtained cosets
               ops.appendTran(transversal, newelem, orbitreps, 
                       newpoint, weight);
               Add(tweight, weight);
               InfoOrbit("#O   Added to transversal of subgroup");
               InfoOrbit(" (element ",Length(transversal),")\n");
            fi;
         od;
      od;
      settled := oldlen;

      if Length(transversal) = oldlen then
         repeat
            weight := weight - 1;
         until weight=0 or Length(w[weight]) > 0;
         settled := 0;
         if weight > 0 then
            wgens := gens{w[weight]}; 
            InfoOrbit("#O   starting weight ", weight,"\n");
         fi;
      fi;

   until Length(transversal) = oldlen and weight = 0;

   InfoOrbit("#O   tweight = ", tweight, "\n");

   return rec(transversal := transversal, 
              orbit := orbitreps, 
              stabiliser := stabiliser,
              tweight := tweight
              );
end;


# Now define default functions for the orbit-stabiliser routine
#
OrbitOps.default := rec();

OrbitOps.default.product := function (x, y)
   return x*y;
end;

OrbitOps.default.inverse := function (x)
   return x^-1;
end;

OrbitOps.default.action := function (x, g)
   return x^g;
end;

OrbitOps.default.appendTran := function (arg)
   # arguments: tran, t, orbreps, o
   local tran, t, orbreps, o;
   tran := arg[1]; t := arg[2]; orbreps := arg[3]; o := arg[4];
   Add(tran, t);
   Add(orbreps, o);
end;

OrbitOps.default.weight := function (g)
   return g.weight;
end;



## Local Variables:
## eval: (make-local-variable 'gap-indent-step)
## eval: (make-local-variable 'gap-indent-step-continued)
## eval: (setq gap-indent-step 3)
## eval: (setq gap-indent-step-continued 2)
## End:
