#############################################################################
##
#A  braid.g                  CHEVIE library                      Jean Michel.
##
#A  $Id: braid.g,v 1.1 1997/01/21 13:46:20 gap Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##
##  This file contains functions to work with elements of the Artin braid
##  group associated to a Coxeter group. Elements are represented as a 
##  record  which has an operation record which currently understands the 
##  operations Print, String, *, / and ^. 
##
#  Maybe the best illustration is the session log below
#  
#  gap> w:=Braid(CoxeterGroup("A",4))(1,2,3,4,3);
#  12343
#  gap> w*w;
#  12132432.43
#  gap> w^4;
#  w0^2.
#
#  Elements are stored as w0^i w_1.w_2...w_n where i is positive or
#  negative and w_1.w_2...w_n  is the left-greedy canonical
#  decomposition of an element of the braid monoid: w_1 is the longest
#  reduced word which is a left prefix, then the rest of the element
#  is recursively decomposed in the same way. This is called "Deligne normal
#  form", and is what is printed by default.
#
#  Elements are represented as records with fields:
#
#   pw0   the power of w0 (positive or negative)
#   elm   the canonical left-greedy decomposition, given as a list of
#         elements of W (permutations)
#   braidOps  see below
#
#
#############################################################################

braidOps:=OperationsRecord("BraidOps");

#F String(b)
## Convert the braid element to a string, according to the value of
## CHEVIE.PrintBraid. The possibilities for CHEVIE.PrintBraid are
##  "Deligne" --- Deligne normal form
##  "Charney" --- Charney normal form
##  "GAP"     --- print into a form which can be input back in GAP.

braidOps.String:=function(b)local W,v,b,res,l,left,neg;
  W:=CoxeterGroup(b);
  if CHEVIE.PrintBraid="Deligne" then
    if b.pw0 <>0 then res:=Concatenation("w0^",String(b.pw0),".");
    else res:="";
    fi;
    for v in List(b.elm,x->CoxeterWord(W,x)) do
       Append(res,IntListToString(v,"[]"));Add(res,'.');
    od;
    if Length(res)>0 then res:=res{[1..Length(res)-1]};fi;
  elif CHEVIE.PrintBraid="Charney" then
    if b.pw0<0 then 
     neg:=Braid(W)((),-b.pw0);
     b:=neg*b;
     left:=function(b)
       if b.pw0>0 then return W.rootInclusion{[1..W.semisimpleRank]};
       elif Length(b.elm)=0 then return [];
       else return LeftDescentSet(W,b.elm[1]);
       fi;
     end;
     v:=Intersection(left(b),left(neg)); 
     while Length(v)>0 do 
      v:=Braid(W)(LongestCoxeterElement(ReflectionSubgroup(W,v)))^-1;
      b:=v*b;neg:=v*neg;
      v:=Intersection(left(b),left(neg)); 
     od;
     res:=Concatenation("(",String(neg),")^-1.");
    else res:="";
    fi;
    CHEVIE.PrintBraid:="Deligne";Append(res,String(b));
    CHEVIE.PrintBraid:="Charney";
  elif CHEVIE.PrintBraid="GAP" then
    if b.pw0<0 then 
      res:=Concatenation("B(",String(WordBraid(Braid(W)((),-b.pw0)*b)),
                         ",",String(b.pw0),")");
    else res:=Concatenation("B(",String(WordBraid(b)),")");
    fi;
  else Error("CHEVIE.PrintBraid must be one of: Charney, Deligne, GAP");
  fi;
  return String(res);
end;

#F Print(b) print b (using String)

braidOps.Print:=function(b)Print(String(b));end;

braidOps.\*:=function(a,b)local W,x,c,res,nres,v,conj,r,i;
  W:=CoxeterGroup(a);
  if W<>CoxeterGroup(b) then Error("not elements of the same braid group");fi;
  res:=rec(pw0:=a.pw0+b.pw0,elm:=b.elm,operations:=braidOps,coxeter:=W);
  conj:=LongestCoxeterElement(W)^(b.pw0 mod 2);
  for x in Reversed(a.elm) do
   x:=x^conj;
   nres:=[];
   while Length(res.elm)>0 do
     v:=res.elm[1];
#  Here is the core of the multiplication algorithm.
#  Replace x,v by  x', v' such that x' * v' = x * v and where
#      x' = x * (longest subword z of v  such that l(xz)=l(x)+l(z))
#
#  The algorithm uses the fact that any reduced left divisor of x.v in
#  the braid group is a left divisor of x'
    i:=1;
    repeat
      if W.rootInclusion[i]^v>W.parentN and 
	 W.rootInclusion[i]^(x^-1)<=W.parentN
      then r:=W.generators[i];x:=x*r;v:=r*v;i:=1;
      else i:=i+1;
      fi;
    until i>W.semisimpleRank;
    Add(nres,x);
    x:=v;
    if x=res.elm[1] then Append(nres,res.elm);res.elm:=[];x:=();
    else res.elm:=res.elm{[2..Length(res.elm)]};
    fi;
   od;
   if x<>() then Add(nres,x);fi;
   res.elm:=nres;
  od;
  while Length(res.elm)>0 and res.elm[1]=LongestCoxeterElement(W) do
    res.pw0:=res.pw0+1;
    res.elm:=res.elm{[2..Length(res.elm)]};
  od;
  return res;
end;

braidOps.\/:=function(a,b)return a*b^-1;end;

braidOps.\^:=function(b,n)local W,p,res,i;
  W:=CoxeterGroup(b);
  if IsInt(n) then
  if n<0 then
    b:=ShallowCopy(b);
    b.elm:=List([1..Length(b.elm)] ,i->
                (b.elm[i]^-1)^(LongestCoxeterElement(W)^((i+1+b.pw0) mod 2)));
    b:=rec(pw0:=-b.pw0-Length(b.elm),
	 elm:=LongestCoxeterElement(W)*Reversed(b.elm),
	 operations:=braidOps,coxeter:=W);
    n:=-n;
  fi;
  p:=b;res:=Braid(W)([]);
  while n>0 do
   if n mod 2 <> 0 then res:=res*p;fi;
   p:=p*p;
   n:=QuoInt(n,2);
  od;
  return res;
  else return n^-1*b*n;
  fi;
end;

braidOps.Frobenius:=function(W,x)
  x:=ShallowCopy(x);x.elm:=List(x.elm,y->Frobenius(W)(y));
  return x;
  end;

Braid:=function(W)
#############################################################################
##
##  . . . . . . . . . . . . . . . . . . . . .  braid-element making function
#F  function(perm [, pw0])  
#F  or function([s1,...,sn],[, pw0])  
#F  or function(s1,...sn)  
##
##  returns the element of the Artin braid monoid corresponding to element
##  perm of the Coxeter group W, or the sequence of integers s1,...,sn
##  representing a (non necessarily reduced) word in the generators of the
##  Coxeter group W.
##  If pw0 (a positive or negative integer) is given, the resulting element
##  is multiplied by w0^pw0. 

  return function(arg)local res,s;
    res:=rec(pw0:=0,elm:=[],operations:=braidOps,coxeter:=W);
    if Length(arg)<>0 then
      if IsPerm(arg[1]) then
	if arg[1]=LongestCoxeterElement(W) then res.pw0:=1;
	elif arg[1]<>() then res.elm:=[arg[1]];
	fi;
	if Length(arg)>1 then res.pw0:=res.pw0+arg[2]; fi;
      else
	if IsList(arg[1]) then 
	   if Length(arg)=2 then res.pw0:=arg[2];fi;
	   arg:=arg[1];
	fi;
	for s in Reflections(W){arg} do
	  res:=res*rec(pw0:=0,elm:=[s],operations:=braidOps,coxeter:=W);
	od;
      fi;
    fi;
    return res;
  end;

end;

#############################################################################
##
#F  PermBraid(w)  . . . . . . . . . . . . . . .  Image in the Coxeter group.
##
##  w is a braid. PermBraid(w) returns the image in CoxeterGroup(w) of w.

PermBraid:=function(x)local res;
  res:=();
  if x.pw0 mod 2 <>0 then res:=LongestCoxeterElement(CoxeterGroup(x));fi;
  if Length(x.elm)=0 then return res;
  else return res*Product(x.elm);
  fi;
end;

#############################################################################
##
#F  WordBraid(w) . . . . . . . . . . . . . . . .  sequence in the generators.
##
##  w is an element of the Artin braid monoid.
##  WordBraid(w) returns the expression of w in the generators,
##  given as a sequence of integers (a "non-reduced CoxeterWord").

WordBraid:=function(b)local res,i,W;
  W:=CoxeterGroup(b);
  if b.pw0<0 then Error(b," must be in the braid monoid");fi;
  res:=[];
  for i in [1..b.pw0] do Append(res,LongestCoxeterWord(W));od;
  for i in b.elm do Append(res,CoxeterWord(W,i));od;
  return res;
end;

###########################################################################
##
#F GoodCoxeterWord( <W>, <w> ) . . . . . . . . . . . . check if w is a good 
## . . . . . . . . element in the braid group in the sense of [Geck-Michel]
##
## 'GoodElement' checks if w (a sequence of generators) represents
## a good element in the braid group, i.e., if w^d (where d is the order of
## the image of w in W) is a product of longest elements in a 
## decreasing chain of parabolic subgroups of W.  If this is true, then the
## list of the corresponding subsets of the generators  together with their
## multiplicities in the chain is returned. Otherwise,  false is returned.

GoodCoxeterWord:=function(W,w)local res,v,I,bw,d;
  d:=OrderPerm(PermCoxeterWord (W,w));
  bw:=Braid(W)(w)^d;
  res:=[];
  if bw.pw0>0 then Add(res,[W.rootInclusion{[1..W.semisimpleRank]},bw.pw0]);fi;
  for v in bw.elm do
    I:=Set(CoxeterWord(W,v));
    if LeftDescentSet(W,v)<>I then 
         InfoChevie("#I ",Braid(W)(w)," is not good\n");
         return false; fi;
    if Length(res)>0 and res[Length(res)][1]=I then 
      res[Length(res)][2]:=res[Length(res)][2]+1;
    else	
      Add(res,[I,1]);
    fi;
  od;
  InfoChevie("#I ",Braid(W)(w),"^",d,"=");
  for I in res do 
    InfoChevie("w_",IntListToString(I[1],"{}"),"^{",I[2],"}.");
  od;
  InfoChevie("\n");
  return res;
end;
