#############################################################################
##
#A  kl.g           CHEVIE library    Meinolf Geck, Jean Michel, Andrew Mathas
##
#A  $Id: kl.g,v 1.1 1997/01/21 13:46:32 gap Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##
##  This file contains  GAP functions to  compute Kazhdan - Lusztig
##  polynomials and left cells, and  for  working with KL bases of
##  Hecke algebras. 
##

#############################################################################
if not IsBound(HeckeAlgebraOps) then ReadChv("prg/hecke");fi;

HeckeAlgebraOps.InitKL:=function(H,msg)local W,i;
  if IsBound(H.genQ) then return;fi;
  W:=CoxeterGroup(H);
  if W.semisimpleRank=0 or 
     not ForAll([1..W.semisimpleRank],i->IsBound(H.sqrtParameter[i])) then
    Error(msg," not defined unless sqrtParameters are all bound");
  elif not ForAll(H.sqrtParameter,x->x=H.sqrtParameter[1]) then
    Error(msg," not implemented unless sqrtParameters are all equal");
  elif ForAny(H.sqrtParameter,x->x=0*x) then
    Error(msg," not implemented unless sqrtParameters are invertible");
  elif ForAny(H.sqrtParameter,IsFFE) then
    Error(msg," not implemented for FFE because DefaultRing doesn't work here");
  else

    ## The KL-basis elements are calculated generically using the Hecke
    ## algebra parameters [genQ^2,...,genQ^2]. If the Hecke algebra is
    ## defined over a polynomial ring then we use this ring, otherwise
    ## we work in Z[genQ,genQ^-1].
    if W.semisimpleRank>0 and IsPolynomial(H.sqrtParameter[1]) then
      H.genQ := H.sqrtParameter[1].baseRing.indeterminate;
    else
      H.genQ := X(DefaultRing(H.sqrtParameter[1]));
    fi;

    ## As the C' basis elements are calculated they are stored in the list
    ## CpBasis; Celts keeps track of which basis elements have already
    ## been calculated.
    H.Celts:=[()]; H.CpBasis:=[H.genQ^0*Basis(H,"T")()];
    for i in [1..W.semisimpleRank] do
      Add(H.Celts,W.generators[i]);
      Add(H.CpBasis,Basis(H,"T")([(),W.generators[i]],[H.genQ^-1,H.genQ^-1]));
    od;
  fi;
end;

##  Alt is The involution of H defined by v -> -v^{-1} and
##  T_w -> (-v^{-2})^{l(w)}T_w.
##  It swaps C_w and C'_w, and D_w and D'_w.
##  Essentially it corresponds to tensoring with the sign representation.

# The function below is made member of HeckeAlgebraOps just to hide it.
# It applies v->-v^-1 to the coefficients of h.
# This is not Alt! One needs also to do some normalisation depending on
# the basis.
HeckeAlgebraOps.Alt := function(arg)local h,q;
  h:=arg[1];
  q:=Hecke(h).genQ;
  h:=ShallowCopy(h);
  if Length(arg)=2 then
   if Hecke(h).sqrtParameter[1]<>q then Error(
     "Alt involution is meaningless if parameter is not an indeterminate");
   fi;
   h.basis:=arg[2];h.operations:=HeckeAlgebraOps.(arg[2]);
  fi;
  h.coeff:=List(h.coeff,x->Value(x*q^0,-q^-1));
  return h;
end;

# The complete Alt for T basis
HeckeAlgebraOps.AltT := function(h)local W,q,i;
  W:=CoxeterGroup(Hecke(h));
  q:=Hecke(h).genQ;
  h:=HeckeAlgebraOps.Alt(h);
  for i in [1..Length(h.coeff)] do
    h.coeff[i]:=(-q^-2)^CoxeterLength(W,h.elm[i])*h.coeff[i];
  od;
  return h;
end;

##  Beta is the involution on H defined by v -> v^-1 and
##  T_w -> v^{-l(w_0)}T_{w_0w}.
## It swaps C'_w and D_{w_0w}, and C_w and D'_{w_0w}.

# The function below is made member of HeckeAlgebraOps just to hide it
# It is not the full Beta!
HeckeAlgebraOps.Beta := function(arg)local h,q;
  h:=arg[1];
  q:=Hecke(h).genQ;
  h:=ShallowCopy(h);
  if Length(arg)=2 then
   if Hecke(h).sqrtParameter[1]<>q then Error(
     "Beta involution is meaningless if parameter is not an indeterminate");
   fi;
   h.basis:=arg[2];h.operations:=HeckeAlgebraOps.(arg[2]);
  fi;
  h.elm:=LongestCoxeterElement(CoxeterGroup(Hecke(h)))*h.elm;
  h.coeff:=List(h.coeff,x->Value(x*q^0,q^-1));
  return h;
end;

# The complete Beta for T basis
HeckeAlgebraOps.BetaT := function(h)
  h:=ShallowCopy(h);
  h:=HeckeAlgebraOps.Beta(h);
  h.coeff:=Hecke(h).genQ^-CoxeterGroup(Hecke(h)).N*h.coeff;
  return h;
end;

# The function below is made member of HeckeAlgebraOps just to hide it
HeckeAlgebraOps.Specialize:=function(h)local H;
  H:=Hecke(h);
  if CoxeterGroup(H).semisimpleRank>0 and H.genQ<>H.sqrtParameter[1] then
    h:=ShallowCopy(h);
    h.coeff := List(h.coeff,i->Value(i,H.sqrtParameter[1]));
  fi;
  return h;
end;

##  This section is copied almost verbatim from code (C) A.Mathas dec.  1994
# The function below is made member of HeckeAlgebraOps just to hide it

HeckeAlgebraOps.getCp := function(H,w)local W,cpw,i,k,l,mu,temp;
  W:=CoxeterGroup(H);cpw := Position(H.Celts, w);
  if cpw<>false then return ShallowCopy(H.CpBasis[cpw]);fi;

  i:=First([1..W.semisimpleRank],j->W.rootInclusion[j]^w>W.parentN);
  cpw:=HeckeEltOps.genprod(H.CpBasis[i+1],
           HeckeAlgebraOps.getCp(H,W.generators[i]*w),
		 List([1..W.semisimpleRank],x->H.genQ^2));
	
  # now we pull off the 'large' degree pieces
  # The expression below is to loop on decreasing length of elements;
  # we use the fact that if v appears in C'(w) then v=w or l(v)<l(w).
  
  for i in OnTuples([Length(cpw.elm)-1,Length(cpw.elm)-2..1],
		    Sortex(List(cpw.elm,i->CoxeterLength(W,i)))^-1) do
    if Degree(cpw.coeff[i])>-1-CoxeterLength(W,cpw.elm[i]) then
      temp := HeckeAlgebraOps.getCp(H,cpw.elm[i]);
      mu:=temp.coeff[Position(temp.elm,cpw.elm[i])].coefficients[1];
      for l in [1..Length(temp.coeff)] do
	k := Position(cpw.elm,temp.elm[l]);  # can not be false
	cpw.coeff[k]:=cpw.coeff[k]-mu*temp.coeff[l]; # non-zero
      od;
    fi;
  od;
  Add(H.CpBasis,cpw);Add(H.Celts,w);
  return ShallowCopy(cpw);
end; 

# The function below is made member of HeckeAlgebraOps just to hide it
HeckeAlgebraOps.ToKL:=function(h,target,index)local i,x,res,index,lens,H;
  H:=Hecke(h);res:=Basis(H,target)([],[]); 

  # To convert from "T", we use the fact that the transition matrix M 
  # from any KL bases to the standard basis is triangular with diagonal 
  # coefficient on T_w equal to v^{-l(w)}. The transition matrix is
  # lower triangular for the C and C' bases, and upper trianguler
  # for the D and D' bases which is what index() is for.

  h:=ShallowCopy(h); 
  while h.elm <> [] do
    lens:=List([1..Length(h.elm)],i->CoxeterLength(CoxeterGroup(H),h.elm[i]));
    x:=index(lens); lens:=Filtered([1..Length(h.elm)],i->lens[i]=x);
    x:=Basis(H,target)(h.elm{lens},H.sqrtParameter[1]^x*h.coeff{lens});
    res:=res+x; h:=h-Basis(H,"T")(x);
  od;
  return res;
end;

CreateHeckeBasis("C",rec(
  init:=function(H)HeckeAlgebraOps.InitKL(H,"C basis");end,
  BetaInvolution:=h->HeckeAlgebraOps.Beta(h,"D'"),
  AltInvolution:=h->HeckeAlgebraOps.Alt(h,"C'"),
  T:=h->h.coeff*List(h.elm,x->HeckeAlgebraOps.Specialize(
    HeckeAlgebraOps.AltT(HeckeAlgebraOps.getCp(Hecke(h),x)))),
  C:=h->HeckeAlgebraOps.ToKL(h,"C",Maximum)
  ));

CreateHeckeBasis("C'",rec(
  init:=function(H)HeckeAlgebraOps.InitKL(H,"C' basis");end,
  BetaInvolution:=h->HeckeAlgebraOps.Beta(h,"D"),
  AltInvolution:=h->HeckeAlgebraOps.Alt(h,"C"),
  T:=h->h.coeff*List(h.elm,x->HeckeAlgebraOps.Specialize(
     HeckeAlgebraOps.getCp(Hecke(h),x))),
  ("C'"):=h->HeckeAlgebraOps.ToKL(h,"C'",Maximum)
  ));

CreateHeckeBasis("D",rec(
  init:=function(H)HeckeAlgebraOps.InitKL(H,"D basis");end,
  BetaInvolution:=h->HeckeAlgebraOps.Beta(h,"C'"),
  AltInvolution:=h->HeckeAlgebraOps.Alt(h,"D'"),
  T:=h->h.coeff*List(h.elm,x->HeckeAlgebraOps.Specialize(HeckeAlgebraOps.BetaT
  (HeckeAlgebraOps.getCp(Hecke(h),
            LongestCoxeterElement(CoxeterGroup(Hecke(h)))*x)))),
  D:=h->HeckeAlgebraOps.ToKL(h,"D",Minimum)
  ));

CreateHeckeBasis("D'",rec(
  init:=function(H)HeckeAlgebraOps.InitKL(H,"D' basis");end,
  BetaInvolution:=h->HeckeAlgebraOps.Beta(h,"C"),
  AltInvolution:=h->HeckeAlgebraOps.Alt(h,"D"),
  T:=h->h.coeff*List(h.elm,x->HeckeAlgebraOps.Specialize(
     HeckeAlgebraOps.AltT(HeckeAlgebraOps.BetaT(HeckeAlgebraOps.getCp(
	Hecke(h),LongestCoxeterElement(CoxeterGroup(Hecke(h)))*x))))),
  ("D'"):=h->HeckeAlgebraOps.ToKL(h,"D'",Minimum)
  ));

#############################################################################
##
#F  CriticalPair( <W>, <y>, <w>, <ly> )  . . . . . . . . . the critical pair 
#F  . . . . . . . . . . . . . . . . associated with a given pair of elements 
##
##  A pair of elements $y$, $w$ in $W$ is called critical if neither the
##  left descent set of $y$ is contained in the left descent set of $w$
##  nor the right descent set of $y$ is contained in that of $w$. The
##  function 'CriticalPair' returns, for a given pair <y>, <w> of elements
##  in a Coxeter group <W>, the pair itseld if this is critical. Otherwise,
##  <y> can be multiplied from the left or the right by a generator $s$,
##  and we can proceed with the pair consisting of this new element and
##  <w>. Repeating in this way, we finally obtain a critical associated
##  with the given pair. The variable <ly> contains the length of <y>.
##  The length of the new element replacing <y> is also returned. The 
##  significance of this construction is that the Kahzdan-Lusztig polynomials 
##  corresponding to the original pair and to the associated critical pair 
##  are equal.
##
CriticalPair:=function(W,y,w,ly)
  local cr,Rw,Lw,sR,rI,pN;
  Lw:=LeftDescentSet(W,w);
  Rw:=RightDescentSet(W,w);
  sR:=W.semisimpleRank;
  rI:=W.rootInclusion;
  pN:=W.parentN;
  
  cr:=function(y,ly)
    local s;
    s:=1;
    while s<=sR and (rI[s]^y > pN or not s in Lw) do
      s:=s+1;
    od;
    if s>sR then
      s:=1;
      while s<=sR and (rI[s]/y > pN or not s in Rw) do
        s:=s+1;
      od;
      if s>sR then
        return [y,w,ly];
      else
        return cr(y*W.generators[s],ly+1);
      fi;
    else
      return cr(W.generators[s]*y,ly+1);
    fi;
  end;
  return cr(y,ly);
end;


#############################################################################
##
#F  KazhdanLusztigPolynomial( <W>, <y>, <w> [, <ly>, <lw> ] )  . . . . . . .
##  . . . . . . . . . . . . . . . . . . . . . . . Kazhdan-Lusztig polynomial
##
##  'KazhdanLusztigPolynomial'   returns   the  Kazhdan - Lusztig  polynomial 
##  in an indeterminate over the rationals corresponding to the elments <y> 
##  and <w>  (given as permutations) in the Coxeter group <W>. The optional
##  variables <ly> and <lw>  contain the length of <y> and <w>, respectively.
##
KazhdanLusztigPolynomial:=function(arg)
  local W,y,w,ly,lw,pw,py,cr,s,v,lv,z,lz,m,tmp,2N,pol,wcr,otc1;
  W:=arg[1]; y:=arg[2]; w:=arg[3];
  if Length(arg)=3 then
    ly:=CoxeterLength(W,y);
    lw:=CoxeterLength(W,w);
  else
    ly:=arg[4]; lw:=arg[5];
  fi;
  if not Bruhat(W,y,w,ly,lw) then
    return 0*X(Rationals);
  fi;
  cr:=CriticalPair(W,y,w,ly);
  ly:=cr[3];
  if lw-ly<=2 then
    return X(Rationals)^0;
  fi; 
  
  if not IsBound(W.criticalPairs) then
    W.criticalPairs:=List([1..W.parentN+1],x->[]);
    W.klpol:=[];  
  fi;
  pw:=Position(CoxeterElementsLength(W,lw),w);
  wcr:=W.criticalPairs[lw+1];
  
  # encode cr[1] in some way to save memory (an element is uniquely determined
  # by its images of the simple roots)
  2N:=2*W.parentN;
  tmp:=OnTuples([1..W.semisimpleRank],cr[1]);
  otc1:=tmp[1];
  for z in [2..Length(tmp)] do
    otc1:=otc1*2N+tmp[z];
  od;
  if IsBound(wcr[pw]) then
    py:=Position(wcr[pw][1],otc1);
    if not py=false then
      return Rationals.operations.Polynomial(Rationals,
                     W.klpol[wcr[pw][2][py]],0);
    fi;
  else
    wcr[pw]:=[[],[]];
  fi;
  
  s:=1;
  while W.rootInclusion[s]^cr[2]<=W.parentN do
    s:=s+1;
  od;
  v:=W.generators[s]*w;
  lv:=lw-1;
  pol:=KazhdanLusztigPolynomial(W,W.generators[s]*cr[1],v,ly-1,lv)
                     +X(Rationals)*KazhdanLusztigPolynomial(W,cr[1],v,ly,lv);
  lz:=lv-1;
  while lw-lz<=2*Degree(pol) do
    for z in CoxeterElementsLength(W,lz) do 
      if lw-lz<=2*Degree(pol) and pol.coefficients[(lw-lz)/2+1]>0 
          and W.rootInclusion[s]^z>W.parentN and Bruhat(W,cr[1],z,ly,lz) then
        m:=KazhdanLusztigMue(W,z,v,lz,lv);
        if m<>0 then
          pol:=pol-m*X(Rationals)^((lv-lz+1)/2)*
                                   KazhdanLusztigPolynomial(W,cr[1],z,ly,lz);
        fi;
      fi;
    od;
    lz:=lz-2;
  od;
  
  Add(wcr[pw][1],otc1);
  py:=Position(W.klpol,pol.coefficients);
  if py=false then
    Add(W.klpol,pol.coefficients);
    py:=Length(W.klpol);
  fi;
  Add(W.criticalPairs[lw+1][pw][2],py);
  return pol;
end;

#############################################################################
##
#F  KazhdanLusztigMue( <W>, <y>, <w> [, <ly>, <lw> ] )  . . . . . . . . . . .
#F  . . . . . . . . . the highest coefficient of a Kazhdan-Lusztig polynomial
##
##  'KazhdanLusztigMue'  returns the coefficient of highest possible degree
##  of the Kazhdan-Lusztig polynomial of the Coxeter group <W> corresponding 
##  to the elements <y> and <w>. The optional variables <ly> and <lw> contain 
##  the  length of <y> and <w>, respectively. That highest possible degree is 
##  $(lw- ly -1)/2$. If <y> and <w> are not related by the Bruhat order, the 
##  value 0 is returned.
##
KazhdanLusztigMue:=function(arg)
  local W,y,w,ly,lw,s,Lw,Rw,pol;
  W:=arg[1]; y:=arg[2]; w:=arg[3];
  if Length(arg)>3 then
    ly:=arg[4]; lw:=arg[5];
  else
    ly:=CoxeterLength(W,y);
    lw:=CoxeterLength(W,w);
  fi;
  if ly=lw or not Bruhat(W,y,w,ly,lw) then
    return 0;
  fi;
  if lw=ly+1 then
    return 1;
  fi; 
  Lw:=LeftDescentSet(W,w);
  Rw:=RightDescentSet(W,w);
  for s in [1..W.semisimpleRank] do
    if (W.rootInclusion[s]^y<=W.parentN and s in Lw)
       or (W.rootInclusion[s]^(y^-1)<=W.parentN and s in Rw) then
         return 0;
    fi;
  od;
  pol:=KazhdanLusztigPolynomial(W,y,w,ly,lw);
  if Degree(pol)=(lw-ly-1)/2 then
    return pol.coefficients[(lw-ly+1)/2];
  else
    return 0;
  fi;
end;


#############################################################################
##
#F  KazhdanLusztigCoefficient( <W>, <y>, <w>, [ <ly>, <lw> ], <k> ) . . . . .
#F  . . . . . . . . . . . . . . . coefficient of a Kazhdan-Lusztig polynomial
##
##  'KazhdanLusztigCoefficient'  returns the <k> th coefficient of the 
##  Kazhdan-Lusztig polynomial of the Coxeter group <W> corresponding to the 
##  elements <y> and <w>. The variables <ly> and <lw> contain the length
##  of <y> and <w>, respectively.  If <y> and <w> are not related by the 
##  Bruhat order, the  value 0 is returned.
##
KazhdanLusztigCoefficient:=function(arg)
  local W,y,w,ly,lw,k,s,Lw,Rw,pol;
  W:=arg[1]; y:=arg[2]; w:=arg[3];
  if Length(arg)>4 then
    ly:=arg[4]; lw:=arg[5]; k:=arg[6];
  else
    k:=arg[4];
    ly:=CoxeterLength(W,y);
    lw:=CoxeterLength(W,w);
  fi;
  if k<0 or not IsInt(k) or ly>lw or (ly<lw and 2*k>lw-ly-1) 
                                         or not Bruhat(W,y,w,ly,lw) then
    return 0;
  fi;
  if k=0 then
    return 1;
  fi;
  pol:=KazhdanLusztigPolynomial(W,y,w,ly,lw);
  if k<=Degree(pol) then
    return pol.coefficients[k+1];
  else
    return 0;
  fi;
end;

#############################################################################
##
#F  KLMueMat( <W>, <list> )  . . . .  the matrix of leading coefficients of
#F  Kazhdan-Lusztig polynomials  of elements in a given list of reduced words
##
##  The elements in <list> must be ordered by increasing length. 
##
##  This should be fixed to return a square matrix.   Jean
##
KLMueMat:=function(W,c) 
  local m,i,j,k,ls,x,n,p,d,s,w0;
  w0:=LongestCoxeterElement(W); 
  n:=Length(c); 
  ls:=List(c,i->CoxeterLength(W,i));
  m:=List([1..n],i->[]);
  for k in [0..ls[n]-ls[1]] do
    for i in [1..n] do
      for j in [1..i] do
	d:=ls[i]-ls[j];
	if d=k then
	  if ls[i]+ls[j]>W.N  then
	     m[i][j]:=KazhdanLusztigMue(W,w0*c[i],w0*c[j],
                                          W.N-ls[i],W.N-ls[j]);
	  else 
             m[i][j]:=KazhdanLusztigMue(W,c[j],c[i],ls[j],ls[i]);
	  fi;
	fi;
      od;
    od;
  od;
  return m;
end;

DecomposedLeftCells := function ( W, c)
  local  Lleq, s1, s2, d, ce, i, j, m, rest, mue;
  Lleq := function ( W, mue, x, y )
    local  i;
    if x = y  then
      return false;
    elif x < y and mue[y][x] = 0  then
      return false;
    elif x > y and mue[x][y] = 0  then
      return false;
    else
      for i  in [ 1 .. W.semisimpleRank ]  do
        if W.rootInclusion[i]^c[x]>W.parentN 
                          and W.rootInclusion[i]^c[y]<=W.parentN  then
          return true;
        fi;
      od;
      return false;
    fi;
  end;
  mue:=KLMueMat(W,c);
  ce := [  ];
  rest := [ 1 .. Length( c ) ];
  while rest <> [  ]  do
      s1 := [ rest[1] ];
      for j  in s1  do
          for i  in [ 1 .. Length( c ) ]  do
              if not i in s1 and Lleq( W, mue, j, i )  then
                  Add( s1, i );
              fi;
          od;
      od;
      Sort( s1 );
      s2 := [ rest[1] ];
      for j  in s2  do
          for i  in [ 1 .. Length( c ) ]  do
              if not i in s2 and Lleq( W, mue, i, j )  then
                  Add( s2, i );
              fi;
          od;
      od;
      Sort( s2 );
      d := Intersection( s1, s2 );
      Sort( d );
      m := [  ];
      for i  in [ 1 .. Length( d ) ]  do
          m[i] := [  ];
          for j  in [ 1 .. i ]  do
              m[i][j] := mue[d[i]][d[j]];
          od;
      od;
      Add( ce, [ List( d, i->CoxeterWord(W,c[i])), m ] );
      SubtractSet( rest, d );
  od;
  return ce;
end;

#############################################################################
##
#F  LeftCells( <W> )  . . . . . . . . . . . . . . . . . . the left cells of W
##
##  'LeftCells' returns a list of pairs. The first  component  of  each  pair
##  consists of the reduced words in the Coxeter group <W> which lie in 
## one left cell C,  the  second  component  consists  of the corresponding 
##  matrix of  highest coefficients mue(y,w), where y,w are in C.
##
LeftCells := function ( arg )
  local  W, t, i, j, v, k, max, res, rw, x;
  W := arg[1];
  t := Combinations( [ 1 .. W.semisimpleRank ] );
  if  Length( arg ) = 2  then
      max := arg[2];
  else
      max := Size(W);
  fi;
  rw := List([ 1 .. Length( t ) ],j->[  ]);
  for i  in Concatenation(List([0..W.N],i->CoxeterElementsLength(W,i))) do
    Add( rw[Position( t, RightDescentSet( W, i ) )], i);
  od;
  res := [  ];
  for i  in rw  do
      if Length(i) <= max then
	  InfoChevie( "#I  R(w) = ", t[Position( rw, i )],
				   " :  Size = ", Length( i ) , "\c" );
	  x := DecomposedLeftCells( W, i);
	  if Length( x ) = 1  then
	      InfoChevie( ",  ", Length( x ), " new cell \n" );
	  else
	      InfoChevie( ",  ", Length( x ), " new cells \n" );
	  fi;
	  Append( res, x );
      fi;
  od;
  for v in res do
    for j in [1..Length(v[1])] do
      for k in [j+1..Length(v[1])] do
	v[2][j][k]:=v[2][k][j];
      od;
    od;
  od;
  return res;
end;

#############################################################################
##
#F  LeftCellRepresentation( <H> , <cell> ) . . . . . . . . . . representation
#F  of  the  Hecke  algebra  associated  to  a  left cell of  a  Coxeter group
##
##  'LeftCellRepresentation' returns a list of matrices giving the left  cell 
##  representation of the  Hecke algebra <H> associated to <cell>, given as
##  a pair with  first component  a  list of reduced  words which  form  a 
##  left cell, and  second component  the corresponding matrix of mue's.
##  Typically, <cell> is the result of the  function 'LeftCells'.
##
LeftCellRepresentation := function ( H, cell )
    local  s, i, j, rep, t, f1, f2, pc, c, mue, v, W;
    if ForAny([1..Length(H.parameter)],i->not IsBound(H.sqrtParameter[i]))
    then
      Error("sqrtParameters must be bound to compute cell representations");
    elif Length( Set( H.parameter ) ) > 1 then
      Error("cell representations for unequal parameters not yet implemented");
    else 
      v := H.sqrtParameter[1];
    fi;
    c := cell[1];
    W:=CoxeterGroup(H);
    pc := List( c, function ( i ) return PermCoxeterWord ( W, i ); end );
    mue := cell[2];
    rep := [  ];
    for s  in W.generators  do
        f1 := [  ];
        for i  in pc  do
            if CoxeterLength( W, s * i ) > CoxeterLength( W, i ) then
                Add( f1, true );
            else
                Add( f1, false );
            fi;
        od;
        f2 := List( pc, i->Position( pc, s * i ) );
        t := [  ];
        for i  in [ 1 .. Length( c ) ]  do
            t[i] := [  ];
            for j  in [ 1 .. Length( c ) ]  do
                if i = j  then
                    if f1[i]  then
                        t[i][i] := v ^ 2;
                    else
                        t[i][i] := -v^0;
                    fi;
                elif i > j  then
                    if f2[j] = i  then
                        t[i][j] := v;
                    else
                        t[i][j] := 0*v;
                    fi;
                else
                    if f1[j] and mue[j][i] <> 0 and not f1[i]  then
                        t[i][j] := mue[j][i]*v;
                    else
                        t[i][j] := 0*v;
                    fi;
                fi;
            od;
        od;
        Add( rep, t );
    od;
    return rep;
end;

